/****************************************************************************
 *                            StimSeqDraw.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a Stimulus Sequence reader that realizes the sequence
 *         in a DFIO trace...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StimSeqDraw.h"


/********************************************************************
 * StimSeqDraw()
 ********************************************************************/
StimSeqDraw::StimSeqDraw() : d_levStack()
{
    setup(0, 0, 0);
}

/********************************************************************
 * StimSeqDraw()
 ********************************************************************/
StimSeqDraw::StimSeqDraw(
        DFIOTrace       *trace, 
        StimulusSeq     *seq, 
        Uint32           maxTime) :
    d_levStack()
{
    setup(trace, seq, maxTime);
}

/********************************************************************
 * setup()
 ********************************************************************/
void StimSeqDraw::setup(
        DFIOTrace       *trace, 
        StimulusSeq     *seq, 
        Uint32           maxTime) 
{
    d_trace   = trace;
    d_seq     = seq;
    d_maxTime = 0;
    d_repeatCnt = 0;
    d_currTime  = 0;
}

/********************************************************************
 * operator ()
 ********************************************************************/
void StimSeqDraw::operator () (
        DFIOTrace            *trace,
        StimulusSeq          *seq)
{
    setup(trace, seq, 0);
}

/********************************************************************
 * Draw()
 ********************************************************************/
void StimSeqDraw::Draw(Uint32 max)
{
    StimSeqLevel    *lev;

    d_maxTime = max;

    lev = new StimSeqLevel(d_seq);
    d_levStack.push(lev);
    d_outMode = (d_seq->getInitDelay())?ModeDelay:ModeOutput;

    DrawHelper();
}


/********************************************************************
 * DrawHelper()
 ********************************************************************/
int StimSeqDraw::DrawHelper()
{
    StimSeqLevel    *lev = d_levStack.top();
    StimulusSeq     *seq;
    StimulusItem    *item;
    Uint32           delay;
    bool             done = false;

    d_currTime = 0;
    if (!(lev = d_levStack.top())) {
        return 0;
    }

    seq = lev->seq;

    while (!done) {
        item = seq->get_values()->idx(lev->seq_idx);

        switch (d_outMode) {

            /********************************************************
             * ModeRepeatDelay
             ********************************************************/
            case ModeRepeatDelay:

                /**** Add the delay to our current time. ****/
                d_currTime += seq->getRepeatDelay();

                d_outMode = ModeOutput;
                break;

            /********************************************************
             * ModeDelay
             ********************************************************/
            case ModeDelay:
                delay=(!lev->seq_idx)?(seq->getInitDelay()):item->get_delay();

                d_outMode = ModeOutput;

                if (delay) {
                    d_currTime += delay;
                }

                /**** Drop through ****/

            /********************************************************
             * ModeOutput
             ********************************************************/
            case ModeOutput:
                if (item->is_sequence()) {
                    StimSeqLevel *level = new StimSeqLevel(item->get_seq());
                    d_levStack.push(level);

                    lev = level;
                    seq = lev->seq;
                    d_outMode = ModeDelay;
                    break;
                } else {
                    /**** Output curr value and move to next ****/
                    d_trace->setTime(d_currTime);
                    d_trace->setValueBitStr(0, item->get_value()->rawValue());

                    if (d_currTime >= d_maxTime) {
                        done = true;
                        break;
                    }
                }
                d_outMode = ModeSetupNext;

                /**** Drop through ****/

            /********************************************************
             * ModeSetupNext
             ********************************************************/
            case ModeSetupNext:
                lev->seq_idx++;

                if (lev->seq_idx >= seq->get_values()->length()) {

                    if ((++lev->repeat_cnt >= seq->getNumRepeat()) &&
                            (seq->getNumRepeat() < 0xFFFFFFFF)) {

                        /**** This level done - move on ****/
                        delete lev;
                        d_levStack.pop();

                        if (!(lev = d_levStack.top())) {
/*                            delete inst; */
                            d_trace->setTime(d_currTime+100);
                            d_trace->setValueBitStr(0, "0");
                            return 0;
                        }

                        /**** Move back to prev level 
                         **** - we go back to SetupNext
                         ****/
                        seq = lev->seq;
                        break;
                    } else {
                        /**** Else, back to the beginning *****/
                        lev->seq_idx = 0;

                        if (seq->getRepeatDelay()) {
                            d_outMode = ModeRepeatDelay;
                        } else {
                            d_outMode = ModeOutput;
                        } 
                    } 
                } else {
                    d_outMode = ModeDelay;
                }
                break;

            default:
                break;
        }
    }
}



