/****************************************************************************
 *                             XPixmapObj.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a wrapper around a pixmap... This allows us to 
 *         implement various back-ends and use the same drawing code...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "XPixmapObj.h"
#include <stdio.h>

#define FP      stderr
#undef  DEBUG_PIXMAP

#ifdef DEBUG_PIXMAP
#define DBG_MSG(x)   fprintf x
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * XPixmapObj()
 ********************************************************************/
XPixmapObj::XPixmapObj(Tk_Window window)
{
    d_copyGC     = 0;
    d_pixmap     = 0;
    d_display    = Tk_Display(window);
    d_window     = window;
    d_width      = 0;
    d_height     = 0;
    d_winPixmap  = true;
}

/********************************************************************
 * XPixmapObj()
 ********************************************************************/
XPixmapObj::XPixmapObj() 
{
    d_copyGC     = 0;
    d_pixmap     = 0;
    d_display    = 0;
    d_window     = 0;
    d_width      = 0;
    d_height     = 0;
    d_winPixmap  = false;
}

/********************************************************************
 * ~XPixmapObj()
 ********************************************************************/
XPixmapObj::~XPixmapObj()
{
    if (!d_winPixmap && d_pixmap && d_display) {
        Tk_FreePixmap(d_display, d_pixmap);
    }
}

/********************************************************************
 * setWindow()
 ********************************************************************/
void XPixmapObj::setWindow(Tk_Window win)
{
    d_display   = Tk_Display(win);
    d_pixmap    = 0;
    d_window    = win;
    d_winPixmap = true;
}

/********************************************************************
 * operator =
 ********************************************************************/
void XPixmapObj::operator = (const PixmapRange &rng)
{
    PixmapObj *src = rng.d_pixmap;

    copy(src, rng.d_x, rng.d_y, rng.d_width, rng.d_height);
}

/********************************************************************
 * line()
 ********************************************************************/
void XPixmapObj::line(GC gc, Uint32 x1, Uint32 y1, 
        Uint32 x2, Uint32 y2)
{
    DBG_MSG((FP, "----> line(%x, (%d %d) - (%d %d)\n",
                gc, x1, y1, x2, y2));

    XDrawLine(d_display, d_pixmap, gc, x1, y1, x2, y2);

    DBG_MSG((FP, "<---- line(%x, (%d %d) - (%d %d)\n",
                gc, x1, y1, x2, y2));
}

/********************************************************************
 * rect()
 ********************************************************************/
void XPixmapObj::rect(GC gc, Uint32 x, Uint32 y, 
        Uint32 width, Uint32 height)
{
    XDrawRectangle(d_display, d_pixmap, gc, x, y, width, height);
}

/********************************************************************
 * fill_rect()
 ********************************************************************/
void XPixmapObj::fill_rect(GC gc, Uint32 x, Uint32 y, 
        Uint32 width, Uint32 height)
{
    XFillRectangle(d_display, d_pixmap, gc, x, y, width, height);
}

/********************************************************************
 * createGC()
 ********************************************************************/
GC XPixmapObj::createGC(GCObj &gcObj)
{
    StippleStruct    &ss = gcObj.getStippleStruct();
    XGCValues        *gc = gcObj.getGCValues();
    GC                tgc;

    /**** If stipple has been set, then create a pixmap to 
     **** describe it...
     ****/
    if (ss.data) {
        ss.pixmap = XCreateBitmapFromData(d_display, Tk_WindowId(d_window),
                (const char *)ss.data, ss.width, ss.height);

        if (!ss.pixmap) {
            fprintf(stderr, "ERROR :: Cannot create pixmap\n");
        }
        gc->stipple = ss.pixmap;
    }

    tgc = Tk_GetGC(d_window, gcObj.getValueMask(), gcObj.getGCValues());

    return tgc;
}

/********************************************************************
 * freeGC()
 ********************************************************************/
void XPixmapObj::freeGC(GCObj &gcObj)
{
    StippleStruct    &ss = gcObj.getStippleStruct();

    if (ss.pixmap) {
        Tk_FreePixmap(d_display, ss.pixmap);
        ss.pixmap = 0;
    }

    Tk_FreeGC(d_display, gcObj.getGC());
}

/********************************************************************
 * copy()
 ********************************************************************/
void XPixmapObj::copy(
        PixmapObj *src, Uint32 x, Uint32 y, Uint32 width,
        Uint32 height, Uint32 x1, Uint32 y1)
{
    DBG_MSG((FP, "----> copy(%x, %d, %d, %d, %d, %d, %d) %x = dest\n",
                src, x, y, width, height, x1, y1, this));

    /**** Copy data from obj to this pixmap... ****/
    XCopyArea(d_display, ((XPixmapObj *)src)->getPixmap(), 
            getPixmap(), getCopyGC(), x, y, width, height, x1, y1);

    DBG_MSG((FP, "<---- copy(%x, %d, %d, %d, %d, %d, %d) %x = dest\n",
                src, x, y, width, height, x1, y1, this));
}

/********************************************************************
 * copy()
 ********************************************************************/
void XPixmapObj::copy(
        PixmapObj *obj, Uint32 x, Uint32 y, Uint32 width, Uint32 height)
{
    copy(obj, x, y, width, height, 0, 0);
}

/********************************************************************
 * width_height()
 ********************************************************************/
void XPixmapObj::width_height(Uint32 width, Uint32 height)
{
    DBG_MSG((FP, "----> width_height( %d, %d )\n", width, height));

    if (width != d_width || height != d_height) {
        if (d_pixmap) {
            DBG_MSG((FP, "\tFree pixmap...\n"));
            Tk_FreePixmap(d_display, d_pixmap);
        }

        DBG_MSG((FP, "\tAlloc pixmap\n"));
        d_pixmap = Tk_GetPixmap(d_display, Tk_WindowId(d_window), 
                width, height, Tk_Depth(d_window));
    }

    d_width  = width;
    d_height = height;

    d_winPixmap  = false;

    DBG_MSG((FP, "<---- width_height( %d, %d )\n", width, height));
}

/********************************************************************
 * getType()
 ********************************************************************/
PixmapObj::PixmapType XPixmapObj::getType() const
{
    return XPixmap;
}

/********************************************************************
 * ComputeTextLayout()
 ********************************************************************/
Tk_TextLayout XPixmapObj::ComputeTextLayout(
                Tk_Font font, Char *str, Int32 len, Int32 wraplen,
                Tk_Justify justify, Int32 flags, Int32 *width, Int32 *height)
{
    return Tk_ComputeTextLayout(font, str, len, wraplen, 
            justify, flags, width, height);
}

/********************************************************************
 * DrawTextLayout()
 ********************************************************************/
void XPixmapObj::DrawTextLayout(GC gc, Tk_TextLayout layout, 
        Uint32 x, Uint32 y, Int32 start, Int32 end)
{
    Tk_DrawTextLayout(d_display, getPixmap(), gc, layout, x, y, start, end);
}

/********************************************************************
 * FreeTextLayout()
 ********************************************************************/
void XPixmapObj::FreeTextLayout(Tk_TextLayout layout)
{
    Tk_FreeTextLayout(layout);
}

/********************************************************************
 * blank()
 ********************************************************************/
void XPixmapObj::blank()
{
    if (!d_background) {
        fprintf(stderr, "PixmapObj ERROR :: blank called w/NULL background\n");
        return;
    }

    if (d_pixmap) {
        Tk_Fill3DRectangle(d_window, d_pixmap, d_background, 0, 0,
                d_width, d_height, 0, TK_RELIEF_FLAT);
    } else if (d_window) {
        Tk_SetWindowBackground(d_window, Tk_3DBorderColor(d_background)->pixel);
    }
}

/********************************************************************
 * getCopyGC()
 ********************************************************************/
GC XPixmapObj::getCopyGC()
{
    if (!d_copyGC) {
        XGCValues    gcv;

        d_copyGC = Tk_GetGC(d_window, 0, &gcv);
    }

    return d_copyGC;
}


