#****************************************************************************
#*                              ConsoleMgr.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a console manager...
#****************************************************************************

namespace eval ConsoleMgr {
    variable d_consoleList     ""
    variable d_cmdHistory      ""

    namespace export do ivi_puts
}

#********************************************************************
#* AddConsole
#********************************************************************
proc ConsoleMgr::AddConsole {console} {
    variable d_consoleList

    lappend d_consoleList $console
}

#********************************************************************
#* RemoveConsole
#********************************************************************
proc ConsoleMgr::RemoveConsole {console} {
    variable d_consoleList

    set idx [lsearch -exact $d_consoleList $console]

    if {$idx == -1} {
        ::puts "ERROR: console \"$console\" isn't registered"
        return
    }

    set tmp ""
    if {$idx > 0} {
        set tmp [lrange $d_consoleList 0 [expr $idx - 1]]
    }

    if {[llength $d_consoleList] > 0} {
        lappend tmp [lrange $d_consoleList [expr $idx + 1] end]
    }

    set d_consoleList $tmp
}

#********************************************************************
#* ivi_puts
#********************************************************************
proc ConsoleMgr::ivi_puts {string} {
    variable d_consoleList

    foreach console $d_consoleList {
        $console print_line $string
    }
}

#********************************************************************
#* RunConsoleCmd
#********************************************************************
proc ConsoleMgr::RunConsoleCmd {console cmd} {
    variable d_cmdHistory

    lappend d_cmdHistory [list $cmd]

    set stat [catch {uplevel #0 $cmd} res]

    #*** TODO: must split result on line-ends
    if {$stat == 1} {
        $console print_line "Error: $res"
    } elseif {[string compare "" "$res"] != 0} {
        $console print_line "$res"
    }

    if {$stat == 1} {
        return -code error
    } else {
        return
    }
}

#********************************************************************
#* GetNextCmdId
#********************************************************************
proc ConsoleMgr::GetNextCmdId {} {
    variable d_cmdHistory

    return [llength $d_cmdHistory]
}

#********************************************************************
#*
#********************************************************************
proc ConsoleMgr::GetHistoryCmd {index} {
    variable d_cmdHistory

    if {$index >= 0 && $index < [llength $d_cmdHistory]} {
        return [join [lindex $d_cmdHistory $index] " "]
    } else {
        return ""
    }
}

#********************************************************************
#* do
#********************************************************************
proc ConsoleMgr::do {file} {
    ConsoleMgr::source $file
}

#********************************************************************
#* source
#********************************************************************
proc ConsoleMgr::source {file} {
    variable d_consoleList

    set fp [open $file "r"]

    set done 0

    while {1} {
        set line ""
        while {1} {
            if {[eof $fp]} {
                set done 1
                break
            }

            set len [gets $fp ltmp]

            #*** If the line ends with "\", then append to the line...
            if {[string index $ltmp [expr $len - 1]] == "\\"} {
                set line "$line [string range $ltmp 0 [expr $len-2]]"
            } else {
                set line "$line$ltmp"
                if {[info complete $line]} {
                    break
                }
            }
        }

        if {$done == 1} {
            break
        }

        #*** TODO: channel result to console
        if {$line != ""} {
            if {[llength $d_consoleList] > 0 && $line != ""} {
                set tline [string trimleft $line]

                if {[string index $tline 0] != "#"} {
                    set console [lindex $d_consoleList 0]

#                    puts "Running command \"$line\""
#                    $console enter_command $line
                    RunConsoleCmd $console $line
                }
            } else {
                uplevel #0 eval $line
            }
        }
    }
}

namespace import ConsoleMgr::*


