#*****************************************************************************
#*                            SigWindow.tcl
#* Author: Matthew Ballance
#* Desc:   This file implements the top-level window for a signal browser.
#*
#* Subwidgets
#*  - SdbWaveReader
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*****************************************************************************

namespace eval SigWindow {
    namespace export SigWindow

    variable configspec {
        {"-sb_width"        12 }
        {"-font"            "-*-arial-medium-r-normal-*-*-*-*-*-*-*"}
        {"-width"           650}
        {"-height"          400}
        {"-win_name"         ""}
        {"-title"            ""}
    }
}

#*********************************************************
#* SigWindow
#*********************************************************
proc SigWindow::SigWindow {path args} {
    variable configspec

    set path [toplevel $path]

    rename $path ::$path:cmd
    proc ::$path { cmd args } "return \[eval SigWindow::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    set data(w:enum_win) ""

    set data(w:toplev) $path

    set data(list_present) 0
    set data(sepChar) "."
    set data(current_menu) ""
    set data(w:signals) ""

    foreach varr $configspec {
        set data([lindex $varr 0]) [lindex $varr 1]
    }

    eval configure $path $args

    SigWindow::ConstructWidget $path
    SigWindow::SetBindings $path

    SigWindow::geometry $path $data(-width)x$data(-height)

    return $path
}

#*********************************************************
#* configure
#*********************************************************
proc SigWindow::configure {w args} {
    upvar #0 $w data

    while {[llength $args] > 0} {

        set arg [lindex $args 0]
        set val [lindex $args 1]
        set varr [array get data $arg]

        if {$varr != ""} {
            set data($arg) $val
        } else {
            error "no conf entry \"$arg\""
        }
        set args [lrange $args 2 end]
    }
}

#*********************************************************
#* cmd
#*********************************************************
proc SigWindow::cmd {path cmd args} {
    upvar #0 $path data

    if {$cmd == "update"} {
        SigWindow::update $path
    } elseif {$cmd == "AddToWave"} {
        SigWindow::AddToWave $path $args
    } elseif {$cmd == "AddSeparator"} {
        SigWindow::AddSeparator $path $args
    } elseif {$cmd == "WaveZoom"} {
        SigWindow::WaveZoom $path $args
    } elseif {$cmd == "show_page"} {
        SigWindow::show_page $path $args
    } elseif {$cmd == "geometry"} {
        SigWindow::geometry $path $args
    } elseif {$cmd == "siglist_width"} {
        eval SigWindow::siglist_width $path $args
    } elseif {$cmd == "sigvallist_width"} {
        eval SigWindow::sigvallist_width $path $args
    } elseif {$cmd == "subwidget"} {
        set val [array get data "w:[lindex $args 0]"]
        if {$val == ""} {
            error "no subwidget \"[lindex $args 0]\" in SigWindow"
        } else {
            return [lindex $val 1]
        }
    } elseif {$cmd == "config" || $cmd == "configure"} {
        eval configure $path $args
    } else {
        error "Unknown SigWindow sub-cmd \"$cmd\""
    }
}

#*********************************************************
#* SetMenu
#*********************************************************
proc SigWindow::SetMenu {w new_menu} {
    upvar #0 $w data

    ::$w:cmd configure -menu $new_menu
}

#*********************************************************
#* ConstructStructPane
#*********************************************************
proc SigWindow::ConstructStructPane {w p} {
    upvar #0 $w data

    set data(w:sdb_pane) [SdbPane::SdbPane $p \
        -browse_cmd "SigWindow::BrowseModule $w"]

    return $p
}

#*********************************************************
#* ConstructWidget
#*********************************************************
proc SigWindow::ConstructWidget {w} {
    upvar #0 $w data

    #***************************************************
    #* Create the menu
    #***************************************************
    set data(w:topframe)  [frame $w.topframe  -width $data(-width) \
                           -height $data(-height)]
    set data(w:menuframe) [frame $data(w:topframe).mf -relief flat \
                           -borderwidth 0]

    #***************************************************
    #* Create the Structure menu-bar
    #***************************************************
    set data(w:struct_menu) \
        [MenuMgr::create menubar SigWindow::StructMenu \
            $data(w:menuframe).struct_menu -var [list w $w]]

    set data(w:wave_menu) \
        [MenuMgr::create menubar SigWindow::WaveMenu \
            $data(w:menuframe).wave_menu -var [list w $w]]

    #***************************************************
    # Setup SdbReaders...
    #***************************************************
    set data(w:SdbWaveReader) [sdbr $w.wave_reader_sdbr]

    #***************************************************
    #* Create the notebook...
    #***************************************************
    set data(w:nb) [NoteBook $data(w:topframe).db -background "grey80"]

    set data(w:structPage) \
        [$data(w:nb) insert end "structPage" -text "Structure" \
        -raisecmd [list SigWindow::SetMenu $w $data(w:struct_menu)]]
    set data(w:wavePage)   \
        [$data(w:nb) insert end "wavePage" -text "Waveform" \
        -raisecmd [list SigWindow::SetMenu $w $data(w:wave_menu)]]

    $data(w:nb) raise structPage

    set data(w:sigPane) [PanedWindow $data(w:structPage).sigPane]

    set data(w:structPane) [$data(w:sigPane) add -weight 1]
    set data(w:signalPane) [$data(w:sigPane) add -weight 1]

    set data(w:structure) [SigWindow::ConstructStructPane $w \
        $data(w:structPane).structure]

    set cont \
       [ScrollContainer::ScrollContainer $data(w:signalPane).signals]
    set lframe [$cont subwidget frame]

    set data(w:signals) [listbox $lframe.listbox -selectmode extended \
        -background white -selectbackground "#000080" -selectforeground "white"]

    set vsb [$cont subwidget vsb]
    set hsb [$cont subwidget hsb]

    $data(w:signals) configure -yscrollcommand "$vsb set"
    $data(w:signals) configure -xscrollcommand "$hsb set"

    $vsb configure -command "$data(w:signals) yview"
    $hsb configure -command "$data(w:signals) xview"

    pack $cont -expand yes -fill both
    pack $data(w:signals) -expand yes -fill both
    pack $data(w:structure) -expand yes -fill both
    pack $data(w:sigPane)   -expand yes -fill both -side left

    #***************************************************
    #* Create the Waveform page...
    #***************************************************
    set data(w:waveWinPane) [WaveWindow::WaveWindow \
        $data(w:wavePage).waveWinPane -sdbr $data(w:SdbWaveReader)]

    pack $data(w:waveWinPane) -expand yes -fill both

    pack $data(w:topframe) -side top -expand yes -fill both
    pack $data(w:menuframe) -side top -expand no -fill x
    pack $data(w:nb)        -side top -expand yes -fill both

    #***************************************************
    #* Construct popup menu
    #***************************************************
    set $w.popup [MenuMgr::create popup SigWindow::SigListPopup \
        $w.popup -var [list w $w list_present 0]]
}

#*********************************************************
#* show_page
#*********************************************************
proc SigWindow::show_page { w name } {
    upvar #0 $w data

    set nb $data(w:nb)
    switch $name {
        wave        {$nb raise wavePage}
        structure   {$nb raise structPage}
        struct      {$nb raise structPage}
        list        {
            if {$data(list_present) == 1} {
                $nb raise listPage
            }
        }
        default          {error "Unknown page $name"}
    }
}

#*********************************************************
#* SaveSetupCmd
#* Wrapper for the SaveSetup proc
#*********************************************************
proc SigWindow::SaveSetupHelper {w} {
    upvar #0 $w data

    set file [tk_getSaveFile]
    if {$file != ""} {
        set fp [open $file "w"]
        SigWindow::SaveSetup $w $fp
        close $fp
    }
}

#*********************************************************
#* Print
#*********************************************************
proc SigWindow::Print {w} {
    upvar #0 $w data

    WaveWindow::Print $data(w:waveWinPane)
}

#*********************************************************
#* SaveSetup
#*
#* Saves the setup of the waveform page to a file
#*********************************************************
proc SigWindow::SaveSetup {w fp args} {
    upvar #0 $w data

    if {$data(-win_name) != "" && $data(-win_name) != " "} {
        set win "-win $data(-win_name) "
    } else {
        set win ""
    }

    puts $fp \
       "#*****************************************************************"
    puts $fp "#**** IVI Signal Window Setup"
    puts $fp "#**** Created: [clock format [clock seconds]]"
    puts $fp \
       "#*****************************************************************"
    puts $fp "set wnd_path \[wave lookup $win\]"
    puts $fp "\$wnd_path geometry [SigWindow::geometry $w]"

    WaveWindow::SaveSetup $data(w:waveWinPane) $fp $data(-win_name) $args

    puts $fp "\$wnd_path siglist_width     [SigWindow::siglist_width $w]"
    puts $fp "\$wnd_path sigvallist_width  [SigWindow::sigvallist_width $w]"
}

#*********************************************************
#* AddRecursive
#*********************************************************
proc SigWindow::AddRecursive {w} {
    upvar #0 $w data
    global CallbackTypes
    set struct [$data(w:sdb_pane) curr_page]
    set ddb [$struct cget -ddb]
    set sdb [$struct cget -sdb]

    set signals ""
    set modules [$struct selected]

    while {[llength $modules] > 0} {
        set new_modules ""
        foreach module $modules {

            foreach mod [$ddb glob -modules "$module$data(sepChar)*"] {
                lappend new_modules "$module$data(sepChar)$mod"
            }

            foreach sig [$ddb glob -signals "$module$data(sepChar)*"] {
                lappend signals "$module$data(sepChar)$sig"
            }
        }
        set modules $new_modules
    }

    callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)
    foreach sig $signals {
        $data(w:SdbWaveReader) add $sig -sdb $sdb 
    }

    callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)
    callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)
}

#*********************************************************
#* BrowseModule
#*********************************************************
proc SigWindow::BrowseModule {w entry} {
    upvar #0 $w data

    if {$data(w:signals) == ""} {
        return
    }

    set slist $data(w:signals)
    $slist delete 0 end

    if {$entry == "@@none@@"} {
        return
    }

    set struct [$data(w:sdb_pane) curr_page]
    set ddb [$struct cget -ddb]

    set sigs [$ddb glob -signals -size "$entry$data(sepChar)*"] 

    eval $slist insert end $sigs

    #*** ensure that we see the first elem... OS X seems
    #*** to show the end
    $slist see 0
}

#*********************************************************
#* WaveSigPrefs
#*********************************************************
proc SigWindow::WaveSigPrefs {w} {
    upvar #0 $w data

    $data(w:waveWinPane) SigPrefDialog
}
#*********************************************************
#* WaveCmd
#*********************************************************
proc SigWindow::WaveCmd args {
    upvar #0 [lindex $args 0] data

    eval $data(w:waveWinPane) [lrange $args 1 end]
}

#*********************************************************
#* SetBindings
#*********************************************************
proc SigWindow::SetBindings {w} {
    upvar #0 $w data

    $w.popup bind $data(w:signals) <<RMB-Popup>>

    bind $w <Configure> "SigWindow::Configure $w"
#    bind $w <Destroy> "prv_wave_close $w"

    #**** This one fails 
    if {0} {
        bind $w <Destroy> "SigWindow::Destroy $w"
    } else {
#        puts "TODO: Fix destroy binding @ SigWindow::SetBindings"
    } 
}

#*********************************************************
#* Destroy
#*********************************************************
proc SigWindow::Destroy {w} {
    upvar #0 $w data

    prv_wave_close $w

    destroy $data(w:waveWinPane)
    destroy $data(w:sdb_pane)
}

#*********************************************************
#* Resize
#*********************************************************
proc SigWindow::Resize {w} {
    upvar #0 $w data

    set rootReq [$data(w:topframe) geometryinfo]
}
 
#*********************************************************
#* Configure
#*********************************************************
proc SigWindow::Configure {w} {
}

#********************************************************************
#* ZoomFull
#********************************************************************
proc SigWindow::WaveZoom {w args} {
    upvar #0 $w data

    eval $data(w:waveWinPane) Zoom $args
}

#********************************************************************
#* AddSeparator
#********************************************************************
proc SigWindow::AddSeparator {w name} {
    upvar #0 $w data

    set result [$data(w:SdbWaveReader) add_sep [lindex $name 0]]
    return $result
}

#********************************************************************
#** AddToWave
#********************************************************************
proc SigWindow::AddToWave {w args} {
    upvar #0 $w data
    global CallbackTypes

    set result   ""
    set res      ""
    set sdb_name ""
    set sdb      ""

    set new_args ""
    set tmp_args [lindex $args 0]
    while {[llength $tmp_args] > 0} {
        set shift 1
        set arg [lindex $tmp_args 0]

        if {$arg == "-sdb"} {
            set sdb_name [lindex $tmp_args 1]
            set shift 2
        } else {
            lappend new_args $arg
        }
        set tmp_args [lrange $tmp_args $shift end]
    }

    if {$sdb_name != ""} {
        #**** Find the sdb
        foreach sdb_n [sdb_mgr list] {
            if {[$sdb_n cget -sdb_id] == $sdb_name} {
                set sdb $sdb_n 
                break
            }
        }
        if {$sdb == ""} {
            ivi_puts "ERROR: Cannot find SDB \"$sdb_name\""
            return
        }
    } else {
        set sdb [lindex [sdb_mgr list] 0]

        if {$sdb == ""} {
            ivi_puts "ERROR: no default SDB"
            return
        }
    }

    if {$sdb == ""} {
        ivi_puts "ERROR: No SDB"
        return
    }

    foreach sig $new_args {
        if [catch {set res [$data(w:SdbWaveReader) add $sig -sdb $sdb]}] {
	    ivi_puts "ERROR: Cannot find signal $sig"
	} else {
            foreach sig $res {
                ivi_puts "Added \"[$sig cget -sig_fullname]\" to wave"
                lappend result $sig
            }
        }
    }

    return $result
}

#*********************************************************
#* AddToWave_Cmd
#*********************************************************
proc SigWindow::AddToWave_Cmd {w add_type} {
    upvar #0 $w data
    global CallbackTypes

    set struct [$data(w:sdb_pane) curr_page]
    set ddb [$struct cget -ddb]
    set sdb [$struct cget -sdb]

    set slist $data(w:signals)

    set sig_list ""

    set mod_path [$struct selected]

    callback disable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)

    if {$add_type == "selected"} {
        set selected [$slist curselection]

        foreach elem $selected {
            set sig_name [$slist get $elem]
            regsub -all {\[[0-9]+:[0-9]+\]} $sig_name "" sig_name
            $data(w:SdbWaveReader) add \
                "$mod_path$data(sepChar)$sig_name" \
                -sdb $sdb
        }
    } elseif {$add_type == "region_all"} {
      
        if {0} {
        set selected [$ddb glob -sigs "$mod_path$data(sepChar)*"]
        foreach elem $selected {
            $data(w:SdbWaveReader) add \
                "$mod_path$data(sepChar)$elem" -sdb $sdb
        }
        }

        $data(w:SdbWaveReader) add "$mod_path$data(sepChar)*" -sdb $sdb

    } else {
        puts "ERROR: Unknown add type"
    }

    callback enable $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)
    callback invoke $CallbackTypes(SDBR_SIGLIST_UPDATE) $data(w:SdbWaveReader)
}

#*********************************************************
#* AddToList_Cmd
#*********************************************************
proc SigWindow::AddToList_Cmd {w} {
    upvar #0 $w data

    set slist $data(w:signals)

    set selected [$slist curselection]

    foreach elem $selected { 
        set sig_name [$slist get $elem]
    }
}


#*********************************************************
#* update
#*********************************************************
proc SigWindow::update {w} {
    upvar #0 $w data

    $data(w:waveWinPane) update
    if {$data(list_present) == 1} {
        $data(w:list_widget) redraw
    }
}

#********************************************************************
#* EditEnum
#********************************************************************
proc SigWindow::EditEnum {w} {
    upvar #0 $w data

    if {[winfo exists $w.enum_win]} {
        puts "ERROR: Already open"
        return
    }

    set sdbr [$data(w:waveWinPane) cget -sdbr]


    set data(w:enum_win) [toplevel $w.enum_win]


    set data(w:enum_edit) [WaveEnum::WaveEnum $data(w:enum_win).enum_edit \
            -sdbr $sdbr]

    set bframe [frame $data(w:enum_win).bframe]
    set apply  [button $bframe.apply -text "Apply" -command \
        [list WaveEnum::Apply $data(w:enum_edit)]]
    set close  [button $bframe.close -text "Close" \
        -command "destroy $data(w:enum_win)"]
        

    pack $bframe -side bottom -fill x
    grid $apply $close
    grid columnconfigure $bframe 0 -weight 0 -pad 2
    grid columnconfigure $bframe 1 -weight 0 -pad 2

    pack $data(w:enum_edit) -expand yes -fill both

    wm title $data(w:enum_win) "$data(-title) Enum"
    bind $data(w:enum_win) <Destroy> [list set $w.enum_win ""]
}

#********************************************************************
#* WaveWinPrefs
#********************************************************************
proc SigWindow::WaveWinPrefs {w} {
    upvar #0 $w data

    set win [DialogBox::DialogBox $w.win_prefs -title "Wave Preferences"]

    upvar #0 $win wdata

    set prefs [WaveWidgetPrefs::WaveWidgetPrefs $win.main \
        -wave_widget [$data(w:waveWinPane) subwidget wave_widget]]

    set button_pane [frame $win.button_pane]
    set ok [button $button_pane.ok -text "Ok" -command \
        "$win ok"]

    set wdata(w:ok) $ok

    set cancel [button $button_pane.cancel -text "Cancel" \
            -command "destroy $win"]

    grid $ok $cancel

    pack $button_pane -side bottom -fill x
    pack $prefs -expand yes -fill both

    $win popup

    if {[$win wait] != ""} {

        WaveWidgetPrefs::ActivateSettings $prefs

        IviConfig::write [file nativename [file join ~ .ivi ivi.conf]]

        #**** Must also get fg/bg colors for signal-list widgets
        set scheme_name \
           "WaveWidget.ColorSchemes.[IviConfig::current WaveWidget.ColorScheme]"
        set siglist_fg [IviConfig::current $scheme_name.siglist_fg]
        set siglist_bg [IviConfig::current $scheme_name.siglist_bg]

        set sig_list [$data(w:waveWinPane) subwidget sigWinSigList]
        set sigl_pad [$data(w:waveWinPane) subwidget sigl_pad]
        set val_list [$data(w:waveWinPane) subwidget val_list]
        set val_list_pad [$data(w:waveWinPane) subwidget val_list_pad]

        $sig_list configure -foreground $siglist_fg -background $siglist_bg
        $sigl_pad configure -background $siglist_bg
        $val_list configure -foreground $siglist_fg -background $siglist_bg
        $val_list_pad configure -background $siglist_bg

        destroy $win
    }
}

#********************************************************************
#* ScrollKnob
#********************************************************************
proc SigWindow::ScrollKnob {w} {
    upvar #0 $w data

    if {[winfo exists "$w.scroll_knob"]} {
        raise $w.scroll_knob
    } else {
        set g [winfo geometry [winfo toplevel $w]]
        set gl [split $g {x+}]

        set top [toplevel $w.scroll_knob]
        set win [WaveScrollWidget::WaveScrollWidget $top.widget]
        pack $win -expand yes -fill both

        ::update

        set s_rw [winfo width $top ]
        set s_rh [winfo height $top]

        #** Want to position new window @ 
        #** - x=$sig_win_x
        #** - y=[$sig_win_y + ($sig_win_h - $s_rh)]
        set y [expr [lindex $gl 3] + ([lindex $gl 1] - $s_rh)]
        set sw_geo "${s_rw}x${s_rh}+[lindex $gl 2]+$y"

        wm title $top "$data(-title) Scroll Win"

        wm geometry $top $sw_geo
    }
}

#*********************************************************
#* geometry
#*********************************************************
proc SigWindow::geometry {w args} {
    upvar #0 $w data

    if {[llength $args] > 0} {
        wm geometry [winfo toplevel $w] [lindex $args 0]
    } else {
        return [wm geometry [winfo toplevel $w]]
    }
}

#*********************************************************
#* siglist_width
#*********************************************************
proc SigWindow::siglist_width {w args} {
    upvar #0 $w data

    upvar #0 $data(w:waveWinPane) wavewin
    set siglist_pw $wavewin(w:waveWinPane)

    if {[llength $args] > 0} {
        ::update idletasks
        $siglist_pw sash place 0 [lindex $args 0] [lindex $args 1]
    } else {
        return [$siglist_pw sash coord 0]
    }
}

#*********************************************************
#* sigvallist_width
#*********************************************************
proc SigWindow::sigvallist_width {w args} {
    upvar #0 $w data

    upvar #0 $data(w:waveWinPane) wavewin
    set sigvallist_pw $wavewin(w:outer_pane)

    if {[llength $args] > 0} {
        ::update idletasks
        $sigvallist_pw sash place 0 [lindex $args 0] [lindex $args 1]
    } else {
        return [$sigvallist_pw sash coord 0]
    }
}

