/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.glib;

import org.gnu.gtk.event.GtkEventType;

/**
 * Base class for all objects that participate in the GLib
 * object system.
 */
public class GObject {

	/** holder for the raw object pointer */
	private int handle = -1;

	protected boolean eventsInitialized = false;

	/**
	 * This class is only instantiable via subclasses.
	 */
	protected GObject() {

		// nothing to do
	}

	/**
	 * Check if two objects refer to the same (native) object.
	 *
	 * @param other the reference object with which to compare.
	 * @return true if both objects refer to the same object.
	 */
	public boolean equals( Object other ){
		return other instanceof GObject && handle == ((GObject) other).getHandle();
	}
	 
	/**
	 * Returns a hash code value for the object. This allows for
	 * using GObjects as keys in hashmaps.
	 *
	 * @return a hash code value for the object.
	 */
	public int hashCode() {
		return getHandle();
	}

	/**
	 * Create a new GObject with a handle to a native resource
	 * returned from a call to the native libraries.
	 * 
	 * @param handle The handle that represents a pointer to a native resource.
	 */
	public GObject(int handle) {
		this.handle = handle;
	}
	
	/** 
	 * Connect an event to a callback method that has the same
	 * name as the event.  For example, the event "clicked" will
	 * be mapped to a method "clicked()" in the object cbrecv.
	 *
	 * @param name The name of the event to map.
	 * @param cvrecv The name of the object that is to recieve
	 *               the event.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, Object cbrecv);

	/** 
	 * Connect a event to a callback method that has the same
	 * name as the event.  For example, the event "clicked" will
	 * be mapped to a method "clicked()" in the object cbrecv.
	 *
	 * @param name The name of the event to map.
	 * @param cvrecv The name of the object that is to recieve
	 *               the event.
	 * @param data User defined data that will be passed to the callback.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, Object cbrecv, Object data);

	/** 
	 * Connect a event to a callback method "func" in object "cbrecv".
	 *
	 * @param name The name of the event to map.
	 * @param func The name of the callback method.
	 * @param cvrecv The name of the object that is to recieve
	 *               the event.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, String func, Object cbrecv);

	/** 
	 * Connect a event to a callback method "func" in object "cbrecv".
	 *
	 * @param name The name of the event to map.
	 * @param func The name of the callback method.
	 * @param cvrecv The name of the object that is to recieve
	 *               the event.
	 * @param data User defined data that will be passed to the callback.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, String func, Object cbrecv, Object data);

	/** 
	 * Connect a event to a static callback method "func" in 
	 * Class "cbrecv".
	 *
	 * @param name The name of the event to map.
	 * @param func The name of the static callback method.
	 * @param cvrecv The name of the class that is to recieve
	 *               the event.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, String func, Class cbrecv);

	/** 
	 * Connect a event to a static callback method "func" in 
	 * Class "cbrecv".
	 *
	 * @param name The name of the event to map.
	 * @param func The name of the static callback method.
	 * @param cvrecv The name of the class that is to recieve
	 *               the event.
	 * @param data User defined data that will be passed to the callback.
	 * @return The handle id of the event that can be used in a
	 *         call to removeEventHandler.
	 */
	public native int addEventHandler(String name, String func, Class cbrecv, Object data);

	/**
	 * Disconnect a event from its' callback method. 
	 *
	 * @param handler_id The handler id of the event.  This is the
	 *                   value returned from a call to addEventHandler().
	 */
	public native void removeEventHandler(int handler);

	 // Methods to associate data with a given object.
	 public Object getData(String key) {
			 return getData(handle, key);
	 }

	 public void setData(String key, Object data) {
			 setData(handle, key, data);
	 }

	 public Class getEventListenerClass(String signal) {
		 return null;
	}

	public GtkEventType getEventType(String signal) {
		return null;
	}


	/**
	 * Get the raw handle value.  This value
	 * should never be modified by the application.  It's
	 * sole use is to pass to native methods.
	 *
	 * @return the handle value.
	 */
	public final int getHandle() {
		return handle;
	}

	protected void setHandle(int hndl) {
 		setData(hndl, "JGObject", this);
 		handle = hndl;
	 }
	 
	 protected GObject retrieveGObject(int hndl) {
	 	return (GObject)getData(hndl, "JGObject");
	 }

	 protected native Object getData(int handle, String key);
	 protected native void setData(int handle, String key, Object data);

	// ------------------------------------------------------------------------
	// static init code
	static {
		System.loadLibrary("gtkjava2.4");
	}

}
