/*
 * JCaptcha, the open source java framework for captcha definition and integration
 * Copyright (c)  2007 jcaptcha.net. All Rights Reserved.
 * See the LICENSE.txt file distributed with this package.
 */

package com.octo.captcha.service;

import com.octo.captcha.engine.CaptchaEngine;

/**
 * Management interface for the {@link CaptchaService} interface.
 */
public interface ManageableCaptchaService extends CaptchaService {
    /**
     * Get the fully qualified class name of the concrete CaptchaEngine used by the service.
     *
     * @return the fully qualified class name of the concrete CaptchaEngine used by the service.
     */
    String getCaptchaEngineClass();

    /**
     * Set the fully qualified class name of the concrete CaptchaEngine used by the service
     *
     * @param theClassName the fully qualified class name of the CaptchaEngine used by the service
     *
     * @throws IllegalArgumentException if className can't be used as the service CaptchaEngine, either because it can't
     *                                  be instanciated by the service or it is not a ImageCaptchaEngine concrete
     *                                  class.
     */
    void setCaptchaEngineClass(String theClassName)
            throws IllegalArgumentException;

    /**
     * @return the engine served by this service
     */
    CaptchaEngine getEngine();

    /**
     * Updates the engine served by this service
     * @param engine
     */
    void setCaptchaEngine(CaptchaEngine engine);

    /**
     * Get the minimum delay (in seconds) a client can be assured that a captcha generated by the service can be
     * retrieved and a response to its challenge tested
     *
     * @return the maximum delay in seconds
     */
    int getMinGuarantedStorageDelayInSeconds();

    /**
     * set the minimum delay (in seconds)a client can be assured that a captcha generated by the service can be
     * retrieved and a response to its challenge tested
     *
     * @param theMinGuarantedStorageDelayInSeconds
     *         the minimum guaranted delay
     */
    void setMinGuarantedStorageDelayInSeconds(int theMinGuarantedStorageDelayInSeconds);

    /**
     * Get the number of captcha generated since the service is up WARNING : this value won't be significant if the real
     * number is > Long.MAX_VALUE
     *
     * @return the number of captcha generated since the service is up
     */
    long getNumberOfGeneratedCaptchas();

    /**
     * Get the number of correct responses to captcha challenges since the service is up. WARNING : this value won't be
     * significant if the real number is > Long.MAX_VALUE
     *
     * @return the number of correct responses since the service is up
     */
    long getNumberOfCorrectResponses();

    /**
     * Get the number of uncorrect responses to captcha challenges since the service is up. WARNING : this value won't
     * be significant if the real number is > Long.MAX_VALUE
     *
     * @return the number of uncorrect responses since the service is up
     */
    long getNumberOfUncorrectResponses();

    /**
     * Get the curent size of the captcha store
     *
     * @return the size of the captcha store
     */
    int getCaptchaStoreSize();

    /**
     * Get the number of captchas that can be garbage collected in the captcha store
     *
     * @return the number of captchas that can be garbage collected in the captcha store
     */
    int getNumberOfGarbageCollectableCaptchas();

    /**
     * Get the number of captcha garbage collected since the service is up WARNING : this value won't be significant if
     * the real number is > Long.MAX_VALUE
     *
     * @return the number of captcha garbage collected since the service is up
     */
    long getNumberOfGarbageCollectedCaptcha();

    /**
     * This max size is used by the service : it will throw a CaptchaServiceException if the store is full when a client
     * ask for a captcha.
     */
    void setCaptchaStoreMaxSize(int size);

    /**
     * @return the desired max size of the captcha store
     */
    int getCaptchaStoreMaxSize();

    /**
     * Garbage collect the captcha store, means all old capthca (captcha in the store wich has been stored more than the
     * MinGuarantedStorageDelayInSecond
     */
    void garbageCollectCaptchaStore();

    /**
     * Empty the Store
     */
    void emptyCaptchaStore();

    /**
     * @return the max captchaStore load before garbage collection of the store
     */
    int getCaptchaStoreSizeBeforeGarbageCollection();

    /**
     * max captchaStore size before garbage collection of the store
     */
    void setCaptchaStoreSizeBeforeGarbageCollection(int captchaStoreSizeBeforeGarbageCollection);
}
