# The Jokosher Extension API
# write proper docstrings so that we can autogenerate the docs

import os, gtk, imp, pickle, Globals, pkg_resources
import gettext
import traceback

_ = gettext.gettext

# Define some constants
EXTENSION_DIR_USER = os.path.expanduser('~/.jokosher/extensions/')
EXTENSION_DIRS = [EXTENSION_DIR_USER, '/usr/share/jokosher/extensions/']
# add your own extension dirs with envar JOKOSHER_EXTENSION_DIRS, colon-separated
OVERRIDE_EXTENSION_DIRS = os.environ.get('JOKOSHER_EXTENSION_DIRS','')
if OVERRIDE_EXTENSION_DIRS:
	EXTENSION_DIRS = OVERRIDE_EXTENSION_DIRS.split(':') + EXTENSION_DIRS
PREFERRED_EXTENSION_DIR = EXTENSION_DIRS[0]

# A couple of small constants; they get used as the default response from a
# dialog, and they're nice and high so they don't conflict with anything else
RESP_INSTALL = 9999
RESP_REPLACE = 9998

# Work out whether I'm being imported by a extension that's being run directly
# or whether I'm being imported by a extension run by Jokosher. If I'm being
# run directly then that isn't right, and probably means that the user has
# just clicked on an extension in the file manager. To be nice to them, we
# offer to install the extension in their .jokosher folder.
import inspect
extension_that_imported_me = inspect.currentframe().f_back
try:
	thing_that_imported_extension = extension_that_imported_me.f_back 
except:
	thing_that_imported_extension = None
	
if thing_that_imported_extension is None and \
			os.path.split(extension_that_imported_me.f_code.co_filename)[1] != 'JokosherApp.py':
	# the extension is being run directly; pop up the error 
	try:
		import gtk
	except:
		# no Gtk either! Print a message and die
		import sys
		Globals.debug(_("This is a Jokosher extension; it is not meant to be run directly."))
		Globals.debug(_("To install it, move it to the directory %s\nand run Jokosher.") % (EXTENSION_DIR_USER))
		sys.exit(1)
		
	message = _("This is a Jokosher extension, which needs to be installed. Would you like to install it?")
	d = gtk.MessageDialog(message_format=message, type=gtk.MESSAGE_ERROR)
	d.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, _('Install'), RESP_INSTALL)
	d.set_default_response(RESP_INSTALL)
	ret = d.run()
	d.destroy()
	if ret == RESP_INSTALL:
		extension_path_and_file = extension_that_imported_me.f_globals['__file__']
		extension_file_name = os.path.split(extension_path_and_file)[1]
		new_extension_path_and_file = os.path.join(PREFERRED_EXTENSION_DIR, extension_file_name)
		if os.path.exists(new_extension_path_and_file):
			message_template = _("You already have a extension with the name %s installed; would you like to replace it?")
			message = message_template % os.path.splitext(extension_file_name)[0]
			d = gtk.MessageDialog(message_format=message, type=gtk.MESSAGE_QUESTION)
			d.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, _('Replace'), RESP_REPLACE)
			d.set_default_response(RESP_REPLACE)
			ret = d.run()
			d.destroy()
			if ret != RESP_REPLACE:
				sys.exit(0)
		# confirm that the new path exists!
		try:
			os.makedirs(os.path.split(new_extension_path_and_file)[0])
		except:
			pass # already exists
		# and move the extension
		os.rename(extension_path_and_file, new_extension_path_and_file)
		d = gtk.MessageDialog(message_format=_("Your new extension is now available in Jokosher!"), buttons=gtk.BUTTONS_OK)
		d.destroy()
		sys.exit(0)

############################################################################
############# The actual extension API #####################################
############################################################################
#required API imports
import ConfigParser
import gst, gobject

def exported_function(f):
	"""
		Wraps any exported functions so that exceptions do not cross the exported API.
		Any exceptions caught by this should be a return error code from exported function.
	"""
	def wrapped(*args, **kwargs):
		try:
			result = f(*args, **kwargs)
			return result
		except:
			Globals.debug("EXTENSION API BUG:\nUnhandled exception thrown in exported function: %s\n%s" %
				(f.func_name, traceback.format_exc()))
			return -2
	wrapped.__doc__ = f.__doc__
	wrapped.__name__ = f.func_name
	return wrapped

class ExtensionAPI:

	def __init__(self, mainapp):
		self.mainapp = mainapp
	
	@exported_function	
	def add_menu_item(self, menu_item_name, callback_function):
		"""
		   Adds a menu item to a Jokosher extension menu.
		"""
		extensions_menu = self.mainapp.wTree.get_widget("extensionsmenu").get_submenu()
		new_menu_item = gtk.MenuItem(menu_item_name)
		new_menu_item.connect("activate", callback_function)
		extensions_menu.prepend(new_menu_item)
		new_menu_item.show()
		return new_menu_item
	
	@exported_function
	def play(self, play_state=True):
		"""
		   If play_state is True, it will play the project from the beginning.
		   Otherwise, it will stop all playing.
		"""
		#Stop current playing (if any) and set to playhead to the beginning
		self.mainapp.Stop()
		if play_state:
			#Commence playing
			self.mainapp.Play()

	@exported_function	
	def stop(self):
		"""
		   Stops the project if it is currently playing.
		   Same as play(play_state=False)
		"""
		self.mainapp.Stop()
	
	@exported_function
	def add_file_to_selected_instrument(self, uri, position=0):
		"""
		   Creates a new event from the file at the given URI and 
		   adds it to the first selected instrument at position (in seconds).
		   Return values:
		   0: success
		   1: bad URI or file could not be loaded
		   2: no instrument selected
		"""
		instr = None
		for i in self.mainapp.project.instruments:
			if i.isSelected:
				instr = i
				break
		
		if not instr:
			#No instrument selected
			return 2
		
		instr.addEventFromFile(position, uri)
		#TODO: find out if the add failed and return 1
		return 0

	@exported_function
	def add_file_to_instrument(self, instr_id, uri, position=0):
		"""
		   Creates a new event from the file at the given URI and 
		   adds it to the instrument with id 'instr_id' at position (in seconds).
		   Return values:
		   0: success
		   1: bad URI or file could not be loaded
		   2: instrument with id 'instr_id' not found
		"""
		for instr in self.mainapp.projects.instruments:
			if instr.id == instr_id:
				instr.addEventFromFile(position, uri)
				#TODO: find out if the add failed and return 1
				return 0
	
		return 2

	@exported_function
	def list_available_instrument_types(self):
		"""
		   Returns a list of tuples in the format:
		   (instr_name, instr_type, instr_pixbuf)
		   for each of the *.instr files that have been cached.
		   These instruments are *not* the ones in the project,
		   but only the ones available to be added to the project.
		"""
		return [(x[0], x[1], x[2].copy()) for x in Globals.getCachedInstruments()]
		
	@exported_function
	def add_instrument(self, instr_type, instr_name=None):
		"""
		   Adds an instrument with the type 'instr_type' and
		   name 'instr_name' from get_available_instruments() 
		   to the project.
		   Return values:
		   -1: that instrument type does not exist
		   >0: success
		   If the instrument is successfully added,
		   the return value will be the ID of that instrument.
		"""
		for i in Globals.getCachedInstruments():
			if i[1] == instr_type:
				if instr_name:
					instr_index = self.mainapp.project.AddInstrument(instr_name, i[1], i[2])
				else:
					instr_index = self.mainapp.project.AddInstrument(i[0], i[1], i[2])
					#i[0] is the default instrument name, i[1] is the instrument type, i[2] is the instrument icon in the .instr file
				self.mainapp.UpdateDisplay()
				return instr_index
		return -1
		
	@exported_function
	def list_project_instruments(self):
		"""
		   Returns a list of tuples in the format:
		   (instr_id_number, instr_name, instr_type, instr_pixbuf)
		   for each of the instruments currently shown in the project.
		"""
		return [(instr.id, instr.name, instr.instrType, instr.pixbuf.copy()) for instr in self.mainapp.project.instruments]
		
	@exported_function
	def delete_instrument(self, instrumentID):
		"""
		   Removes the instrument with the ID
		   that equals instrumentID.
		"""
		self.mainapp.project.DeleteInstrument(instrumentID)
		self.mainapp.UpdateDisplay()
		#time for a Newfie Joke: 
		#How many Newfies does it take to go ice fishing?
		#Four. One to cut a hole in the ice and three to push the boat through.

	def __get_config_dict_fn(self):
		"""
			 Calculate the config dictionary filename for the calling
			 extension.
		"""
		# First, see if there is a config directory at all
		CONFIGPATH = os.path.join(EXTENSION_DIR_USER,'../extension-config')
		if not os.path.exists(CONFIGPATH):
			os.makedirs(CONFIGPATH)
		# Next, check if this extension has a saved config dict
		# we go back twice because our immediate caller is (get|set)_config_value
		mycallerframe = inspect.currentframe().f_back.f_back
		mycallerfn = os.path.split(mycallerframe.f_code.co_filename)[1]
		mycaller = os.path.splitext(mycallerfn)[0]
		config_dict_fn = os.path.join(CONFIGPATH,mycaller + ".config")
		return os.path.normpath(config_dict_fn)

	@exported_function
	def get_config_value(self, key):
		"""
		   Returns the config value saved under this key,
		   or None if there is no such value.
		"""
		try:
			# Open the extension's config dict
			# Return the value
			config_dict_fn = self.__get_config_dict_fn()
			fp = open(config_dict_fn)
			config_dict = pickle.load(fp)
			fp.close()
			return config_dict[key]
		except:
			return None

	@exported_function
	def set_config_value(self, key, value):
		"""
		   Sets a new config value under key for later retrieval.
		"""
		config_dict_fn = self.__get_config_dict_fn()
		if os.path.exists(config_dict_fn):
			fp = open(config_dict_fn)
			config_dict = pickle.load(fp)
			fp.close()
		else:
			config_dict = {}
		# Set the config value
		config_dict[key] = value
		# And save it again
		fp = open(config_dict_fn,"wb")
		pickle.dump(config_dict, fp)
		fp.close()

	@exported_function
	def set_instrument_volume(self, instr_id, instr_volume):
		"""
		   Sets the volume of instrument with id 'instr_id'
		   to volume 'instr_volume'
		   
		   Return Values:
		   0: success
		   1: instrument with id 'instr_id' doesn't exist
		"""
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:
				instr.SetVolume(min(instr_volume, 1))
				return 0
		return 1

	@exported_function
	def get_instrument_volume(self, instr_id):
		"""
		   returns the lever of instrument with id 'instr_id'

		   Return Values:
		   On Success: volume of instrument with id 'instr_id'
		   1: instrument with id 'instr_id' does not exist
		"""
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:
				return instr.volume
		return 1

	@exported_function
	def toggle_mute_instrument(self, instr_id):
		"""
		   mutes the instrument with id 'instr_id'

		   Return Values:
		   0: success
		   1: instrument with id 'instr_id' doesn't exist
		"""
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:
				instr.ToggleMuted(False)
				return 0
		return 1

	#My Nan and Pop from Newfoundland aren't quite this bad, but they're close: 
	#http://youtube.com/watch?v=It_0XzPVHaU

	@exported_function
	def get_instrument_effects(self, instr_id):
		"""
		   Gets the current effects applied to instrument
		   with id 'instr_id'

		   Return Values:
		   success: returns the effects applied to instrument with id 'instr_id'
		   1: instrument with id 'instr_id' not found
		"""
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:
				#return a copy so they can't append or remove items from our list
				return instr.effects[:]

		return 1

	@exported_function
	def list_available_effects(self):
		"""
		   returns the available LADSPA effects
		"""
		#return a copy so they can't append or remove items from our list
		return Globals.LADSPA_NAME_MAP[:]

	@exported_function
	def add_instrument_effect(self, instr_id, effect_name):
		"""
		   Adds the effect 'effect_name' to instrument
		   with id 'instr_id'

		   Return Values:
		   0: success
		   1: instrument with id 'instr_id' not found
		   2: LADSPA plugin 'effect_name' not found
		"""
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:
				if instr.effects == []:
					instr.converterElement.unlink(instr.volumeElement)

				for effect in Globals.LADSPA_NAME_MAP:
					if effect[0] == effect_name:
						instr.effects.append(gst.element_factory_make(effect_name, effect_name))
						return 0
				return 2
		return 1

	@exported_function
	def remove_instrument_effect(self, instr_id, effect_num):
		"""
		   This function removes the effect of index 'effect_num' 
		   (in the instrument effects array) from instrument with
		   id 'instr_id'

		   Return Values:
		   0: success
		   1: effect_num out of range
		   2: isntrument with id 'instr_id' not found
		"""
		
		for instr in self.mainapp.project.instruments:
			if instr.id == instr_id:	
				if effect_num <= len(instr.effects) - 1:
					try:
						#FIXME: throws a gstreamer warning sometimes...I don't know gstreamer well, so 
						#I don't even know why this is here
						instr.effectsbin.remove(instr.effects[effect_num])
					except:
						pass
			
					instr.effects.pop(effect_num)
					return 0
				return 1
		return 2

	#TODO: function for editing existing effects

	@exported_function
	def create_new_instrument_type(self, defaultName, typeString, imagePath):
		"""
		   Creates and new instrument type in the user's 
		   ~/.jokosher/instruments folder. It will then be automatically
		   loaded on startup.
		   
		   defaultName - the en_GB name of the instrument
		   typeString - a unique string to this particular instrument file
		   imagePath - absolute path to the instruments image
		   
		   Return values:
		   0: success
		   1: file exists or defaultName is already used by a loaded instrument
		   2: cannot load image
		   3: cannot write to ~/.jokosher/instruments or ~/.jokosher/instruments/images
		"""
		typeList = [x[1] for x in Globals.getCachedInstruments()]
		if typeString in typeList:
			#if the typeString is already being used, just add a number to the end like "guitar2"
			count = 1
			path = os.path.join(Globals.INSTR_PATHS[1], "%s" + ".instr")
			typeString = typeString + str(count)
			while typeString in typeList and not os.path.exists(path% (typeString)):
				count += 1
				typeString  = typeString[:-1] + str(count)
				
		#check if the type string is being used by any other loaded instrument
		if defaultName in [x[0] for x in Globals.getCachedInstruments()]:
			Globals.debug("An instrument already loaded with name", defaultName)
			return 1
		
		try:
			pixbuf = gtk.gdk.pixbuf_new_from_file(imagePath)
		except (gobject.GError, TypeError):
			return 2
		if pixbuf.get_width() != 48 or pixbuf.get_height() != 48:
			pb = pixbuf.scale_simple(48, 48, gtk.gdk.INTERP_BILINEAR)
			pixbuf = pb
		try:
			pixbuf.save(Globals.INSTR_PATHS[1] + "/images/" + typeString + ".png", "png")
		except gobject.GError:
			return 3
		
		Globals.debug("Creating new instrument type")
		instr = ConfigParser.ConfigParser()
		instr.add_section("core")
		instr.add_section("i18n")
		instr.set("core", "icon", typeString + ".png")
		instr.set("core", "type", typeString)
		instr.set("i18n", "en", defaultName)

		instrument_file = os.path.join(Globals.INSTR_PATHS[1], typeString + ".instr")
		try:
			file = open(instrument_file, 'w')
			instr.write(file)
		except IOError:
			if file:
				file.close()
			return 3
		else:
			file.close()
		
		#refresh the instrument list so our new instrument gets loaded.
		Globals.getCachedInstruments(checkForNew=True)
		
		return 0
		
	@exported_function
	def add_export_format(self, description, extension, encodeBin):
		"""
		   Adds a new format that the use can select from
		   the filetype drop down box in the 'Mixdown Project' dialog.
		   description - string for the drop down box. ex: 'Ogg Vorbis (.ogg)'
		   extension - string of the file extension without a '.'. ex: 'ogg'
		   encodeBin - string used by gst_parse_bin_from_description to create
		         a bin that can encode and mux the audio when added to a pipeline. ex: 'vorbisenc ! oggmux'
		   
		   Return values:
		   0: success
		   1: invalid options
		   2: a format with the same three values already exists
		   3: cannot parse or create encoder/muxer bin
		"""
		
		if not description or not extension and not pipelineString:
			return 1
		try:
			bin = gst.gst_parse_bin_from_description("audioconvert ! %s" % encodeBin, True)
			del bin
		except gobject.GError:
			return 3
		
		propslist = (description, extension, encodeBin)
		propsdict = dict(zip(Globals._export_template, propslist))
		if propsdict in Globals.EXPORT_FORMATS:
			return 2
		else:
			Globals.EXPORT_FORMATS.append(propsdict)
			return 0
		
	@exported_function
	def remove_export_format(self, description, extension, encodeBin):
		"""
		   Removes an export format that was previously added using
		   add_export_format. The parameters are the same as the ones
		   that were passed to the add_export_format function.
		   
		   Return values:
		   0: successfully removed the export format
		   1: no export format exists with those parameters
		"""
		propslist = (description, extension, encodeBin)
		propsdict = dict(zip(Globals._export_template, propslist))
		if propsdict in Globals.EXPORT_FORMATS:
			Globals.EXPORT_FORMATS.remove(propsdict)
			return 0
		else:
			return 1
		
		
API = None
