/*
 * jpilot-Mail Copyright (C) 2000 Oliver Kurth
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

/*
 * $Id: mail.c,v 1.12 2003/08/30 20:20:39 rousseau Exp $ 
 */

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include <ctype.h>
#include <gdk/gdkkeysyms.h>

#include <gtk/gtk.h>
#include "libplugin.h"
#include <pi-file.h>
#include <pi-mail.h>
#include <pi-source.h>

#include "utils.h"
#include "masqmail.h"
#include "mail.h"
#include "mime_coder.h"

#define SELECT_INBOX_FILENAME 1
#define SELECT_SENTBOX_FILENAME 2
static int Selection = 0;

static GtkWidget *prefs_window = NULL;

static GtkWidget *mail_main_hbox = NULL;

static int show_category = 0;
static int glob_row_selected;

static struct MyMail *glob_mymail_list = NULL;

struct Preferences prefs;
struct MailSyncPref mailSyncPref;
struct MailSignaturePref mailSigPref;

static GtkWidget *FileSelector = NULL;
static GtkWidget *inbox_filename_entry;
static GtkWidget *sentbox_filename_entry;
static void file_select(GtkWidget *widget, gpointer data);
static void file_cancel(void);
static void file_ok_sel(GtkWidget *w, GtkFileSelection *fs);

void plugin_version(int *major_version, int *minor_version)
{
	*major_version = 0;
	*minor_version = 99;
}

static void free_mymail_list(struct MyMail **ppM)
{
	struct MyMail *pM, *next_pM;

	for (pM = *ppM; pM; pM = next_pM)
	{
		next_pM = pM->next;
		free(pM);
	}
	*ppM = NULL;
}

/*
 * Given a user id, this returns the passwd structure
 */
static struct passwd *getuserent(uid_t id)
{
	struct passwd *ent = NULL;

	ent = getpwuid(id);
	if (ent == NULL)
		jp_logf(JP_LOG_WARN, "Mail::getuserent pwent==NULL\n");

	return ent;
}

/*
 * This is a mandatory plugin function.
 */
int plugin_get_name(char *name, int len)
{
	snprintf(name, len, "%s %s", PACKAGE, VERSION);
	return 0;
}

/*
 * This is an optional plugin function.
 * This is the name that will show up in the plugins menu in J-Pilot.
 */
int plugin_get_menu_name(char *name, int len)
{
	strncpy(name, "Mail", len);
	return 0;
}

/*
 * This is an optional plugin function.
 * This is the name that will show up in the plugins help menu in J-Pilot.
 * If this function is used then plugin_help must be also.
 */
int plugin_get_help_name(char *name, int len)
{
	strncpy(name, "About Mail", len);
	return 0;
}

/*
 * This is an optional plugin function.
 * This is the palm database that will automatically be synced.
 */
int plugin_get_db_name(char *name, int len)
{
	strncpy(name, "MailDB", len);
	return 0;
}

#if 0
static void cb_not_yet(GtkWidget * widget, gpointer data)
{
	if (GTK_IS_WIDGET(data))
	{
		gtk_widget_destroy(data);
	}
}
#endif

static gboolean prefs_write()
{
	gchar fname[255];
	FILE *fptr;

	get_home_file_name("jpilot-Mail.rc", fname, 255);
	fptr = fopen(fname, "wt");

	if (fptr)
	{
		fprintf(fptr, "smtp_server %s\n", prefs.smtp_server);
		fprintf(fptr, "smtp_port %d\n", prefs.smtp_port);
		fprintf(fptr, "from_address %s\n", prefs.from_address);
		fprintf(fptr, "from_name %s\n", prefs.from_name);
		fprintf(fptr, "default_domain %s\n", prefs.default_domain);
		fprintf(fptr, "folder_path_inbox %s\n", prefs.folder_path_inbox);
		fprintf(fptr, "folder_path_sentmail %s\n", prefs.folder_path_sentmail);
		fprintf(fptr, "pilot_charset %s\n", prefs.pilot_charset);
		if (prefs.header_encoding == ENC_QP) {
			fprintf(fptr, "header_encoding %s\n", EncodingName[ENC_QP]);
		} else if (prefs.header_encoding == ENC_B64) {
			fprintf(fptr, "header_encoding %s\n", EncodingName[ENC_B64]);
		}
		fprintf(fptr, "get_read %d\n", prefs.get_read);
		fprintf(fptr, "presync_get %d\n", prefs.presync_get);
		fprintf(fptr, "postsync_send %d\n", prefs.postsync_send);

		fclose(fptr);
		return TRUE;
	}
	return FALSE;
}

static gboolean prefs_read()
{
	gchar fname[255];
	FILE *fptr;

	get_home_file_name("jpilot-Mail.rc", fname, 255);
	fptr = fopen(fname, "rt");

	if (fptr)
	{
		gchar *line = NULL;
		gint line_size = 0;

		while (!feof(fptr))
		{
			gchar key[64];
			gchar *p, *q = key;

			getline(&line, &line_size, fptr);
			g_strchomp(line);
			p = line;
			while (!isspace(*p) && *p)
				*q++ = *p++;
			if (*p)
			{
				*q = 0;
				p++;

				if (strcasecmp("smtp_server", key) == 0)
				{
					g_free(prefs.smtp_server);
					prefs.smtp_server = g_strdup(p);
				}
				else if (strcasecmp("smtp_port", key) == 0)
				{
					prefs.smtp_port = atoi(p);
				}
				else if (strcasecmp("from_address", key) == 0)
				{
					g_free(prefs.from_address);
					prefs.from_address = g_strdup(p);
				}
				else if (strcasecmp("from_name", key) == 0)
				{
					g_free(prefs.from_name);
					prefs.from_name = g_strdup(p);
				}
				else if (strcasecmp("default_domain", key) == 0)
				{
					g_free(prefs.default_domain);
					prefs.default_domain = g_strdup(p);
				}
				else if (strcasecmp("folder_path_inbox", key) == 0)
				{
					g_free(prefs.folder_path_inbox);
					prefs.folder_path_inbox = g_strdup(p);
				}
				else if (strcasecmp("pilot_charset", key) == 0)
				{
					g_free(prefs.pilot_charset);
					prefs.pilot_charset = g_strdup(p);
				}
				else if (strcasecmp("header_encoding", key) == 0)
				{
					if (g_strcasecmp(p, EncodingName[ENC_B64]) == 0) {
						prefs.header_encoding = ENC_B64;
					} else if (g_strcasecmp(p, EncodingName[ENC_QP]) == 0) {
						prefs.header_encoding = ENC_QP;
					}
				}
				else if (strcasecmp("folder_path_sentmail", key) == 0)
				{
					g_free(prefs.folder_path_sentmail);
					prefs.folder_path_sentmail = g_strdup(p);
				}
				else if (strcasecmp("get_read", key) == 0)
				{
					prefs.get_read = atoi(p);
				}
				else if (strcasecmp("presync_get", key) == 0)
				{
					prefs.presync_get = atoi(p);
				}
				else if (strcasecmp("postsync_send", key) == 0)
				{
					prefs.postsync_send = atoi(p);
				}
			}
		}

		fclose(fptr);
		return TRUE;
	}
	return FALSE;
}

static void prefs_apply()
{
	GtkWidget *entry;

	g_free(prefs.smtp_server);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "smtp_server");
	prefs.smtp_server = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "smtp_port");
	prefs.smtp_port = atoi(gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1));

	g_free(prefs.pilot_charset);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "pilot_charset");
	prefs.pilot_charset = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	g_free(prefs.from_address);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "from_address");
	prefs.from_address = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	g_free(prefs.from_name);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "from_name");
	prefs.from_name = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	g_free(prefs.default_domain);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "default_domain");
	prefs.default_domain = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	g_free(prefs.folder_path_inbox);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "folder_path_inbox");
	prefs.folder_path_inbox =
		gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	g_free(prefs.folder_path_sentmail);
	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "folder_path_sentmail");
	prefs.folder_path_sentmail =
		gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "get_read");
	prefs.get_read = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(entry));

	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "presync_get");
	prefs.presync_get = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(entry));

	entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "postsync_send");
	prefs.postsync_send =
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(entry));

	{
		GtkWidget *menu, *item;
		char *s;

		entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "sync_truncate");
		menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(entry));

		item = gtk_menu_get_active(GTK_MENU(menu));
		mailSyncPref.truncate =
			atoi(gtk_object_get_data(GTK_OBJECT(item), "value"));

		jp_logf(JP_LOG_DEBUG,
			"selected: truncate = %d\n", mailSyncPref.truncate);

		entry = gtk_object_get_data(GTK_OBJECT(prefs_window), "header_encoding");
		menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(entry));

		item = gtk_menu_get_active(GTK_MENU(menu));
		s = gtk_object_get_data(GTK_OBJECT(item), "value");
		if (!strcmp(s, EncodingName[ENC_QP])) {
			prefs.header_encoding = ENC_QP;
		} else if (!strcmp(s, EncodingName[ENC_B64])) {
			prefs.header_encoding = ENC_B64;
		}
	}
}

static void prefs_read_pilot()
{
	gchar fname[255];
	struct pi_file *pi_fp;
	int size;
	int id;
	unsigned long type, mail_type = makelong("mail");
	void *record;

	jp_logf(JP_LOG_DEBUG, "Mail: prefs_read_pilot()\n");

	get_home_file_name("Saved Preferences.prc", fname, 255);
	pi_fp = pi_file_open(fname);

	if (pi_fp)
	{
		int i = 0;

		while (pi_file_read_resource PI_ARGS((pi_fp, i++, &record, &size,
					&type, &id)) >= 0)
		{
			if (type == mail_type)
			{
				switch (id)
				{
					case 1:
						/*
						 * 'local' sync prefs: 
						 */
						unpack_MailSyncPref(&mailSyncPref, record + 2, size);
						jp_logf(JP_LOG_DEBUG, "Mail: truncate = %d\n",
							mailSyncPref.truncate);
						break;
					case 2:
						/*
						 * 'remote' sync prefs: 
						 */
						break;
					case 3:
						/*
						 * signature 
						 */
						unpack_MailSignaturePref(&mailSigPref, record + 2,
							size);
						jp_logf(JP_LOG_DEBUG, "Mail: signature = %s\n",
							mailSigPref.signature);
						break;
				}
			}
		}
		pi_file_close(pi_fp);
	}
	else
		jp_logf(JP_LOG_WARN, "could not open %s: %s\n", fname, strerror(errno));
}

static void cb_prefs_destroy(GtkWidget * widget, gpointer data)
{
	prefs_window = NULL;
}

static void cb_prefs_cancel(GtkWidget * widget, gpointer data)
{
	jp_logf(JP_LOG_DEBUG, "cb_prefs_cancel\n");
	if (GTK_IS_WIDGET(data))
		gtk_widget_destroy(data);
}

static void cb_prefs_ok(GtkWidget * widget, gpointer data)
{
	jp_logf(JP_LOG_DEBUG, "cb_prefs_apply\n");

	prefs_apply();
	prefs_write();

	if (GTK_IS_WIDGET(data))
		gtk_widget_destroy(data);
}

static void cb_pref_menu(GtkWidget * widget, gpointer data)
{
	if (!widget)
		return;
	if (!(GTK_CHECK_MENU_ITEM(widget))->active)
		return;

	/*
	 * gtk_object_set_data(GTK_OBJECT(prefs_window), "from_address",
	 * data); 
	 */
	return;
}

static GtkWidget *make_pref_menu(gchar * item_texts[], gchar * sel_text)
{
	GtkWidget *pref_menu;
	GtkWidget *menu_item;
	GtkWidget *menu;
	GSList *group;
	gint i, sel_i = 0;

	jp_logf(JP_LOG_DEBUG, "make_pref_menu()\n");

	pref_menu = gtk_option_menu_new();

	menu = gtk_menu_new();
	group = NULL;

	i = 0;
	while (item_texts[i])
	{
		gchar *text_ptr = item_texts[i];

		menu_item = gtk_radio_menu_item_new_with_label(group, text_ptr);
		gtk_signal_connect(GTK_OBJECT(menu_item), "activate", cb_pref_menu,
			text_ptr);
		gtk_object_set_data(GTK_OBJECT(menu_item), "value", text_ptr);
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
		gtk_menu_append(GTK_MENU(menu), menu_item);

		if (strcmp(text_ptr, sel_text) == 0)
		{
			sel_i = i;
			gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),
				i);
		}
		gtk_widget_show(menu_item);
		i++;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(pref_menu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(pref_menu), sel_i);

	return pref_menu;
}

gchar *truncate_items[] = {
	"250", "500", "1000", "2000", "4000", "6000", "8000", NULL
};

void cb_prefs(GtkWidget * widget, gpointer data)
{
	GtkWidget *label, *separator;
	GtkWidget *button;

	/*
	 * GtkWidget *radio_button_sendmail, *radio_button_SMTP;
	 */
	GtkWidget *check_postsync_send, *check_presync_get;
	GtkWidget *vbox, *vbox_tmp, *hbox_tmp;
	GtkWidget *vbox_servers, *vbox_identity, *vbox_sync, *vbox_sig;
	GtkWidget *notebook;
	GtkWidget *entry, *sig_text;

	jp_logf(JP_LOG_DEBUG, "cb_prefs\n");
	if (GTK_IS_WINDOW(prefs_window))
	{
		jp_logf(JP_LOG_DEBUG, "pref_window is already up\n");
		return;
	}

	prefs_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	/*
	 * gtk_window_set_default_size(GTK_WINDOW(window), 500, 300); 
	 */

	gtk_container_set_border_width(GTK_CONTAINER(prefs_window), 10);
	gtk_window_set_title(GTK_WINDOW(prefs_window), "Mail Preferences");

	gtk_signal_connect(GTK_OBJECT(prefs_window), "destroy",
		GTK_SIGNAL_FUNC(cb_prefs_destroy), prefs_window);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 2);
	gtk_container_add(GTK_CONTAINER(prefs_window), vbox);

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	/*
	 * ******** Mail Delivery **********
	 */

	{
		gchar *tmp_text;

		label = gtk_label_new("Mail Delivery");
		vbox_servers = gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(vbox_servers), 5);
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_servers, label);

		/*
		 * sendmail/SMTP 
		 */
		vbox_tmp = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), vbox_tmp, FALSE, FALSE, 0);

		/*
		 * group = NULL;
		 * 
		 * radio_button_sendmail = gtk_radio_button_new_with_label(group,
		 * "sendmail"); group =
		 * gtk_radio_button_group(GTK_RADIO_BUTTON(radio_button_sendmail));
		 * gtk_box_pack_start(GTK_BOX(vbox_tmp), radio_button_sendmail,
		 * FALSE, FALSE, 0);
		 * 
		 * radio_button_SMTP = gtk_radio_button_new_with_label(group,
		 * "SMTP"); group =
		 * gtk_radio_button_group(GTK_RADIO_BUTTON(radio_button_SMTP));
		 * gtk_box_pack_start(GTK_BOX(vbox_tmp), radio_button_SMTP, FALSE,
		 * FALSE, 0); 
		 */

		/*
		 * Mail server 
		 */

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Mail server ");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(entry), 79);
		gtk_entry_set_text(GTK_ENTRY(entry), prefs.smtp_server);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "smtp_server", entry);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Port ");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		tmp_text = g_strdup_printf("%d", prefs.smtp_port);
		entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(entry), 5);
		gtk_entry_set_text(GTK_ENTRY(entry), tmp_text);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "smtp_port", entry);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Default domain ");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(entry), 79);
		gtk_entry_set_text(GTK_ENTRY(entry), prefs.default_domain);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "default_domain", entry);

		hbox_tmp = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, TRUE, 5);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Pilot charset ");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		entry = gtk_entry_new();
		gtk_entry_set_max_length(GTK_ENTRY(entry), 49);
		gtk_entry_set_text(GTK_ENTRY(entry), prefs.pilot_charset);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "pilot_charset", entry);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_servers), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Header Encoding ");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		{
			GtkWidget *menu;
			gchar *header_encodings[] = {
				(gchar *)EncodingName[ENC_QP],
				(gchar *)EncodingName[ENC_B64],
				NULL
			};

			menu = make_pref_menu(header_encodings,
					(gchar *)EncodingName[prefs.header_encoding]);
			gtk_box_pack_end(GTK_BOX(hbox_tmp), menu, FALSE, FALSE, 0);
			gtk_object_set_data(GTK_OBJECT(prefs_window), "header_encoding", menu);
		}
	}
	/*
	 * ******** Identity **********
	 */

	label = gtk_label_new("Identity");
	vbox_identity = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox_identity), 5);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_identity, label);

	hbox_tmp = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox_identity), hbox_tmp, FALSE, FALSE, 0);

	label = gtk_label_new("Your Name:");
	gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

	entry = gtk_entry_new_with_max_length(79);
	gtk_entry_set_text(GTK_ENTRY(entry), prefs.from_name);

	gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(prefs_window), "from_name", entry);
	hbox_tmp = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox_identity), hbox_tmp, FALSE, FALSE, 0);

	label = gtk_label_new("From address:");
	gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

	entry = gtk_entry_new_with_max_length(79);
	gtk_entry_set_text(GTK_ENTRY(entry), prefs.from_address);

	gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(prefs_window), "from_address", entry);

	/*
	 * ******** Folder **********
	 */
	{
		GtkWidget *label = gtk_label_new("Folders");
		GtkWidget *vbox_folders = gtk_vbox_new(FALSE, 0);
		GtkWidget *hbox_tmp;
		GtkWidget *entry, *check_get_read;

		gtk_container_border_width(GTK_CONTAINER(vbox_folders), 5);
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_folders, label);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_folders), hbox_tmp, FALSE, FALSE, 0);

		button = gtk_button_new_with_label("Path to Inbox folder:");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), button, FALSE, FALSE, 0);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(file_select),
			GINT_TO_POINTER(SELECT_INBOX_FILENAME));

		entry = gtk_entry_new_with_max_length(FILENAME_MAX);
		inbox_filename_entry = entry;
		gtk_entry_set_text(GTK_ENTRY(entry),
			prefs.folder_path_inbox);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "folder_path_inbox",
			entry);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_folders), hbox_tmp, FALSE, FALSE, 0);
		
		button = gtk_button_new_with_label("Path to Sent Mail folder:");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), button, FALSE, FALSE, 0);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(file_select),
			GINT_TO_POINTER(SELECT_SENTBOX_FILENAME));

		entry = gtk_entry_new_with_max_length(FILENAME_MAX);
		sentbox_filename_entry = entry;
		gtk_entry_set_text(GTK_ENTRY(entry), prefs.folder_path_sentmail);

		gtk_box_pack_start(GTK_BOX(hbox_tmp), entry, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "folder_path_sentmail",
			entry);
			
		label = gtk_label_new("(Leave blank for none)");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		check_get_read = gtk_check_button_new_with_label("Get read mail");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_get_read),
			prefs.get_read);
		gtk_box_pack_start(GTK_BOX(vbox_folders), check_get_read, FALSE,
			FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "get_read",
			check_get_read);

		/*
		 * hbox_tmp = gtk_hbox_new (FALSE, 0);
		 * gtk_box_pack_start(GTK_BOX(vbox_folders), hbox_tmp, FALSE,
		 * FALSE, 0); 
		 */
	}
	/*
	 * ******** Sync Options **********
	 */
	{
		GtkWidget *menu;
		GtkWidget *hbox_tmp;
		gchar *str;

		label = gtk_label_new("Sync");
		vbox_sync = gtk_vbox_new(FALSE, 0);
		gtk_container_border_width(GTK_CONTAINER(vbox_sync), 5);
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_sync, label);

		vbox_tmp = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_sync), vbox_tmp, FALSE, FALSE, 0);

		check_presync_get = gtk_check_button_new_with_label
			("Before sync, get mail from folder");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check_presync_get),
			prefs.presync_get);
		gtk_box_pack_start(GTK_BOX(vbox_tmp), check_presync_get, FALSE,
			FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "presync_get",
			check_presync_get);

		check_postsync_send = gtk_check_button_new_with_label
			("After sync, send mail from Outbox");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
			(check_postsync_send), prefs.postsync_send);
		gtk_box_pack_start(GTK_BOX(vbox_tmp), check_postsync_send, FALSE,
			FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "postsync_send",
			check_postsync_send);

		hbox_tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox_sync), hbox_tmp, FALSE, FALSE, 0);

		label = gtk_label_new("Truncate to size (bytes):");
		gtk_box_pack_start(GTK_BOX(hbox_tmp), label, FALSE, FALSE, 0);

		str = g_strdup_printf("%d", mailSyncPref.truncate);
		menu = make_pref_menu(truncate_items, str);
		g_free(str);

		gtk_box_pack_end(GTK_BOX(hbox_tmp), menu, FALSE, FALSE, 0);
		gtk_object_set_data(GTK_OBJECT(prefs_window), "sync_truncate", menu);
	}

	/*
	 * ******** Signature **********
	 */

	label = gtk_label_new("Signature");
	vbox_sig = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox_sig), 5);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_sig, label);

	sig_text = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(sig_text), TRUE);
	gtk_text_set_word_wrap(GTK_TEXT(sig_text), FALSE);

	if (mailSigPref.signature)
	{
		gtk_text_insert(GTK_TEXT(sig_text), NULL, NULL, NULL,
			mailSigPref.signature, -1);
	}

	gtk_box_pack_start(GTK_BOX(vbox_sig), sig_text, FALSE, FALSE, 0);
	gtk_object_set_data(GTK_OBJECT(prefs_window), "signature", sig_text);

	/*
	 * Separator 
	 */
	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 5);
	/*
	 * gtk_widget_show(separator);
	 */

	/*
	 * Create a "Quit" button 
	 */
	hbox_tmp = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox_tmp, FALSE, FALSE, 0);

	button = gtk_button_new_with_label("OK");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(cb_prefs_ok), prefs_window);
	gtk_box_pack_start(GTK_BOX(hbox_tmp), button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	button = gtk_button_new_with_label("Cancel");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(cb_prefs_cancel), prefs_window);
	gtk_box_pack_end(GTK_BOX(hbox_tmp), button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

	gtk_widget_show_all(prefs_window);
}

void cb_new_mail_mode(GtkWidget * widget, gpointer data)
{
	/*
	 * clear_address_entries(); 
	 */
	/*
	 * gtk_widget_hide(vbox2_1); gtk_widget_show(vbox2_2); 
	 */
}

/*
 * This function just adds the record to the clist on the left side of
 * the screen.
 */
int display_record(GtkWidget * mail_clist, struct MyMail *pMyMail)
{
	gchar *empty_line[] = { "", "", "" };
	char date[12];
	GdkColor color;
	GdkColormap *colormap;
	unsigned char cat = pMyMail->attrib & 0x0f;

	jp_logf(JP_LOG_DEBUG, "display_record\n");

	gtk_clist_prepend(GTK_CLIST(mail_clist), empty_line);

	if (pMyMail->rt == NEW_PC_REC)
	{
		colormap = gtk_widget_get_colormap(mail_clist);
		color.red = CLIST_NEW_RED;
		color.green = CLIST_NEW_GREEN;
		color.blue = CLIST_NEW_BLUE;
		gdk_color_alloc(colormap, &color);
		gtk_clist_set_background(GTK_CLIST(mail_clist), 0, &color);
	}

	gtk_clist_set_row_data(GTK_CLIST(mail_clist), 0, pMyMail);

	sprintf(date, "%2d/%2d", pMyMail->mail.date.tm_mon + 1,
		pMyMail->mail.date.tm_mday);

	gtk_clist_set_text(GTK_CLIST(mail_clist), 0, 0, date);

	if ((cat == ATTRIB_OUTBOX) || (cat == ATTRIB_DRAFT)
		|| (cat == ATTRIB_FILED))
		gtk_clist_set_text(GTK_CLIST(mail_clist), 0, 1,
			pMyMail->mail.to ? pMyMail->mail.to : "(no To:)");
	else
		gtk_clist_set_text(GTK_CLIST(mail_clist), 0, 1,
			pMyMail->mail.from ? pMyMail->mail.from : "(no From:)");

	gtk_clist_set_text(GTK_CLIST(mail_clist), 0, 2, pMyMail->mail.subject);

	return 0;
}

/*
 * This function lists the records in the clist on the left side of
 * the screen.
 */
void display_records(void)
{
	GtkWidget *mail_clist;
	int num, i, num_shown;
	struct MyMail *pMyMail;
	GList *records;
	GList *temp_list;
	buf_rec *br;

	jp_logf(JP_LOG_DEBUG, "display_records: show_category = %d\n",
		show_category);

	records = NULL;

	jp_logf(JP_LOG_DEBUG, "Mail: display_records\n");

	if (glob_mymail_list != NULL)
		free_mymail_list(&glob_mymail_list);

	mail_clist =
		gtk_object_get_data(GTK_OBJECT(mail_main_hbox), "mail_clist");
	gtk_clist_freeze(GTK_CLIST(mail_clist));
	gtk_clist_clear(GTK_CLIST(mail_clist));

	/*
	 * This function takes care of reading the Database for us 
	 */
	num = jp_read_DB_files("MailDB", &records);

	/*
	 * Go to first entry in the list 
	 */
	for (temp_list = records; temp_list; temp_list = temp_list->prev)
		records = temp_list;

	num_shown = 0;
	for (i = 0, temp_list = records; temp_list;
		temp_list = temp_list->next, i++)
	{
		int size;

		if (temp_list->data)
			br = temp_list->data;
		else
			continue;

		if (!br->buf)
			continue;

		/*
		 * Since deleted and modified records are also returned and we
		 * don't want to see those we skip over them. 
		 */
		if ((br->rt == DELETED_PALM_REC) || (br->rt == MODIFIED_PALM_REC))
			continue;

		if ((br->attrib & 0x0F) != show_category)
			continue;

		pMyMail = malloc(sizeof(struct MyMail));
		pMyMail->next = NULL;
		pMyMail->attrib = br->attrib;
		pMyMail->unique_id = br->unique_id;
		pMyMail->rt = br->rt;

		/*
		 * We need to unpack the record blobs from the database.
		 * unpack_Mail is already written in pilot-link, but normally an
		 * unpack must be written for each type of application 
		 */
		if ((size = unpack_Mail(&(pMyMail->mail), br->buf, br->size)) != 0)
		{
			pMyMail->rec_size = br->size;
			display_record(mail_clist, pMyMail);
		}
		//jp_logf(JP_LOG_DEBUG, "size = %d, br->size = %d\n", size, br->size);

		if (glob_mymail_list == NULL)
			glob_mymail_list = pMyMail;
		else
			glob_mymail_list->next = pMyMail;

		num_shown++;
	}

	if (num_shown)
		gtk_clist_select_row(GTK_CLIST(mail_clist), 0, 0);

	gtk_clist_thaw(GTK_CLIST(mail_clist));

	jp_free_DB_records(&records);
}

void jp_pc_mail_write(struct Mail *pMail, unsigned char attrib)
{
	gint size = pack_Mail(pMail, NULL, 0);

	if (size > 0)
	{
		char *buf = g_malloc(size);

		if (buf)
		{
			buf_rec br;

			pack_Mail(pMail, buf, size);

			br.rt = NEW_PC_REC;

			br.attrib = attrib;
			br.buf = buf;
			br.size = size;

			jp_pc_write("MailDB", &br);

			g_free(buf);
		}
	}
}

void mail_delete(struct MyMail *pMail, int flag)
{
	int size;

	size = pack_Mail(&(pMail->mail), NULL, 0);
	if (size > 0)
	{
		buf_rec br;
		char *buf = g_malloc(size);

		if (buf)
		{
			pack_Mail(&(pMail->mail), buf, size);

			br.rt = pMail->rt;
			br.unique_id = pMail->unique_id;
			br.attrib = pMail->attrib;
			br.buf = buf;
			br.size = pMail->rec_size;

			jp_delete_record("MailDB", &br, flag);

			g_free(buf);
		}
	}
}

/*
 * This function gets called when the "delete" button is pressed 
 */
void cb_delete(GtkWidget * widget, int data)
{
	GtkWidget *mail_clist;
	struct MyMail *pMail;

	mail_clist = gtk_object_get_data(GTK_OBJECT(mail_main_hbox), "mail_clist");
	pMail = gtk_clist_get_row_data(GTK_CLIST(mail_clist), glob_row_selected);
	if (pMail == NULL)
		return;

	/* delete the J-Pilot record */
	mail_delete(pMail, DELETE_FLAG);

	/* remove the GTK entry */
	gtk_clist_remove(GTK_CLIST(mail_clist), glob_row_selected);

	/* if it was the last entry select the previous one */
	pMail = gtk_clist_get_row_data(GTK_CLIST(mail_clist), glob_row_selected);
	if (pMail == NULL)
	{
		glob_row_selected--;
		gtk_clist_select_row(GTK_CLIST(mail_clist), glob_row_selected, 0);
	}
}

static void set_mail_clist_titles(int cat)
{
	GtkWidget *mail_clist =
		gtk_object_get_data(GTK_OBJECT(mail_main_hbox), "mail_clist");

	gtk_clist_set_column_title(GTK_CLIST(mail_clist), 0, "Date");
	if ((cat == ATTRIB_OUTBOX) || (cat == ATTRIB_DRAFT)
		|| (cat == ATTRIB_FILED))
	{
		gtk_clist_set_column_title(GTK_CLIST(mail_clist), 1, "To");
	}
	else
	{
		gtk_clist_set_column_title(GTK_CLIST(mail_clist), 1, "From");
	}
	gtk_clist_set_column_title(GTK_CLIST(mail_clist), 2, "Subject");
}

/*
 * All menus use this same callback function.  I use the value parameter
 * to determine which menu was changed and which item was selected from it.
 */
void cb_menu(GtkWidget * item, unsigned int index)
{
#ifdef _CAN_EDIT_MAIL
	GtkWidget *edit_button;
#endif
	jp_logf(JP_LOG_DEBUG, "cb_menu: show_category = %d\n",
		show_category);

	if (!item)
	{
		jp_logf(JP_LOG_DEBUG, "cb_menu: no item!\n");
		return;
	}
	if (!(GTK_CHECK_MENU_ITEM(item))->active)
	{
		jp_logf(JP_LOG_DEBUG, "cb_menu: not active!\n");
		return;
	}

	show_category = index;

	jp_logf(JP_LOG_DEBUG, "show_category = %d\n", show_category);

	set_mail_clist_titles(show_category);
	display_records();
#ifdef _CAN_EDIT_MAIL
	edit_button =
		gtk_object_get_data(GTK_OBJECT(mail_main_hbox), "edit_button");
	gtk_widget_set_sensitive(edit_button, (show_category == ATTRIB_OUTBOX)
		|| (show_category == ATTRIB_DRAFT));
#endif
}

/*
 * Just a convenience function for passing in an array of strings and getting
 * them all stuffed into a menu.
 */
GtkWidget *make_menu(char *items[], GtkWidget * menu_items[])
{
	int i;
	GSList *group;
	GtkWidget *option_menu;
	GtkWidget *menu;

	option_menu = gtk_option_menu_new();

	menu = gtk_menu_new();

	group = NULL;

	for (i = 0; items[i]; i++)
	{
		GtkWidget *menu_item =
			gtk_radio_menu_item_new_with_label(group, items[i]);

		menu_items[i] = menu_item;

		gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			cb_menu, GINT_TO_POINTER(i));
		group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
		gtk_menu_append(GTK_MENU(menu), menu_item);
		gtk_widget_show(menu_item);
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	/*
	 * Make this one show up by default 
	 */
	gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), 0);

	gtk_widget_show(option_menu);

	return option_menu;
}

static GtkWidget *make_menus()
{
	struct MailAppInfo mai;
	unsigned char *buf;
	int buf_size;
	int i;
	GtkWidget *menu_category;
	GtkWidget *menu_item_category1[16];
	char *categories[17];

	categories[0] = NULL;

	if (jp_get_app_info("MailDB", &buf, &buf_size) >= 0)
	{

		unpack_MailAppInfo(&mai, buf, buf_size);

		for (i = 0; i < 16; i++)
		{
			if (mai.category.name[i][0] == '\0')
			{
				break;
			}
			categories[i] = mai.category.name[i];
		}
		categories[i] = NULL;

		free(buf);
	}
	else
		jp_logf(JP_LOG_WARN,
			"\nCould not read from MailDB.\nMaybe you should sync first?\n");

	menu_category = make_menu(categories, menu_item_category1);

	return menu_category;
}

static void cb_clist_selection(GtkWidget * mail_clist,
	gint row, gint column, GdkEventButton * event, gpointer data)
{
	GtkWidget *mail_text;
	struct MyMail *pMail;
	gchar *tmp_text;

	if (row < 0)
		return;

	mail_text = gtk_object_get_data(GTK_OBJECT(mail_main_hbox), "mail_text");

	if (!mail_text)
		return;

	glob_row_selected = row;

	if ((pMail =
			(struct MyMail *) gtk_clist_get_row_data(GTK_CLIST(mail_clist),
				row)) == NULL)
		return;

	gtk_text_freeze(GTK_TEXT(mail_text));

	gtk_text_set_point(GTK_TEXT(mail_text), 0);
	gtk_text_forward_delete(GTK_TEXT(mail_text),
		gtk_text_get_length(GTK_TEXT(mail_text)));

	/*
	 * tmp_text = g_strdup_printf("read = %d\nsignature = %d\n"
	 * "confirmRead = %d\nconfirmDelivery = %d\n" "priority =
	 * %d\naddressing = %d\n", pMail->mail.read, pMail->mail.signature,
	 * pMail->mail.confirmRead, pMail->mail.confirmDelivery,
	 * pMail->mail.priority, pMail->mail.addressing);
	 * gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, tmp_text, -1);
	 * g_free(tmp_text); 
	 */

	if (pMail->mail.from != NULL)
	{
		tmp_text = g_strdup_printf("From: %s\r\n", pMail->mail.from);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.to != NULL)
	{
		tmp_text = g_strdup_printf("To: %s\r\n", pMail->mail.to);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.cc != NULL)
	{
		tmp_text = g_strdup_printf("Cc: %s\r\n", pMail->mail.cc);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.bcc != NULL)
	{
		tmp_text = g_strdup_printf("Bcc: %s\r\n", pMail->mail.bcc);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.replyTo != NULL)
	{
		tmp_text = g_strdup_printf("Reply To: %s\r\n", pMail->mail.replyTo);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.sentTo != NULL)
	{
		tmp_text = g_strdup_printf("Sent To: %s\r\n", pMail->mail.sentTo);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.subject != NULL)
	{
		tmp_text = g_strdup_printf("Subject: %s\r\n", pMail->mail.subject);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}
	if (pMail->mail.body != NULL)
	{
		tmp_text = g_strdup_printf("\r\n%s\r\n", pMail->mail.body);
		gtk_text_insert(GTK_TEXT(mail_text), NULL, NULL, NULL, tmp_text, -1);
		g_free(tmp_text);
	}

	gtk_text_thaw(GTK_TEXT(mail_text));
}

/*
 * This function is called by J-Pilot when the user selects this plugin
 * from the plugin menu, or from the search window when a search result
 * record is chosen.  In the latter case, unique ID will be set.  This
 * application should go directly to that record in the case.
 */
int plugin_gui(GtkWidget * vbox, GtkWidget * hbox, unsigned int unique_id)
{
	/*
	 * much to do 
	 */
	GtkWidget *mail_list_window;
	GtkWidget *mail_clist;
	GtkWidget *vbox1;
	GtkWidget *hbox_temp;
	GtkWidget *vbox2_1;
	GtkWidget *vbox2_1_sub1;
	GtkWidget *temp_hbox;
	GtkWidget *vscrollbar;
	GtkWidget *separator;
	GtkWidget *label;
	GtkWidget *frame;
	GtkWidget *mail_text;
	GtkWidget *menu_category;
	GtkWidget *vpaned;
	time_t ltime;
	struct tm *now;
	char *titles[] = { "Date", "From", "Subject" };
	static GtkTooltips *glob_tooltips = NULL;
	static GtkAccelGroup *accel_group = NULL;

	jp_logf(JP_LOG_DEBUG, "plugin_gui()\n");

	mail_main_hbox = hbox;

	if (accel_group == NULL)
	{
		accel_group = gtk_accel_group_new();
		gtk_window_add_accel_group(
			GTK_WINDOW(gtk_widget_get_toplevel(vbox)), accel_group);
	}

	if (glob_tooltips == NULL)
		glob_tooltips = gtk_tooltips_new();

	/*
	 * File selector
	 */
	if (FileSelector == NULL)
	{
		FileSelector = gtk_file_selection_new ("File selection");

		/* Connect the ok_button to file_ok_sel function */
		gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (FileSelector)->ok_button),
			"clicked", (GtkSignalFunc) file_ok_sel, FileSelector );

		gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (FileSelector)->cancel_button),
			"clicked", (GtkSignalFunc) file_cancel, 0);

		/* Set as modal */
		gtk_window_set_modal (GTK_WINDOW(FileSelector),TRUE);
	}

	/*
	 * the buttons on the left side 
	 */
	{
		GtkWidget *button;
		GtkWidget *separator;

		button = gtk_button_new_with_label("Get");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_mail_get), NULL);
		gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
		gtk_tooltips_set_tip(glob_tooltips, button, "Get mails from mbox to jpilot   Ctrl+G", NULL);
		gtk_widget_add_accelerator(button, "clicked", accel_group, GDK_g, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
		gtk_widget_show(button);

		button = gtk_button_new_with_label("Send");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_mail_send), NULL);
		gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
		gtk_tooltips_set_tip(glob_tooltips, button, "Send mails from jpilot   Ctrl+S", NULL);
		gtk_widget_add_accelerator(button, "clicked", accel_group, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
		gtk_widget_show(button);

		separator = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, TRUE, 5);
		gtk_widget_show(separator);

		button = gtk_button_new_with_label("Prefs");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_prefs), NULL);
		gtk_box_pack_start(GTK_BOX(vbox), button, TRUE, TRUE, 0);
		gtk_tooltips_set_tip(glob_tooltips, button, "Set jpilot-Mail preferences   Ctrl+P", NULL);
		gtk_widget_add_accelerator(button, "clicked", accel_group, GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
		gtk_widget_show(button);
	}

	/*
	 * the 'area' 
	 */

	show_category = 0;
	glob_row_selected = 0;

	time(&ltime);
	now = localtime(&ltime);

	/*
	 * Make the menus 
	 */
	menu_category = make_menus();

	/*
	 * left and right main boxes 
	 */
	vpaned = gtk_vpaned_new();
	vbox1 = gtk_vbox_new(FALSE, 0);
	vbox2_1 = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(hbox), vpaned);
	gtk_paned_pack1 (GTK_PANED (vpaned), vbox1, TRUE, FALSE);
	gtk_paned_pack2 (GTK_PANED (vpaned), vbox2_1, FALSE, FALSE);

	gtk_widget_show(vpaned);

	/*
	 * Make a temporary hbox 
	 */
	temp_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), temp_hbox, FALSE, FALSE, 0);

	label = gtk_label_new("Category: ");
	gtk_box_pack_start(GTK_BOX(temp_hbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(temp_hbox), menu_category, TRUE, TRUE, 0);

	/*
	 * Scrolled Window 
	 */
	mail_list_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_container_set_border_width(GTK_CONTAINER(mail_list_window), 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mail_list_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox1), mail_list_window, TRUE, TRUE, 0);

	gtk_widget_set_usize(GTK_WIDGET(vbox1), 230, 0);

	/*
	 * Clist 
	 */
	mail_clist = gtk_clist_new_with_titles(3, titles);

	gtk_object_set_data(GTK_OBJECT(mail_main_hbox), "mail_clist", mail_clist);

	set_mail_clist_titles(show_category);

	gtk_signal_connect(GTK_OBJECT(mail_clist), "select_row",
		GTK_SIGNAL_FUNC(cb_clist_selection), NULL);
	/*
	 * gtk_clist_set_shadow_type(GTK_CLIST(mail_clist), SHADOW);
	 */
	gtk_clist_set_selection_mode(GTK_CLIST(mail_clist),
		GTK_SELECTION_BROWSE);
	gtk_clist_set_column_width(GTK_CLIST(mail_clist), 0, 50);
	gtk_clist_set_column_width(GTK_CLIST(mail_clist), 1, 100);
	gtk_clist_set_column_width(GTK_CLIST(mail_clist), 2, 110);
	gtk_container_add(GTK_CONTAINER(mail_list_window),
		GTK_WIDGET(mail_clist));

	/*
	 * The Frame on the right side 
	 */
	frame = gtk_frame_new("Quick View");
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.5, 0.0);
	gtk_box_pack_start(GTK_BOX(vbox2_1), frame, TRUE, TRUE, 0);
	vbox2_1_sub1 = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox2_1_sub1), 5);
	gtk_container_add(GTK_CONTAINER(frame), vbox2_1_sub1);
	gtk_widget_show(frame);
	gtk_widget_show_all(vbox2_1_sub1);

	/*
	 * The Right hand side of the screen 
	 */

	/*
	 * Buttons on top 
	 */
	{
		GtkWidget *button;
		GtkWidget *hbox_temp = gtk_hbox_new(FALSE, 0);

		gtk_box_pack_start(GTK_BOX(vbox2_1_sub1), hbox_temp, FALSE, FALSE, 0);
		/*
		 * gtk_container_add(GTK_CONTAINER(vbox2_1_sub1), hbox_temp);
		 */

#ifdef _CAN_EDIT_MAIL

		button = gtk_button_new_with_label("New");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_new_mail_mode), NULL);
		gtk_box_pack_start(GTK_BOX(hbox_temp), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

		button = gtk_button_new_with_label("Edit");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_mail_edit), NULL);
		gtk_box_pack_start(GTK_BOX(hbox_temp), button, TRUE, TRUE, 0);
		gtk_widget_show(button);
		gtk_widget_set_sensitive(button,
			(show_category == ATTRIB_OUTBOX)
			|| (show_category == ATTRIB_DRAFT));
		gtk_object_set_data(GTK_OBJECT(mail_main_hbox), "edit_button",
			button);

		button = gtk_button_new_with_label("Reply");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_not_yet), NULL);
		gtk_box_pack_start(GTK_BOX(hbox_temp), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

#endif /*_CAN_EDIT_MAIL */

#ifdef _CAN_FORWARD_MAIL
		button = gtk_button_new_with_label("Forward");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_not_yet), NULL);
		gtk_box_pack_start(GTK_BOX(hbox_temp), button, TRUE, TRUE, 0);
		gtk_widget_show(button);

#endif	/* _CAN_FORWARD_MAIL */

		button = gtk_button_new_with_label("Delete");
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(cb_delete), NULL);
		gtk_box_pack_start(GTK_BOX(hbox_temp), button, TRUE, TRUE, 0);
		gtk_tooltips_set_tip(glob_tooltips, button, "Delete a mail   Ctrl+D",
			NULL);
		gtk_widget_add_accelerator(button, "clicked", accel_group, GDK_d,
			GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE);
		gtk_widget_show(button);

		gtk_widget_show_all(hbox_temp);
	}

	/*
	 * Separator 
	 */
	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox2_1_sub1), separator, FALSE, FALSE, 5);
	gtk_widget_show(separator);

	/*
	 * The text box on the right side 
	 */
	hbox_temp = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2_1_sub1), hbox_temp, TRUE, TRUE, 0);

	mail_text = gtk_text_new(NULL, NULL);

	gtk_object_set_data(GTK_OBJECT(mail_main_hbox), "mail_text",
		mail_text);

	gtk_text_set_editable(GTK_TEXT(mail_text), FALSE);
	gtk_text_set_word_wrap(GTK_TEXT(mail_text), TRUE);
	vscrollbar = gtk_vscrollbar_new(GTK_TEXT(mail_text)->vadj);
	gtk_box_pack_start(GTK_BOX(hbox_temp), mail_text, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox_temp), vscrollbar, FALSE, FALSE, 0);
	/*
	 * gtk_widget_set_usize (GTK_WIDGET(text), 100, 50); 
	 */
	gtk_widget_show(mail_text);
	gtk_widget_show(vscrollbar);
	gtk_widget_show(hbox_temp);

	gtk_widget_show_all(vbox2_1);

	gtk_widget_show_all(hbox);
	gtk_widget_show_all(vbox);

	display_records();

	/*
	 * if (unique_id) { expense_find(unique_id); } 
	 */
	return 0;
}

/*
 * This function is called by J-Pilot when the user selects this plugin
 * from the plugin menu, or from the search window when a search result
 * record is chosen.  In the latter case, unique ID will be set.  This
 * application should go directly to that record in the case.
 */
int plugin_gui_cleanup()
{
	/*
	 * clean up 
	 */
	return 0;
}

/*
 * This is a plugin callback function that is executed when J-Pilot starts up.
 * base_dir is where J-Pilot is compiled to be installed at (e.g. /usr/local/)
 */
int plugin_startup(jp_startup_info * info)
{
	struct passwd *ent;

	jp_init();

	ent = getuserent(getuid());
	if (ent == NULL)
		jp_logf(JP_LOG_WARN, "Mail::plugin_startup: ent == NULL\n");

	/*
	 * set prefs to some sane values: 
	 */
	mailSyncPref.syncType = 0;
	mailSyncPref.getHigh = 0;
	mailSyncPref.getContaining = 0;
	mailSyncPref.truncate = 4000;
	mailSyncPref.filterTo = NULL;
	mailSyncPref.filterFrom = NULL;
	mailSyncPref.filterSubject = NULL;

	mailSigPref.signature = NULL;

	prefs.smtp_server = g_strdup("localhost");
	prefs.smtp_port = 25;
	prefs.from_name = g_strdup(ent->pw_gecos);

	prefs.from_address = g_strdup("");
	if (getenv("MAIL"))
		prefs.folder_path_inbox = g_strdup_printf(getenv("MAIL"));
	else
		prefs.folder_path_inbox = g_strdup_printf("%s/nsmail/Inbox", ent->pw_dir);

	prefs.folder_path_sentmail = g_strdup_printf("");

	prefs.default_domain = g_malloc(64);
	gethostname(prefs.default_domain, 64);

	prefs.pilot_charset = g_strdup("iso-8859-1");
	prefs.header_encoding = ENC_QP;

	prefs.get_read = TRUE;
	prefs.presync_get = FALSE;
	prefs.postsync_send = FALSE;

	prefs_read();
	prefs_read_pilot();

	return 0;
}

/*
 * This is a plugin callback function that is executed before a sync occurs.
 * Any sync preperation can be done here.
 */
int plugin_pre_sync(void)
{
	if (prefs.presync_get)
		cb_mail_get(NULL, NULL);

	return 0;
}

/*
 * This is a plugin callback function that is executed during a sync.
 * Notice that I don't need to sync the Mail application.  Since I used
 * the plugin_get_db_name call to tell J-Pilot what to sync for me.  It will
 * be done automatically.
 */
int plugin_sync(int sd)
{
	return 0;
}

/*
 * This function is called when the user does a search.  It should return
 * records which match the search string.
 */
int plugin_search(const char *search_string, int case_sense,
	struct search_result **sr)
{
	int count = 0;

	return count;
}

int plugin_help(char **text, int *width, int *height)
{
	/*
	 * We could also pass back *text=NULL and implement whatever we wanted 
	 * to here. 
	 */
	*text = strdup(
		 /*-------------------------------------------*/
		"Mail plugin for J-Pilot was written by\n"
		"Oliver Kurth (c) 1999-2000.\n"
		"Ludovic Rousseau (c) 2002-2003.\n"
		"Report bugs to: ludovic.rousseau@free.fr\n"
		"http://ludovic.rousseau.free.fr/softwares/jpilot-Mail/index.html\n");
	*height = 0;
	*width = 0;

	return 0;
}

/*
 * This is a plugin callback function called after a sync.
 */
int plugin_post_sync(void)
{
	jp_logf(JP_LOG_DEBUG, "Mail: plugin_post_sync\n");
	prefs_read_pilot();

	if (prefs.postsync_send)
		cb_mail_send(NULL, NULL);

	return 0;
}

/*
 * This is a plugin callback function called during program exit.
 */
int plugin_exit_cleanup(void)
{
	g_free(prefs.smtp_server);
	g_free(prefs.from_address);
	g_free(prefs.from_name);
	g_free(prefs.folder_path_inbox);
	g_free(prefs.default_domain);
	g_free(prefs.pilot_charset);

	/*
	 * does this segfault ? jp_logf(JP_LOG_DEBUG, "Mail:
	 * plugin_exit_cleanup\n"); 
	 */
	return 0;
}

static void file_select(GtkWidget *widget, gpointer data)
{
	/* Use the current file as default filename */
	Selection = GPOINTER_TO_INT(data);
	switch (Selection)
	{
		case SELECT_INBOX_FILENAME:
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(FileSelector),
				gtk_entry_get_text(GTK_ENTRY(inbox_filename_entry)));
			break;

		case SELECT_SENTBOX_FILENAME:
			gtk_file_selection_set_filename(GTK_FILE_SELECTION(FileSelector),
				gtk_entry_get_text(GTK_ENTRY(sentbox_filename_entry)));
			break;

		default:
			return;
	}

	gtk_widget_show(FileSelector);
} /* file_select */

static void file_cancel(void)
{
	gtk_widget_hide(FileSelector);
} /* file_cancel */

/* Get the selected filename and print it to the console */
static void file_ok_sel(GtkWidget *w, GtkFileSelection *fs)
{
	switch (Selection)
	{
		case SELECT_INBOX_FILENAME:
			gtk_entry_set_text(GTK_ENTRY(inbox_filename_entry),
				gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
			break;

		case SELECT_SENTBOX_FILENAME:
			gtk_entry_set_text(GTK_ENTRY(sentbox_filename_entry),
				gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)));
			break;
	}

	gtk_widget_hide(FileSelector);
} /* file_ok_sel */

