/*
 * jpilot-Mail Copyright (C) 2000 Oliver Kurth
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

/*
 * $Id 
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <locale.h>

#include <gtk/gtk.h>
#include "libplugin.h"
#include <pi-mail.h>

#include "masqmail.h"
#include "smtp_out.h"
#include "mail.h"
#include "accept.h"

masqmail_conf conf;

#define LOG_BUFFER_SIZE 128

void logwrite(int pri, const char *fmt, ...)
{
	va_list args;
	char log_buffer[LOG_BUFFER_SIZE];
	int level;

	va_start(args, fmt);

	vsnprintf(log_buffer, LOG_BUFFER_SIZE, fmt, args);

	/* convert from syslog levels to jpilot levels */
	switch (pri)
	{
		case LOG_NOTICE:
			level = JP_LOG_INFO;
			break;

		case LOG_WARNING:
			level = JP_LOG_WARN;
			break;

		case LOG_ALERT:
			level = JP_LOG_FATAL;
			break;

		default:
			level = JP_LOG_FATAL;
	}
	jp_logf(pri, "%s", log_buffer);

	va_end(args);
}

void debugf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);

	vfprintf(stderr, fmt, args);
	fflush(stderr);

	/*
	 * jp_logf(LOG_DEBUG, fmt, args);
	 */

	va_end(args);
}


static void dump_header_to_file(FILE *fptr, GList *hdr_list)
{
 	GList *node;
  
  	if(hdr_list)
 	{
 		foreach(hdr_list, node)
		{
			if(node->data)
			{
				header *hdr = (header *)(node->data);
				if(hdr->header)
				{
	  				fprintf(fptr, "%s", hdr->header);
				}
			}
		}
  }

  /* empty line separating headers from data: */
  fprintf(fptr, "\n");  
    
}

static void dump_data_to_file(FILE *fptr, message *msg)
{
  GList *node;

  /* data */
  if(msg->data_list){
    for(node = g_list_first(msg->data_list); node; node = g_list_next(node)){
      if(node->data){
	fprintf(fptr, "%s", (char *)node->data);
      }
    }
  }

  fprintf(fptr, "\n");
}






static gboolean mail_send(struct MyMail *pMail, int i)
{
	message *msg = create_message();
	gboolean failed = FALSE;

	/*
	 * initialize msg structure 
	 */
	GList *resolve_list = g_list_append(NULL, resolve_byname);

	msg->transfer_id = i;

	if (msg == NULL)
		_exit(EXIT_FAILURE);
	msg->received_host = NULL;
	msg->return_path = NULL;
	if (prefs.from_address)
		if (strlen(prefs.from_address))
			msg->return_path = create_address(prefs.from_address, TRUE);
	msg->received_prot = PROT_LOCAL;
	msg->rcpt_list = NULL;

	/*
	 * headers 
	 */
	if (prefs.from_name && prefs.from_address)
	{
		if (strlen(prefs.from_name) && strlen(prefs.from_address))
		{
			const size_t header_len = strlen("From: ");
			size_t ilen = strlen(prefs.from_name);
			size_t olen = enclen_rfc2047(ilen, strlen(prefs.pilot_charset),
					prefs.header_encoding, header_len) + 1;
			char *enc_from_name = (char *)g_malloc(olen);

			encode_rfc2047(enc_from_name, olen, prefs.from_name, ilen,
					prefs.pilot_charset, prefs.header_encoding, header_len);
			msg->hdr_list =
				g_list_append(msg->hdr_list,
				create_header(HEAD_FROM,
					"From: %s <%s>\n", enc_from_name,
					prefs.from_address));
			g_free(enc_from_name);
		}
	}

	if (pMail->mail.to)
		msg->hdr_list = g_list_append(msg->hdr_list,
			create_header(HEAD_TO, "To: %s\n", pMail->mail.to));
	if (pMail->mail.cc)
		msg->hdr_list = g_list_append(msg->hdr_list,
			create_header(HEAD_CC, "Cc: %s\n", pMail->mail.cc));
	if (pMail->mail.bcc)
		msg->hdr_list = g_list_append(msg->hdr_list,
			create_header(HEAD_BCC, "Bcc: %s\n", pMail->mail.bcc));
	if (pMail->mail.replyTo)
		msg->hdr_list =
			g_list_append(msg->hdr_list,
			create_header(HEAD_REPLY_TO, "Reply-To: %s\n",
				pMail->mail.replyTo));
	if (pMail->mail.subject)
	{
		const size_t header_len = strlen("Subject: ");
		char *subject = pMail->mail.subject;
		size_t ilen = strlen(subject);
		size_t olen = enclen_rfc2047(ilen, strlen(prefs.pilot_charset),
				prefs.header_encoding, header_len) + 1;
		char *enc_subject = (char *)g_malloc(olen);

		encode_rfc2047(enc_subject, olen, subject, ilen,
				prefs.pilot_charset, prefs.header_encoding, header_len);
		msg->hdr_list =
			g_list_append(msg->hdr_list,
			create_header(HEAD_SUBJECT, "Subject: %s\n",
				enc_subject));
		g_free(enc_subject);
	}

	msg->hdr_list =
		g_list_append(msg->hdr_list,
		create_header(HEAD_UNKNOWN, "X-Mailer: %s %s\n", PACKAGE,
			VERSION));

	msg->hdr_list =
		g_list_append(msg->hdr_list,
		create_header(HEAD_UNKNOWN, "MIME-Version: 1.0\n"));

	/*
	 * check for necessary MIME type: 
	 */
	{
		gboolean is_ascii = TRUE;

		if (pMail->mail.body != NULL)
		{
			guchar *p = pMail->mail.body;

			while (*p)
			{
				if (*p >= 127)
				{
					is_ascii = FALSE;
					break;
				}
				p++;
			}
		}
		if (is_ascii)
		{
			msg->hdr_list =
				g_list_append(msg->hdr_list,
				create_header(HEAD_UNKNOWN,
					"Content-Type: text/plain; charset=us-ascii\n"));
			msg->hdr_list =
				g_list_append(msg->hdr_list,
				create_header(HEAD_UNKNOWN,
					"Content-Transfer-Encoding: 7bit\n"));
		}
		else
		{
			msg->hdr_list =
				g_list_append(msg->hdr_list,
				create_header(HEAD_UNKNOWN,
					"Content-Type: text/plain; charset=%s\n",
					prefs.pilot_charset));
			msg->hdr_list =
				g_list_append(msg->hdr_list,
				create_header(HEAD_UNKNOWN,
					"Content-Transfer-Encoding: 8bit\n"));
		}
	}

	/*
	 * body 
	 */
	if (pMail->mail.body != NULL)
	{
		msg->data_list =
			g_list_append(msg->data_list, g_strdup(pMail->mail.body));

		if (pMail->mail.body[strlen(pMail->mail.body) - 1] != '\n')
			msg->data_list = g_list_append(msg->data_list, g_strdup("\n"));
	}

	/*
	 * signature 
	 */
	if (mailSigPref.signature != NULL)
	{
		if (strlen(mailSigPref.signature))
		{
			msg->data_list =
				g_list_append(msg->data_list, g_strdup("\n-- \n"));
			msg->data_list =
				g_list_append(msg->data_list,
				g_strdup(mailSigPref.signature));
			if (mailSigPref.signature[strlen(mailSigPref.signature) - 1] !=
				'\n')
				msg->data_list =
					g_list_append(msg->data_list, g_strdup("\n"));
		}
	}

	{
		accept_error err;

		/*
		 * rest of mail preparation is done by accept_message_prepare 
		 */
		if ((err =
				accept_message_prepare(msg,
					ACC_DEL_BCC | ACC_RCPT_FROM_HEAD | ACC_NO_RECVD_HDR))
			== AERR_OK)
		{

			jp_logf(JP_LOG_INFO | JP_LOG_GUI,
				"delivering message %d\n", i + 1);
			if (!(smtp_deliver(prefs.smtp_server, prefs.smtp_port,
						resolve_list, msg, NULL, NULL) == smtp_ok))
				debugf("smtp_deliver failed.\n");
			jp_logf(JP_LOG_DEBUG, "after delivering\n");

			{
				GList *rcpt_node;

				foreach(msg->rcpt_list, rcpt_node)
				{
					address *rcpt = (address *) (rcpt_node->data);

					if (!addr_is_delivered(rcpt))
					{
						jp_logf(JP_LOG_WARN,
							"message could not be delivered to %s@%s\n",
							rcpt->local_part, rcpt->domain);
						failed = TRUE;
					}
					if (!failed)
					{
						jp_pc_mail_write(&(pMail->mail), ATTRIB_FILED);
						mail_delete(pMail, DELETE_FLAG);
					}
				}
			}
		}
	}
	
	// Add to sent mail folder
	if (strcmp(prefs.folder_path_sentmail, "")!=0)
	{
		FILE *fptr;
		
		fptr = fopen(prefs.folder_path_sentmail, "at");
		if (fptr == NULL)
		{
			jp_logf(JP_LOG_WARN,
				"Could not open Sent Mail folder %s\n",
				prefs.folder_path_sentmail);
		}
		else
		{
			// Flag new email in uxix mailbox
			fprintf(fptr, "From %s\n", prefs.from_address);
			// Write msg->hdr_list
			dump_header_to_file(fptr, msg->hdr_list);
			// Write msg->data_list
			dump_data_to_file(fptr, msg);
			fclose(fptr);
			logwrite(LOG_NOTICE, "Added sent message to %s\n", prefs.folder_path_sentmail);
		}
	}
	
	destroy_message(msg);

	return !failed;
}

gint mail_send_all()
{
	int i = 0;
	GList *rec_list = NULL, *rec_node;
	gint sent_cnt = 0;

	conf.host_name = g_strdup(prefs.default_domain);
	// conf.debug_level = 5;

	/*
	 * Make sure that the Date header will be conforming: 
	 */
	setlocale(LC_ALL, "C");

	jp_read_DB_files("MailDB", &rec_list);
	jp_logf(JP_LOG_INFO | JP_LOG_GUI, "starting message delivery\n");
	foreach(rec_list, rec_node)
	{
		buf_rec *pBufRec = (buf_rec *) (rec_node->data);

		if (!((pBufRec->rt == DELETED_PALM_REC)
				|| (pBufRec->rt == MODIFIED_PALM_REC)))
		{
			if ((pBufRec->attrib & 0x0f) == ATTRIB_OUTBOX)
			{
				struct MyMail *pMail = g_malloc(sizeof(struct MyMail));

				pMail->next = NULL;
				pMail->attrib = pBufRec->attrib;
				pMail->unique_id = pBufRec->unique_id;
				pMail->rt = pBufRec->rt;

				if (unpack_Mail(&(pMail->mail), pBufRec->buf,
						pBufRec->size) != 0)
				{
					pMail->rec_size = pBufRec->size;
					if (mail_send(pMail, i++))
						sent_cnt++;
				}
				g_free(pMail);
			}
		}
	}
	setlocale(LC_ALL, "");
	jp_logf(JP_LOG_INFO | JP_LOG_GUI, "finished message delivery\n");

	return sent_cnt;
}

void cb_mail_send(GtkWidget * widget, gpointer data)
{
	gint cnt;

	if ((cnt = mail_send_all()) > 0)
		jp_logf(JP_LOG_INFO | JP_LOG_GUI,
			"%d mail(s) sent. Do another sync to delete the mail(s) on the palm.\n",
			cnt);
}
