// Copyright 2012, 2013 Canonical Ltd.
// Licensed under the LGPLv3, see LICENSE file for details.

package cmd_test

import (
	"io/ioutil"
	"os"
	"path/filepath"

	gitjujutesting "github.com/juju/testing"
	"github.com/juju/utils"
	gc "gopkg.in/check.v1"
	"launchpad.net/gnuflag"

	"github.com/juju/cmd"
	"github.com/juju/cmd/cmdtesting"
)

type FileVarSuite struct {
	gitjujutesting.FakeHomeSuite
	ctx         *cmd.Context
	ValidPath   string
	InvalidPath string // invalid path refers to a file which is not readable
}

var _ = gc.Suite(&FileVarSuite{})

func (s *FileVarSuite) SetUpTest(c *gc.C) {
	s.FakeHomeSuite.SetUpTest(c)
	s.ctx = cmdtesting.Context(c)
	s.ValidPath = s.ctx.AbsPath("valid.yaml")
	s.InvalidPath = s.ctx.AbsPath("invalid.yaml")
	f, err := os.Create(s.ValidPath)
	c.Assert(err, gc.IsNil)
	f.Close()
	f, err = os.Create(s.InvalidPath)
	c.Assert(err, gc.IsNil)
	f.Close()
	err = os.Chmod(s.InvalidPath, 0) // make unreadable
	c.Assert(err, gc.IsNil)
}

func (s *FileVarSuite) TestTildeFileVar(c *gc.C) {
	path := filepath.Join(utils.Home(), "config.yaml")
	err := ioutil.WriteFile(path, []byte("abc"), 0644)
	c.Assert(err, gc.IsNil)

	var config cmd.FileVar
	config.Set("~/config.yaml")
	file, err := config.Read(s.ctx)
	c.Assert(err, gc.IsNil)
	c.Assert(string(file), gc.Equals, "abc")
}

func (s *FileVarSuite) TestValidFileVar(c *gc.C) {
	fs, config := fs()
	err := fs.Parse(false, []string{"--config", s.ValidPath})
	c.Assert(err, gc.IsNil)
	c.Assert(config.Path, gc.Equals, s.ValidPath)
	_, err = config.Read(s.ctx)
	c.Assert(err, gc.IsNil)
}

func (s *FileVarSuite) TestInvalidFileVar(c *gc.C) {
	fs, config := fs()
	err := fs.Parse(false, []string{"--config", s.InvalidPath})
	c.Assert(config.Path, gc.Equals, s.InvalidPath)
	_, err = config.Read(s.ctx)
	c.Assert(err, gc.ErrorMatches, "*permission denied")
}

func fs() (*gnuflag.FlagSet, *cmd.FileVar) {
	var config cmd.FileVar
	fs := cmdtesting.NewFlagSet()
	fs.Var(&config, "config", "the config")
	return fs, &config
}
