from twisted.internet.defer import inlineCallbacks, returnValue

from txzookeeper.client import ConnectionTimeoutException

from juju.errors import EnvironmentPending, NoConnection
from juju.state.sshclient import SSHClient

from .utils import log


class ZookeeperConnect(object):

    def __init__(self, provider):
        self._provider = provider

    @inlineCallbacks
    def run(self, share=False):
        """Attempt to connect to a running zookeeper node.

        :param bool share: where feasible, attempt to share a connection with
            other clients

        :return: an open :class:`txzookeeper.client.ZookeeperClient`
        :rtype: :class:`twisted.internet.defer.Deferred`

        :raises: :exc:`juju.errors.EnvironmentNotFound` when no zookeepers
            exist
        :raises: :exc:`juju.errors.EnvironmentPending` when zookeepers
            exist but connection attempt fails
        """
        candidates = yield self._provider.get_zookeeper_machines()
        chosen = yield self._pick_machine(candidates)
        client = yield self._connect_to_machine(chosen, share)
        yield self.wait_for_initialization(client)
        returnValue(client)

    def _pick_machine(self, machines):
        # TODO Should we pick a random entry from the nodes list?
        for machine in machines:
            if machine.dns_name:
                return machine
        raise EnvironmentPending("No machines have addresses assigned yet")

    def _connect_to_machine(self, machine, share):
        log.info("Connecting to environment.")
        result = SSHClient().connect(
            machine.dns_name + ":2181", timeout=30, share=share)
        result.addErrback(self._cannot_connect, machine)
        return result

    def _cannot_connect(self, failure, machine):
        failure.trap(NoConnection, ConnectionTimeoutException)
        raise EnvironmentPending(
            "Cannot connect to machine %s (perhaps still initializing): %s"
            % (machine.instance_id, str(failure.value)))

    @inlineCallbacks
    def wait_for_initialization(self, client):
        exists_d, watch_d = client.exists_and_watch("/initialized")
        exists = yield exists_d
        if not exists:
            log.info("Environment still initializing. Will wait.")
            yield watch_d
        else:
            log.debug("Environment already initialized.")
