// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// extrude_faces_implementation

class extrude_faces_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >,
	public k3d::mouse_event_observer
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	extrude_faces_implementation(k3d::idocument& Document) :
		base(Document),
		mouse_event_observer("LMB Drag to extrude faces"),
		m_distance(k3d::init_name("distance") + k3d::init_description("Distance [number]") + k3d::init_value(1.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_cap(k3d::init_name("cap") + k3d::init_description("Cap [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_distance));
		enable_serialization(k3d::persistence::proxy(m_cap));
		
		register_property(m_distance);
		register_property(m_cap);
		
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &extrude_faces_implementation::on_reset_geometry));
		m_cap.changed_signal().connect(SigC::slot(*this, &extrude_faces_implementation::on_reshape_geometry));
		
		m_distance.changed_signal().connect(SigC::slot(*this, &extrude_faces_implementation::on_reshape_geometry));
		
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &extrude_faces_implementation::on_create_geometry));
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double sensitivity = 1.0;
		m_distance.set_value(m_distance.value() + sensitivity * (Last[1] - Current[1]));

		return true;
	}

	void on_reset_geometry()
	{
		m_new_points.clear();
		m_output_mesh.reset();
	}

	/// Returns true iff the given object is selected
	struct is_selected
	{
		bool operator()(k3d::face* const Face)
		{
			return Face->selected;
		}
	};

	/// Returns true iff the given face has neighbors
	struct has_neighbors
	{
		bool operator()(k3d::face* const Face)
		{
			for(k3d::split_edge* edge = Face->first_edge; edge; edge = edge->face_clockwise)
				{
					if(edge->companion)
						return true;

					if(edge->face_clockwise == Face->first_edge)
						break;
				}

			return false;
		}
	};

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		reshape_geometry();
	
		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		const bool cap = m_cap.property_value();

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				// Get the set of all selected faces ...
				k3d::polyhedron::faces_t selected_faces;
				k3d::copy_if((*polyhedron)->faces.begin(), (*polyhedron)->faces.end(), std::inserter(selected_faces, selected_faces.end()), is_selected());

				if(selected_faces.empty())
					selected_faces = (*polyhedron)->faces;
			
				// Remove "interior" faces (faces that have adjacent faces)
				selected_faces.erase(std::remove_if(selected_faces.begin(), selected_faces.end(), has_neighbors()), selected_faces.end());
			
				// For each face ...
				for(k3d::polyhedron::faces_t::iterator selected_face = selected_faces.begin(); selected_face != selected_faces.end(); ++selected_face)
					{
						k3d::face& face = **selected_face;

						// Calculate the face normal ...
						k3d::vector3 face_normal = k3d::normal(face);
						if(0.0 == face_normal.Length2())
							continue;
						face_normal = k3d::normalize(face_normal);
			
						// Get the set of loops to be processed ...
						typedef std::vector<k3d::split_edge**> loops_t;
						loops_t loops;
						loops.push_back(&face.first_edge);
						for(k3d::face::holes_t::iterator hole = face.holes.begin(); hole != face.holes.end(); ++hole)
							loops.push_back(&(*hole));
				
						// Create one new point for each original ...
						typedef std::map<k3d::point*, k3d::point*> point_map_t;
						point_map_t point_map;

						for(loops_t::iterator loop = loops.begin(); loop != loops.end(); ++loop)
							{
								for(k3d::split_edge* edge = **loop; edge && edge->face_clockwise; edge = edge->face_clockwise)
									{
										k3d::point* const point = new k3d::point(*edge->vertex);
										output->points.push_back(point);
										point_map[edge->vertex] = point;

										m_new_points.push_back(new_point(edge->vertex->position, -face_normal, *point));

										if(edge->face_clockwise == **loop)
											break;
									}
							}

						// Create one new edge and one new face per each original edge ...
						typedef std::map<k3d::split_edge*, k3d::split_edge*> edge_map_t;
						edge_map_t edge_map;
						
						for(loops_t::iterator loop = loops.begin(); loop != loops.end(); ++loop)
							{
								std::vector<k3d::split_edge*> new_edges;
								for(k3d::split_edge* edge = **loop; edge && edge->face_clockwise; edge = edge->face_clockwise)
									{
										k3d::split_edge* const new_edge = new k3d::split_edge(point_map[edge->face_clockwise->vertex]);
										(*polyhedron)->edges.push_back(new_edge);
										edge_map[edge] = new_edge;
										new_edges.push_back(new_edge);

										if(edge->face_clockwise == **loop)
											break;
									}
								std::reverse(new_edges.begin(), new_edges.end());
								k3d::loop_edges(new_edges.begin(), new_edges.end());

								unsigned long i = 0;
								const unsigned long edge_count = new_edges.size();
								boost::multi_array<k3d::split_edge*, 2> extrusion_edges(boost::extents[edge_count][4]);
								for(k3d::split_edge* edge = **loop; edge && edge->face_clockwise; edge = edge->face_clockwise, ++i)
									{
										extrusion_edges[i][0] = new k3d::split_edge(edge->face_clockwise->vertex);
										extrusion_edges[i][1] = new k3d::split_edge(edge->vertex);
										extrusion_edges[i][2] = new k3d::split_edge(point_map[edge->vertex]);
										extrusion_edges[i][3] = new k3d::split_edge(point_map[edge->face_clockwise->vertex]);

										extrusion_edges[i][0]->face_clockwise = extrusion_edges[i][1];
										extrusion_edges[i][1]->face_clockwise = extrusion_edges[i][2];
										extrusion_edges[i][2]->face_clockwise = extrusion_edges[i][3];
										extrusion_edges[i][3]->face_clockwise = extrusion_edges[i][0];

										(*polyhedron)->edges.push_back(extrusion_edges[i][0]);
										(*polyhedron)->edges.push_back(extrusion_edges[i][1]);
										(*polyhedron)->edges.push_back(extrusion_edges[i][2]);
										(*polyhedron)->edges.push_back(extrusion_edges[i][3]);

										k3d::join_edges(*extrusion_edges[i][0], *edge);
										k3d::join_edges(*extrusion_edges[i][2], *edge_map[edge]);

										k3d::face* const new_face = new k3d::face(extrusion_edges[i][2]);
										(*polyhedron)->faces.push_back(new_face);

										if(edge->face_clockwise == **loop)
											break;
									}

								for(unsigned long i = 0; i != edge_count; ++i)
									k3d::join_edges(*extrusion_edges[i][3], *extrusion_edges[(i+1)%edge_count][1]);
							}

						// Create a new face ...
						k3d::face* const new_face = new k3d::face(face);
						(*polyhedron)->faces.push_back(new_face);
						new_face->first_edge = edge_map[new_face->first_edge];
						for(k3d::face::holes_t::iterator hole = new_face->holes.begin(); hole != new_face->holes.end(); ++hole)
							*hole = edge_map[*hole];
					}

				assert_warning(is_valid(**polyhedron));
				assert_warning(is_solid(**polyhedron));
			}

		// Position all the new geometry we've created ...
		reshape_geometry();

		return output;
	}

	void reshape_geometry()
	{
		const double distance = m_distance.property_value();

		for(new_points_t::iterator new_point = m_new_points.begin(); new_point != m_new_points.end(); ++new_point)
			new_point->update(distance);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<extrude_faces_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xdbd23c4e, 0x7119478c, 0x9ab2ee83, 0x5b56d527),
				"ExtrudeFaces",
				"Extrudes selected faces along their normals",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_distance;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_cap;

	/// Caches new geometry for better interactive performance
	class new_point
	{
	public:
		new_point(const k3d::vector3& OldPosition, const k3d::vector3& Direction, k3d::point& Point) :
			old_position(OldPosition),
			direction(Direction),
			point(&Point)
		{
		}
	
		void update(const double Distance)
		{
			point->position = old_position + (Distance * direction);
		}
	
	private:
		k3d::vector3 old_position;
		k3d::vector3 direction;
		k3d::point* point;
	};
	
	/// Caches new geometry for better interactive performance
	typedef std::vector<new_point> new_points_t;
	/// Caches new geometry for better interactive performance
	new_points_t m_new_points;

};

/////////////////////////////////////////////////////////////////////////////
// extrude_faces_factory

k3d::iplugin_factory& extrude_faces_factory()
{
	return extrude_faces_implementation::get_factory();
}

} // namespace libk3dmesh

