// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/irenderman.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

#include "blobby_polygonizer.h"

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// polygonize_blobbies_implementation

class polygonize_blobbies_implementation :
	public k3d::material_collection<k3d::mesh_filter<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_filter<k3d::persistent<k3d::object> > > base;

public:
	polygonize_blobbies_implementation(k3d::idocument& Document) :
		base(Document),
		m_voxels(k3d::init_name("voxels") + k3d::init_description("Voxels number [integer]") + k3d::init_document(Document) + k3d::init_value(30) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_voxels));

		register_property(m_voxels);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &polygonize_blobbies_implementation::on_reset_geometry));

		m_material.changed_signal().connect(SigC::slot(*this, &polygonize_blobbies_implementation::on_reset_geometry));
		m_voxels.changed_signal().connect(SigC::slot(*this, &polygonize_blobbies_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &polygonize_blobbies_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we convert input mesh polyhedra to blobbies ...
		k3d::mesh* const output = new k3d::mesh();
		update_geometry(*input, *output);

		return output;
	}

	void update_geometry(const k3d::mesh& Input, k3d::mesh& Output)
	{
		const unsigned long voxels = m_voxels.property_value();

		// Polygonize each blobby ...
		for(k3d::mesh::blobbies_t::const_iterator blobby = Input.blobbies.begin(); blobby != Input.blobbies.end(); blobby++)
			{
				Output.polyhedra.push_back(new k3d::polyhedron());
				k3d::polyhedron& polyhedron = *Output.polyhedra.back();
				polyhedron.material = m_material.interface();

				// Polygonize blobby
				detail::vertices_t blobby_vertices;
				detail::vertices_t blobby_normals;
				detail::polygons_t blobby_polygons;
				detail::polygonize_blobby(*blobby, voxels, blobby_vertices, blobby_normals, blobby_polygons);

				// Add geometry to the mesh ...
				detail::vertices_t::const_iterator vertex = blobby_vertices.begin();
				detail::vertices_t::const_iterator normal = blobby_normals.begin();
				detail::vertices_t::const_iterator vertices_end = blobby_vertices.end();
				for(; vertex != vertices_end; vertex++, normal++)
					{
						k3d::point* point = new k3d::point(*vertex);
						point->vertex_data["N"] = k3d::ri::normal(-(*normal));
						Output.points.push_back(point);
					}

				for(unsigned long i = 0; i != blobby_polygons.size(); ++i)
					{
						if(!blobby_polygons[i].size())
							continue;

						k3d::polyhedron::edges_t edges;
						for(unsigned long j = 0; j != blobby_polygons[i].size(); ++j)
							edges.push_back(new k3d::split_edge(Output.points[blobby_polygons[i][j]]));
						for(unsigned long j = 0; j != blobby_polygons[i].size(); ++j)
							edges[j]->face_clockwise = edges[(j+1) % blobby_polygons[i].size()];

						polyhedron.edges.insert(polyhedron.edges.end(), edges.begin(), edges.end());
						polyhedron.faces.push_back(new k3d::face(edges.front()));
					}

				assert_warning(is_valid(polyhedron));
				//assert_warning(is_solid(polyhedron));
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<polygonize_blobbies_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0x446d652b, 0x938042d6, 0x9f006d66, 0x3cbff023),
				"PolygonizeBlobbies",
				"Polygonizes blobby implicit surfaces",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_voxels;
};

/////////////////////////////////////////////////////////////////////////////
// polygonize_blobbies_factory

k3d::iplugin_factory& polygonize_blobbies_factory()
{
	return polygonize_blobbies_implementation::get_factory();
}

} // namespace libk3dmesh


