// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

#include "bitmap_filter_simple.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_colordiff_implementation

class bitmap_matte_colordiff_implementation :
	public bitmap_filter_simple
{
	typedef bitmap_filter_simple base;

public:
	bitmap_matte_colordiff_implementation(k3d::idocument& Document) :
		base(Document),
		m_value(k3d::init_name("value") + k3d::init_description("Threshold value [double]") + k3d::init_value(0.0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_value));
		register_property(m_value);
		m_value.changed_signal().connect(SigC::slot(*this, &bitmap_matte_colordiff_implementation::on_value_change));
	}
	
	void on_value_change()
	{
		m_value_cache = m_value.property_value();
		on_reset_bitmap();
	}
	
        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		O.red   = I.red;
		O.green = I.green;
		O.blue  = std::min(I.green, I.blue);
		O.alpha = I.blue > m_value_cache ? 1 - (I.blue - std::max(I.red, I.green)) : 1;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_matte_colordiff_implementation>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xeefaccf2, 0x65bc4c78, 0xbd46cbdb, 0x5ca5d3e2),
				"BitmapMatteColorDiff",
				"Create alpha channel using color difference",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_value;
	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_colordiff_factory

k3d::iplugin_factory& bitmap_matte_colordiff_factory()
{
	return bitmap_matte_colordiff_implementation::get_factory();
}

} // namespace libk3dbitmap

