// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// face_normals_implementation

class face_normals_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	face_normals_implementation(k3d::idocument& Document) :
		base(Document),
		m_normalize(k3d::init_name("normalize") + k3d::init_description("Normalize normals [boolean]") + k3d::init_value(false) + k3d::init_document(Document)),
		m_scale(k3d::init_name("normals_scale") + k3d::init_description("Normals scale [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_normalize));
		enable_serialization(k3d::persistence::proxy(m_scale));

		register_property(m_normalize);
		register_property(m_scale);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &face_normals_implementation::on_reset_geometry));

		m_normalize.changed_signal().connect(SigC::slot(*this, &face_normals_implementation::on_reset_geometry));
		m_scale.changed_signal().connect(SigC::slot(*this, &face_normals_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &face_normals_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we convert input mesh polyhedra to blobbies ...
		k3d::mesh* const output = new k3d::mesh();
		update_geometry(*input, *output);

		return output;
	}

	void update_geometry(const k3d::mesh& input, k3d::mesh& output)
	{
		const bool normalize = m_normalize.property_value();
		const double scale = m_scale.property_value();

		// Create output geometry
		k3d::linear_curve_group* const group = new k3d::linear_curve_group();
		output.linear_curve_groups.push_back(group);

		// Build face normals
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = input.polyhedra.begin(); polyhedron != input.polyhedra.end(); ++polyhedron)
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				k3d::split_edge* first = (*face)->first_edge;
				if(!first)
					// Skip empty faces
					continue;

				k3d::vector3 v_sum = first->vertex->position;
				unsigned long n_vertices = 1;

				k3d::split_edge* current_edge = first->face_clockwise;
				while(current_edge)
				{
					v_sum += current_edge->vertex->position;
					++n_vertices;

					current_edge = current_edge->face_clockwise;
					if(current_edge == first)
						break;
				}

				// Get barycentric point and face normal
				v_sum /= static_cast<double>(n_vertices);
				k3d::vector3 normal = k3d::normal((*face)->first_edge);
				if(normalize)
					normal.Normalize();
				normal *= scale;

				// Create normal in output geometry
				k3d::linear_curve* const curve = new k3d::linear_curve();

				const k3d::vector3 base = v_sum;
				const k3d::vector3 tip = v_sum + normal;

				k3d::point* const base_p = new k3d::point(base);
				k3d::point* const tip_p = new k3d::point(tip);

				output.points.push_back(base_p);
				output.points.push_back(tip_p);

				curve->control_points.push_back(base_p);
				curve->control_points.push_back(tip_p);

				group->curves.push_back(curve);
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<face_normals_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0xb5cd532b, 0x6ebd4848, 0x86a72136, 0xe25151eb),
				"FaceNormals",
				"Face normals",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_normalize;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_scale;
};

/////////////////////////////////////////////////////////////////////////////
// face_normals_factory

k3d::iplugin_factory& face_normals_factory()
{
	return face_normals_implementation::get_factory();
}

} // namespace libk3dmesh


