#ifndef K3DSDK_IAPPLICATION_H
#define K3DSDK_IAPPLICATION_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares k3d::iapplication, an abstract interface for the global K-3D Application object
		\author Tim Shead (tshead@k-3d.com)
*/

#include "signal_accumulators.h"
#include "iplugin_factory_collection.h"
#include "iunknown.h"
#include "signal_system.h"
#include "sl.h"

#include <string>
#include <vector>

namespace boost { namespace filesystem { class path; } }

namespace k3d
{

class idocument;
class irender_farm;
class iscript_engine;
class iuser_interface;

/// Abstract interface for the global K-3D Application object
class iapplication :
	public virtual iunknown
{
public:
	/// Returns a reference to a user interface object
	virtual iuser_interface& user_interface() = 0;

	/// Exit K-3D if safe to do so
	virtual bool exit() = 0;

	/// Creates a new K-3D document (could return NULL)
	virtual idocument* create_document() = 0;
	/// Closes an open K-3D document
	virtual void close_document(idocument& Document) = 0;

	/// A collection of idocument objects
	typedef std::vector<idocument*> document_collection_t;
	/// Returns the collection of open documents
	virtual const document_collection_t documents() = 0;

	/// Returns the absolute path to the directory where compiled shaders will be cached
	virtual const boost::filesystem::path shader_cache_path() = 0;
	/// Returns the absolute path to the shared-data directory
	virtual const boost::filesystem::path share_path() = 0;

	/// Returns the collection of available plugin factories
	virtual const iplugin_factory_collection::factories_t& plugins() = 0;
	/// Returns the collection of available RenderMan shaders
	virtual const sl::shaders_t& shaders() = 0;
	/// Returns the current render farm
	virtual irender_farm& render_farm() = 0;

	// Signals

	/// Defines a signal emitted to display progress messages to the user during application startup
	typedef sigc::signal1<void, const std::string&> startup_message_signal_t;
	virtual startup_message_signal_t& startup_message_signal() = 0;

	/// Defines a signal emitted before the application is closed - observers can return false to cancel closing (e.g. if a resource is unsaved), true otherwise
	typedef sigc::signal0<bool, signal::cancelable> safe_to_close_signal_t;
	virtual safe_to_close_signal_t& safe_to_close_signal() = 0;

	/// Defines a signal emitted when the application is closing
	typedef sigc::signal0<void> close_signal_t;
	virtual close_signal_t& close_signal() = 0;

	/// Defines a signal emitted immediately prior to creating a document
	typedef sigc::signal0<void> pre_create_document_signal_t;
	virtual pre_create_document_signal_t& pre_create_document_signal() = 0;

	/// Defines a signal emitted when an existing document is closed
	typedef sigc::signal1<void, idocument&> close_document_signal_t;
	virtual close_document_signal_t& close_document_signal() = 0;

protected:
	iapplication() {}
	iapplication(const iapplication&) {}
	iapplication& operator = (const iapplication&) { return *this; }
	virtual ~iapplication() {}
};

} // namespace k3d

#endif // K3DSDK_IAPPLICATION_H


