// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Dan Erikson (derikson@montana.com)
*/

#include "idocument.h"
#include "istate_container.h"
#include "istate_recorder.h"
#include "result.h"
#include "state_change_set.h"
#include "utility.h"

#include <algorithm>
#include <functional>
#include <vector>

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// state_change_set

state_change_set::state_change_set() :
	m_recording_done(false)
{
	m_recording_done_signal.connect(sigc::mem_fun(*this, &state_change_set::on_recording_done));
}

state_change_set::~state_change_set()
{
	std::for_each(m_old_states.begin(), m_old_states.end(), delete_object());
	std::for_each(m_new_states.begin(), m_new_states.end(), delete_object());
}

void state_change_set::on_recording_done()
{
	if(m_recording_done)
		k3d::log() << warning << "state change set already finalized" << std::endl;

	if(m_old_states.empty())
		k3d::log() << warning << "state change set missing undo data" << std::endl;
	
	m_recording_done = true;
}

void state_change_set::record_old_state(istate_container* const OldState)
{
	// Sanity checks ...
	return_if_fail(OldState);
	m_old_states.push_back(OldState);
}

void state_change_set::record_new_state(istate_container* const NewState)
{
	// Sanity checks ...
	return_if_fail(NewState);
	m_new_states.push_back(NewState);
}

state_change_set::recording_done_signal_t& state_change_set::recording_done_signal()
{
	return m_recording_done_signal;
}

state_change_set::undo_signal_t& state_change_set::undo_signal()
{
	return m_undo_signal;
}

state_change_set::redo_signal_t& state_change_set::redo_signal()
{
	return m_redo_signal;
}

void state_change_set::undo()
{
	std::for_each(m_old_states.rbegin(), m_old_states.rend(), std::mem_fun(&istate_container::restore_state));
	m_undo_signal.emit();
}

void state_change_set::redo()
{
	std::for_each(m_new_states.begin(), m_new_states.end(), std::mem_fun(&istate_container::restore_state));
	m_redo_signal.emit();
}

/////////////////////////////////////////////////////////////////////////////
// create_state_change_set

std::auto_ptr<state_change_set> create_state_change_set()
{
	return std::auto_ptr<state_change_set>(new state_change_set());
}

/////////////////////////////////////////////////////////////////////////////
// start_state_change_set

void  start_state_change_set(idocument& Document)
{
	Document.state_recorder().start_recording(create_state_change_set());
}

/////////////////////////////////////////////////////////////////////////////
// cancel_state_change_set

void cancel_state_change_set(idocument& Document)
{
	// Tell the document to stop recording ...
	const std::auto_ptr<state_change_set> changeset(Document.state_recorder().stop_recording());

	// Undo any changes up to this point, and let the leftover data get destroyed ...
	return_if_fail(changeset.get());
	changeset->undo();
}

/////////////////////////////////////////////////////////////////////////////
// finish_state_change_set

void finish_state_change_set(idocument& Document, const std::string& Label)
{
	Document.state_recorder().commit_change_set(Document.state_recorder().stop_recording(), Label);
}

/////////////////////////////////////////////////////////////////////////////
// record_state_change_set

record_state_change_set::record_state_change_set(idocument& Document, const std::string& Label) :
	m_document(Document),
	m_label(Label)
{
	// Sanity checks ...
	assert_warning(m_label.size());

	start_state_change_set(m_document);
}

record_state_change_set::~record_state_change_set()
{
	finish_state_change_set(m_document, m_label);
}

} //namespace k3d


