#ifndef K3DSDK_SYSTEM
#define K3DSDK_SYSTEM

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares system related functions
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <boost/filesystem/path.hpp>
#include <string>
#include <vector>

namespace k3d
{

namespace system
{

/// Safely returns an environment variable
const std::string get_env(const std::string& EnvironmentVariable);
/// Safely returns the user's home directory
const boost::filesystem::path get_home_directory();
/// Safely returns the user's temp directory
const boost::filesystem::path get_temp_directory();
/// Returns a unique temporary file path
const boost::filesystem::path generate_temp_file();

/// Returns the most recent modification time of a file
bool file_modification_time(const boost::filesystem::path& File, time_t& ModificationTime);

/// Runs an external process asynchronously
bool async_run_process(const std::string& CommandLine);
/// Runs an external process, blocking until it completes
bool run_process(const std::string& CommandLine);
/// Runs an external process, blocking until it completes, and redirects its stdout to a stream
bool read_from_process(const std::string& CommandLine, std::ostream& Stream);
/// Runs an external process, blocking until it completes, and redirects its stdin from a stream
bool write_to_process(const std::string& CommandLine, std::istream& Stream);

/// Defines a collection of paths
typedef std::vector<boost::filesystem::path> paths_t;
/// Split a string containing zero-or-more paths separated by delimiters into a collection of paths ...
void decompose_path_list(const std::string Input, paths_t& Output);

/// Blocks the calling thread for the given number of seconds
void sleep(const double Seconds);

} // namespace system

} // namespace k3d

#endif // !K3DSDK_SYSTEM


