#ifndef K3DSDK_VECTORS_H
#define K3DSDK_VECTORS_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Vectors and algebra routines
		\author Timothy M. Shead (tshead@k-3d.com)
*/

/****************************************************************
*
* C++ Vector and Matrix Algebra routines
* Author: Jean-Francois DOUE
* Version 3.1 --- October 1993
*
****************************************************************/

//
//	From "Graphics Gems IV / Edited by Paul S. Heckbert
//	Academic Press, 1994, ISBN 0-12-336156-9
//	"You are free to use and modify this code in any way
//	you like." (p. xv)
//
//	Modified by J. Nagle, March 1997
//	-	All functions are inline.
//	-	All functions are const-correct.
//	-	All checking is via the standard "assert" macro.
//

// Modified by Tim Shead for use with K-3D, January 1998

#include <cassert>
#include <cmath>
#include <iostream>
#include <vector>

namespace k3d
{

// Forward declarations ...
class vector2;
class vector3;
class vector4;
class normal2;
class normal3;

/////////////////////////////////////////////////////////////////////////////
// vector2

/// A two-dimensional vector
class vector2
{
public:
	/// Stores the vector values
	double n[2];

	vector2();
	vector2(const double x, const double y);
	vector2(const double d);
	vector2(const double[2]);
	vector2(const vector2& v);
	vector2& operator=(const vector2& v);
	/// Assigns a C/C++ style array
	vector2& operator=(const double[2]);
	/// Adds a vector2
	vector2& operator+=(const vector2& v);
	/// Subtracts a vector2
	vector2& operator-=(const vector2& v);
	/// Multiplication by a constant
	vector2& operator*=(const double d);
	/// Division by a constant
	vector2& operator/=(const double d);
	/// Returns an indexed dimension by reference
	double& operator[](int i);
	/// Returns an indexed dimension by value
	double vector2::operator[](int i) const;
	/// Returns the vector length
	double Length() const;
	/// Returns the square of the vector length
	double Length2() const;
	/// Normalizes the vector in place
	vector2& Normalize();
/*
	/// Returns the perpendicular to this vector
	vector2 Perpendicular() const;
*/
	/// Returns the angle of this vector with respect to the positive X axis in radians [-pi, pi]
	double angle() const
	{
		return atan2(n[1], n[0]);
	}

	friend std::ostream& operator<<(std::ostream& Stream, const vector2& RHS)
	{
		Stream << RHS.n[0] << " " << RHS.n[1];
		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, vector2& RHS)
	{
		Stream >> RHS.n[0];
		RHS.n[1] = RHS.n[0];
		Stream >> RHS.n[1];

		return Stream;
	}
};

/// Negation
vector2 operator-(const vector2& v);
/// Addition
vector2 operator+(const vector2& a, const vector2& b);
/// Subtraction
vector2 operator-(const vector2& a, const vector2& b);
/// Multiplication by a constant
vector2 operator*(const vector2& a, const double d);
/// Multiplication by a constant
vector2 operator*(const double d, const vector2& a);
/// Returns the dot product of two vectors
double operator*(const vector2& a, const vector2& b);
/// Division by a constant
vector2 operator/(const vector2& a, const double d);
/// Returns the cross product of two vectors
vector3 operator^(const vector2& a, const vector2& b);
/// Tests for equality
bool operator==(const vector2& a, const vector2& b);
/// Tests for non-equality
bool operator!=(const vector2& a, const vector2& b);

/////////////////////////////////////////////////////////////////////////////
// vector3

/// A three-dimensional vector
class vector3
{
public:
	/// Stores the vector values
	double n[3];

	// Constructors
	vector3();
	vector3(const double x, const double y, const double z);
	vector3(const double d);
	vector3(const double d[3]);
	/// Copy constructor
	vector3(const vector3& v);
	/// Casts an vector2 to an vector3 (with zero third dimension)
	vector3(const vector2& v);
	/// Casts an vector2 to an vector3 with the given third dimension
	vector3(const vector2& v, double d);
	/// Assignment of an vector3
	vector3& operator=(const vector3& v);
	/// Assignment of a C/C++ style array
	vector3& operator=(const double d[3]);
	/// Addition
	vector3& operator+=(const vector3& v);
	vector3& operator+=(const normal3& v);
	/// Subtraction
	vector3& operator-=(const vector3& v);
	vector3& operator-=(const normal3& v);
	/// Multiplication by a constant
	vector3& operator*=(const double d);
	/// Division by a constant
	vector3& operator/=(const double d);
	/// Returns the given dimension by reference
	double& operator[](int i);
	/// Returns the given dimension by value
	double operator[](int i) const;
	/// Returns the vector length
	double Length() const;
	/// Returns the squared vector length
	double Length2() const;
	/// Normalizes the vector in place
	vector3& Normalize();
	/// Converts Cartesian coordinates to spherical coordinates
	vector3 Spherical() const;
	/// Sets each dimension to its absolute value in place
	vector3& Abs();
	/// Casting to a C/C++ style array pointer
	/** \deprecated */
	operator double*() { return &n[0]; }
	/// A better way to integrate with legacy array-oriented APIs
	const double* data() const { return &n[0]; }
	/// A better way to integrate with legacy array-oriented APIs
	double* data() { return &n[0]; }
	/// Copies the vector into a C/C++ style array
	void CopyArray(float f[3]) const;
	/// Copies the vector into a C/C++ style array
	void CopyArray(double d[3]) const;

	friend std::ostream& operator<<(std::ostream& Stream, const vector3& RHS)
	{
		Stream << RHS.n[0] << " " << RHS.n[1] << " " << RHS.n[2];
		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, vector3& RHS)
	{
		Stream >> RHS.n[0];
		RHS.n[1] = RHS.n[2] = RHS.n[0];
		Stream >> RHS.n[1] >> RHS.n[2];

		return Stream;
	}
};

/// Negation
vector3 operator-(const vector3& v);
/// Addition
vector3 operator+(const vector3& a, const vector3& b);
/// Subtraction
vector3 operator-(const vector3& a, const vector3& b);
/// Multiplication by a constant
vector3 operator*(const vector3& a, const double d);
/// Multiplication by a constant
vector3 operator*(const double d, const vector3& a);
/// Returns the dot product of two vectors
double operator*(const vector3& a, const vector3& b);
/// Division by a constant
vector3 operator/(const vector3& a, const double d);
/// Returns the cross product of two vectors
vector3 operator^(const vector3& a, const vector3& b);
/// Equality
bool operator==(const vector3& a, const vector3& b);
/// Non-equality
bool operator!=(const vector3& a, const vector3& b);
/// Returns the term-by-term minimum of two vectors
vector3 vectorMin(const vector3& a, const vector3& b);
/// Returns the term-by-term maximum of two vectors
vector3 vectorMax(const vector3& a, const vector3& b);

/////////////////////////////////////////////////////////////////////////////
// vector4

/// A four-dimensional vector
class vector4
{
public:
	/// Stores the vector values
	double n[4];

	// Constructors
	vector4();
	vector4(const double x, const double y, const double z, const double w);
	vector4(const double d);
	vector4(const double d[4]);
	/// Copy Constructor
	vector4(const vector4& v);
	/// Casts an vector3 to an vector4 with zero fourth dimension
	vector4(const vector3& v);
	/// Casts an vector3 to an vector4 with the given fourth dimension
	vector4(const vector3& v, const double d);
	/// Assignment of an vector4
	vector4& operator=(const vector4& v);
	/// Assignment of a C/C++ array
	vector4& operator=(const double d[4]);
	/// Addition
	vector4& operator+=(const vector4& v);
	/// Subtraction
	vector4& operator-=(const vector4& v);
	/// Multiplication by a constant
	vector4& operator*=(const double d);
	/// Division by a constant
	vector4& operator/=(const double d);
	/// Returns a vector element by reference
	double& operator[](int i);
	/// Returns a vector element by value
	double operator[](int i) const;
	/// Returns the length of a vector
	double Length() const;
	/// Returns the square of the vector length
	double Length2() const;
	/// Normalizes the vector in place
	vector4& Normalize();
	/// Casts the contents of the vector to a C/C++ array
	operator double*() { return &n[0]; }

	friend std::ostream& operator<<(std::ostream& Stream, const vector4& RHS)
	{
		Stream << RHS.n[0] << " " << RHS.n[1] << " " << RHS.n[2] << " " << RHS.n[3];
		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, vector4& RHS)
	{
		Stream >> RHS.n[0];
		RHS.n[1] = RHS.n[2] = RHS.n[3] = RHS.n[0];
		Stream >> RHS.n[1] >> RHS.n[2] >> RHS.n[3];

		return Stream;
	}
};

/// Negation
vector4 operator-(const vector4& v);
/// Addition
vector4 operator+(const vector4& a, const vector4& b);
/// Subtraction
vector4 operator-(const vector4& a, const vector4& b);
/// Multiplication by a constant
vector4 operator*(const vector4& a, const double d);
/// Multiplication by a constant
vector4 operator*(const double d, const vector4& a);
/// Returns the dot product of two vectors
double operator*(const vector4& a, const vector4& b);
/// Division by a constant
vector4 operator/(const vector4& a, const double d);
/// Equality
bool operator==(const vector4& a, const vector4& b);
/// Non-equality
bool operator!=(const vector4& a, const vector4& b);

/////////////////////////////////////////////////////////////////////////////
// normal2

/// Encapsulates a 2D normal vector (a direction, without any position)
class normal2
{
public:
	/// Stores the normal values
	double n[2];

	normal2()
	{
		n[0] = n[1] = 0.0;
	}
	
	normal2(const double x, const double y)
	{
		n[0] = x;
		n[1] = y;
	}

	normal2& operator+=(const normal2& v)
	{
		n[0] += v.n[0];
		n[1] += v.n[1];
		return *this;
	}
	
	normal2& operator-=(const normal2& v)
	{
		n[0] += v.n[0];
		n[1] += v.n[1];
		return *this;
	}
	
	normal2& operator*=(const double d)
	{
		n[0] *= d;
		n[1] *= d;
		return *this;
	}
	
	normal2& operator/=(const double d)
	{
		n[0] /= d;
		n[1] /= d;
		return *this;
	}
	
	double& operator[](const unsigned int i)
	{
		return n[i];
	}
	
	double operator[](const unsigned int i) const
	{
		return n[i];
	}
	
	/// Returns the normal length
	double length() const
	{
		return sqrt(length2());
	}
	
	/// Returns the squared normal length
	double length2() const
	{
		return n[0] * n[0] + n[1] * n[1];
	}

	friend std::ostream& operator<<(std::ostream& Stream, const normal2& RHS)
	{
		Stream << RHS.n[0] << " " << RHS.n[1];
		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, normal2& RHS)
	{
		Stream >> RHS.n[0] >> RHS.n[1];
		return Stream;
	}
};

/////////////////////////////////////////////////////////////////////////////
// normal3

/// Encapsulates a 3D normal vector (a direction, without any position)
class normal3
{
public:
	/// Stores the normal values
	double n[3];

	normal3()
	{
		n[0] = n[1] = n[3] = 0.0;
	}
	
	normal3(const double x, const double y, const double z)
	{
		n[0] = x;
		n[1] = y;
		n[2] = z;
	}

	normal3& operator+=(const normal3& v)
	{
		n[0] += v.n[0];
		n[1] += v.n[1];
		n[2] += v.n[2];
		return *this;
	}
	
	normal3& operator-=(const normal3& v)
	{
		n[0] += v.n[0];
		n[1] += v.n[1];
		n[2] += v.n[2];
		return *this;
	}
	
	normal3& operator*=(const double d)
	{
		n[0] *= d;
		n[1] *= d;
		n[2] *= d;
		return *this;
	}
	
	normal3& operator/=(const double d)
	{
		n[0] /= d;
		n[1] /= d;
		n[2] /= d;
		return *this;
	}
	
	double& operator[](const unsigned int i)
	{
		return n[i];
	}
	
	double operator[](const unsigned int i) const
	{
		return n[i];
	}
	
	/// Returns the normal length
	double length() const
	{
		return sqrt(length2());
	}
	
	/// Returns the squared normal length
	double length2() const
	{
		return n[0] * n[0] + n[1] * n[1] + n[2] * n[2];
	}

	friend std::ostream& operator<<(std::ostream& Stream, const normal3& RHS)
	{
		Stream << RHS.n[0] << " " << RHS.n[1] << " " << RHS.n[2];
		return Stream;
	}

	friend std::istream& operator>>(std::istream& Stream, normal3& RHS)
	{
		Stream >> RHS.n[0] >> RHS.n[1] >> RHS.n[2];
		return Stream;
	}
};

/////////////////////////////////////////////////////////////////////////////
// vector2 implementation

inline vector2::vector2()
{ n[0] = n[1] = 0.0; }

inline vector2::vector2(const double x, const double y)
{ n[0] = x; n[1] = y; }

inline vector2::vector2(const double d)
{ n[0] = n[1] = d; }

inline vector2::vector2(const double d[2])
{ n[0] = d[0]; n[1] = d[1]; }

inline vector2::vector2(const vector2& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; }

inline vector2& vector2::operator=(const vector2& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; return *this; }

inline vector2& vector2::operator=(const double d[2])
{ n[0] = d[0]; n[1] = d[1]; return *this; }

inline vector2& vector2::operator+=(const vector2& v)
{ n[0] += v.n[0]; n[1] += v.n[1]; return *this; }

inline vector2& vector2::operator-=(const vector2& v)
{ n[0] -= v.n[0]; n[1] -= v.n[1]; return *this; }

inline vector2& vector2::operator*=(const double d)
{ n[0] *= d; n[1] *= d; return *this; }

inline vector2& vector2::operator/=(const double d)
{ double d_inv = 1./d; n[0] *= d_inv; n[1] *= d_inv; return *this; }

inline double& vector2::operator[](int i)
{
 assert(!(i < 0 || i > 1));
 return n[i];
}

inline double vector2::operator[](int i) const
{
 assert(!(i < 0 || i > 1));
 return n[i];
}

inline double vector2::Length() const
{ return sqrt(Length2()); }

inline double vector2::Length2() const
{ return n[0]*n[0] + n[1]*n[1]; }

inline vector2& vector2::Normalize()
{
	if(const double length = Length())
		*this /= length;

	return *this;
}

/*
inline vector2 vector2::Perpendicular() const
{
	return vector2(-n[1], n[0]);
}
*/

inline vector2 operator-(const vector2& a)
{ return vector2(-a.n[0],-a.n[1]); }

inline vector2 operator+(const vector2& a, const vector2& b)
{ return vector2(a.n[0]+ b.n[0], a.n[1] + b.n[1]); }

inline vector2 operator-(const vector2& a, const vector2& b)
{ return vector2(a.n[0]-b.n[0], a.n[1]-b.n[1]); }

inline vector2 operator*(const vector2& a, const double d)
{ return vector2(d*a.n[0], d*a.n[1]); }

inline vector2 operator*(const double d, const vector2& a)
{ return a*d; }

inline double operator*(const vector2& a, const vector2& b)
{ return (a.n[0]*b.n[0] + a.n[1]*b.n[1]); }

inline vector2 operator/(const vector2& a, const double d)
{ double d_inv = 1./d; return vector2(a.n[0]*d_inv, a.n[1]*d_inv); }

inline vector3 operator^(const vector2& a, const vector2& b)
{ return vector3(0, 0, a.n[0] * b.n[1] - b.n[0] * a.n[1]); }

inline bool operator==(const vector2& a, const vector2& b)
{ return (a.n[0] == b.n[0]) && (a.n[1] == b.n[1]); }

inline bool operator!=(const vector2& a, const vector2& b)
{ return !(a == b); }

/////////////////////////////////////////////////////////////////////////////
// vector3 implementation

inline vector3::vector3()
{ n[0] = n[1] = n[2] = 0.0; }

inline vector3::vector3(const double x, const double y, const double z)
{ n[0] = x; n[1] = y; n[2] = z; }

inline vector3::vector3(const double d)
{ n[0] = n[1] = n[2] = d; }

inline vector3::vector3(const double d[3])
{ n[0] = d[0]; n[1] = d[1]; n[2] = d[2]; }

inline vector3::vector3(const vector3& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; }

inline vector3::vector3(const vector2& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = 1.0; }

inline vector3::vector3(const vector2& v, double d)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = d; }

inline vector3& vector3::operator=(const vector3& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; return *this; }

inline vector3& vector3::operator=(const double d[3])
{ n[0] = d[0]; n[1] = d[1]; n[2] = d[2]; return *this; }

inline vector3& vector3::operator+=(const vector3& v)
{ n[0] += v.n[0]; n[1] += v.n[1]; n[2] += v.n[2]; return *this; }

inline vector3& vector3::operator+=(const normal3& v)
{ n[0] += v.n[0]; n[1] += v.n[1]; n[2] += v.n[2]; return *this; }

inline vector3& vector3::operator-=(const vector3& v)
{ n[0] -= v.n[0]; n[1] -= v.n[1]; n[2] -= v.n[2]; return *this; }

inline vector3& vector3::operator-=(const normal3& v)
{ n[0] -= v.n[0]; n[1] -= v.n[1]; n[2] -= v.n[2]; return *this; }

inline vector3& vector3::operator*=( const double d )
{ n[0] *= d; n[1] *= d; n[2] *= d; return *this; }

inline vector3& vector3::operator/=( const double d )
{ double d_inv = 1./d; n[0] *= d_inv; n[1] *= d_inv; n[2] *= d_inv;
 return *this; }

inline double& vector3::operator[](int i) {
 assert(! (i < 0 || i > 2));
 return n[i];
}

inline double vector3::operator[](int i) const {
 assert(! (i < 0 || i > 2));
 return n[i];
}

inline double vector3::Length() const
{ return sqrt(Length2()); }

inline double vector3::Length2() const
{ return n[0]*n[0] + n[1]*n[1] + n[2]*n[2]; }

inline vector3& vector3::Normalize()
{
	if(const double length = Length())
		*this /= length;

	return *this;
}

inline void vector3::CopyArray(float f[3]) const
{	f[0] = (float)n[0];	f[1] = (float)n[1];	f[2] = (float)n[2]; }

inline void vector3::CopyArray(double d[3]) const
{	d[0] = n[0]; d[1] = n[1]; d[2] = n[2]; }

inline vector3 vector3::Spherical() const
{
	return vector3(Length(), atan2(n[0], n[2]), atan2(n[1], sqrt(n[0] * n[0] + n[2] * n[2])));
/*
	// Handle the singularity at the poles
	if(0.0 == n[0] && 0.0 == n[2])
		return vector3(n[1] > 0.0 ? sdpPiOver2 : -sdpPiOver2, 0.0, Length());


	return vector3(atan2(n[1], sqrt(n[0] * n[0] + n[2] * n[2])), atan2(n[0], n[2]), Length());
*/
}

inline vector3& vector3::Abs()
{	n[0] = fabs(n[0]); n[1] = fabs(n[1]); n[2] = fabs(n[2]); return *this; }

inline vector3 operator-(const vector3& a)
{ return vector3(-a.n[0],-a.n[1],-a.n[2]); }

inline vector3 operator+(const vector3& a, const vector3& b)
{ return vector3(a.n[0]+ b.n[0], a.n[1] + b.n[1], a.n[2] + b.n[2]); }

inline vector3 operator-(const vector3& a, const vector3& b)
{ return vector3(a.n[0]-b.n[0], a.n[1]-b.n[1], a.n[2]-b.n[2]); }

inline vector3 operator*(const vector3& a, const double d)
{ return vector3(d*a.n[0], d*a.n[1], d*a.n[2]); }

inline vector3 operator*(const double d, const vector3& a)
{ return a*d; }

inline double operator*(const vector3& a, const vector3& b)
{ return (a.n[0]*b.n[0] + a.n[1]*b.n[1] + a.n[2]*b.n[2]); }

inline vector3 operator/(const vector3& a, const double d)
{ double d_inv = 1./d; return vector3(a.n[0]*d_inv, a.n[1]*d_inv,
 a.n[2]*d_inv); }

inline vector3 operator^(const vector3& a, const vector3& b) {
 return vector3(a.n[1]*b.n[2] - a.n[2]*b.n[1],
		a.n[2]*b.n[0] - a.n[0]*b.n[2],
		a.n[0]*b.n[1] - a.n[1]*b.n[0]);
}

inline bool operator==(const vector3& a, const vector3& b)
{ return (a.n[0] == b.n[0]) && (a.n[1] == b.n[1]) && (a.n[2] == b.n[2]);
}

inline bool operator!=(const vector3& a, const vector3& b)
{ return !(a == b); }

inline vector3 vectorMin(const vector3& a, const vector3& b)
{ return vector3(std::min(a.n[0], b.n[0]), std::min(a.n[1], b.n[1]), std::min(a.n[2],
 b.n[2])); }

inline vector3 vectorMax(const vector3& a, const vector3& b)
{ return vector3(std::max(a.n[0], b.n[0]), std::max(a.n[1], b.n[1]), std::max(a.n[2],
 b.n[2])); }

/////////////////////////////////////////////////////////////////////////////
// vector4 implementation

inline vector4::vector4()
{ n[0] = n[1] = n[2] = n[3] = 0.0; }

inline vector4::vector4(const double x, const double y, const double z, const double w)
{ n[0] = x; n[1] = y; n[2] = z; n[3] = w; }

inline vector4::vector4(const double d)
{ n[0] = n[1] = n[2] = n[3] = d; }

inline vector4::vector4(const double d[4])
{ n[0] = d[0]; n[1] = d[1]; n[2] = d[2]; n[3] = d[3]; }

inline vector4::vector4(const vector4& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; n[3] = v.n[3]; }

inline vector4::vector4(const vector3& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; n[3] = 1.0; }

inline vector4::vector4(const vector3& v, const double d)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; n[3] = d; }

inline vector4& vector4::operator=(const vector4& v)
{ n[0] = v.n[0]; n[1] = v.n[1]; n[2] = v.n[2]; n[3] = v.n[3];
return *this; }

inline vector4& vector4::operator=(const double d[4])
{ n[0] = d[0]; n[1] = d[1]; n[2] = d[2]; n[3] = d[3]; return *this; }

inline vector4& vector4::operator+=( const vector4& v )
{ n[0] += v.n[0]; n[1] += v.n[1]; n[2] += v.n[2]; n[3] += v.n[3];
return *this; }

inline vector4& vector4::operator-=( const vector4& v )
{ n[0] -= v.n[0]; n[1] -= v.n[1]; n[2] -= v.n[2]; n[3] -= v.n[3];
return *this; }

inline vector4& vector4::operator*=( const double d )
{ n[0] *= d; n[1] *= d; n[2] *= d; n[3] *= d; return *this; }

inline vector4& vector4::operator/=( const double d )
{ double d_inv = 1./d; n[0] *= d_inv; n[1] *= d_inv; n[2] *= d_inv;
 n[3] *= d_inv; return *this; }

inline double& vector4::operator[](int i) {
 assert(! (i < 0 || i > 3));
 return n[i];
}

inline double vector4::operator[](int i) const {
 assert(! (i < 0 || i > 3));
 return n[i];
}

inline double vector4::Length() const
{ return sqrt(Length2()); }

inline double vector4::Length2() const
{ return n[0]*n[0] + n[1]*n[1] + n[2]*n[2] + n[3]*n[3]; }

inline vector4& vector4::Normalize()
{
	if(const double length = Length())
		*this /= length;

	return *this;
}

inline vector4 operator-(const vector4& a)
{ return vector4(-a.n[0],-a.n[1],-a.n[2],-a.n[3]); }

inline vector4 operator+(const vector4& a, const vector4& b)
{ return vector4(a.n[0] + b.n[0], a.n[1] + b.n[1], a.n[2] + b.n[2],
 a.n[3] + b.n[3]); }

inline vector4 operator-(const vector4& a, const vector4& b)
{ return vector4(a.n[0] - b.n[0], a.n[1] - b.n[1], a.n[2] - b.n[2],
 a.n[3] - b.n[3]); }

inline vector4 operator*(const vector4& a, const double d)
{ return vector4(d*a.n[0], d*a.n[1], d*a.n[2], d*a.n[3] ); }

inline vector4 operator*(const double d, const vector4& a)
{ return a*d; }

inline double operator*(const vector4& a, const vector4& b)
{ return (a.n[0]*b.n[0] + a.n[1]*b.n[1] + a.n[2]*b.n[2] +
 a.n[3]*b.n[3]); }

inline vector4 operator/(const vector4& a, const double d)
{ double d_inv = 1./d; return vector4(a.n[0]*d_inv, a.n[1]*d_inv, a.n[2]*d_inv,
 a.n[3]*d_inv); }

inline bool operator==(const vector4& a, const vector4& b)
{ return (a.n[0] == b.n[0]) && (a.n[1] == b.n[1]) && (a.n[2] == b.n[2])
 && (a.n[3] == b.n[3]); }

inline bool operator!=(const vector4& a, const vector4& b)
{ return !(a == b); }

/// Negation
inline const normal2 operator-(const normal2& v)
{
	return normal2(-v.n[0], -v.n[1]);
}

/// Addition
inline const normal2 operator+(const normal2& a, const normal2& b)
{
	return normal2(a.n[0] + b.n[0], a.n[1] + b.n[1]);
}

/// Adds a vector and a normal, returning the moved vector
inline const vector2 operator+(const vector2& a, const normal2& b)
{
	return vector2(a.n[0] + b.n[0], a.n[1] + b.n[1]);
}

/// Adds a normal and a vector, returning the moved vector
inline const vector2 operator+(const normal2& a, const vector2& b)
{
	return vector2(a.n[0] + b.n[0], a.n[1] + b.n[1]);
}

/// Subtraction
inline const normal2 operator-(const normal2& a, const normal2& b)
{
	return normal2(a.n[0] - b.n[0], a.n[1] - b.n[1]);
}

/// Multiplication by a constant
inline const normal2 operator*(const normal2& a, const double d)
{
	return normal2(a.n[0] * d, a.n[1] * d);
}

/// Multiplication by a constant
inline const normal2 operator*(const double d, const normal2& a)
{
	return normal2(a.n[0] * d, a.n[1] * d);
}

/// Returns the dot product of two normals
inline const double operator*(const normal2& a, const normal2& b)
{
	return a.n[0] * b.n[0] + a.n[1] * b.n[1];
}

/// Division by a constant
inline const normal2 operator/(const normal2& a, const double d)
{
	return normal2(a.n[0] / d, a.n[1] / d);
}

/// Equality
inline const bool operator==(const normal2& a, const normal2& b)
{
	return a.n[0] == b.n[0] && a.n[1] == b.n[1];
}

/// Non-equality
inline const bool operator!=(const normal2& a, const normal2& b)
{
	return a.n[0] != b.n[0] || a.n[1] != b.n[1];
}

/// Explicit conversion
inline const normal2 to_normal(const vector2& v)
{
	return normal2(v.n[0], v.n[1]);
}

/// Explicit conversion
inline const vector2 to_vector(const normal2& v)
{
	return vector2(v.n[0], v.n[1]);
}

/// Negation
inline const normal3 operator-(const normal3& v)
{
	return normal3(-v.n[0], -v.n[1], -v.n[2]);
}

/// Addition
inline const normal3 operator+(const normal3& a, const normal3& b)
{
	return normal3(a.n[0] + b.n[0], a.n[1] + b.n[1], a.n[2] + b.n[2]);
}

/// Adds a vector and a normal, returning the moved vector
inline const vector3 operator+(const vector3& a, const normal3& b)
{
	return vector3(a.n[0] + b.n[0], a.n[1] + b.n[1], a.n[2] + b.n[2]);
}

/// Subtracts a normal from a vector, returning the modified vector
inline const vector3 operator-(const vector3& a, const normal3& b)
{
	return vector3(a.n[0] - b.n[0], a.n[1] - b.n[1], a.n[2] - b.n[2]);
}

/// Adds a normal and a vector, returning the moved vector
inline const vector3 operator+(const normal3& a, const vector3& b)
{
	return vector3(a.n[0] + b.n[0], a.n[1] + b.n[1], a.n[2] + b.n[2]);
}

/// Subtraction
inline const normal3 operator-(const normal3& a, const normal3& b)
{
	return normal3(a.n[0] - b.n[0], a.n[1] - b.n[1], a.n[2] - b.n[2]);
}

/// Multiplication by a constant
inline const normal3 operator*(const normal3& a, const double d)
{
	return normal3(a.n[0] * d, a.n[1] * d, a.n[2] * d);
}

/// Multiplication by a constant
inline const normal3 operator*(const double d, const normal3& a)
{
	return normal3(a.n[0] * d, a.n[1] * d, a.n[2] * d);
}

/// Returns the dot product of two normals
inline const double operator*(const normal3& a, const normal3& b)
{
	return a.n[0] * b.n[0] + a.n[1] * b.n[1] + a.n[2] * b.n[2];
}

/// Division by a constant
inline const normal3 operator/(const normal3& a, const double d)
{
	return normal3(a.n[0] / d, a.n[1] / d, a.n[2] / d);
}

/// Returns the cross product of two normals
inline const normal3 operator^(const normal3& a, const normal3& b)
{
	return normal3(a.n[1] * b.n[2] - a.n[2] * b.n[1], a.n[2] * b.n[0] - a.n[0] * b.n[2], a.n[0] * b.n[1] - a.n[1] * b.n[0]);
}

/// Equality
inline const bool operator==(const normal3& a, const normal3& b)
{
	return a.n[0] == b.n[0] && a.n[1] == b.n[1] && a.n[2] == b.n[2];
}

/// Non-equality
inline const bool operator!=(const normal3& a, const normal3& b)
{
	return a.n[0] != b.n[0] || a.n[1] != b.n[1] || a.n[2] != b.n[2];
}

/// Explicit conversion
inline const normal3 to_normal(const vector3& v)
{
	return normal3(v.n[0], v.n[1], v.n[2]);
}

/// Explicit conversion
inline const vector3 to_vector(const normal3& v)
{
	return vector3(v.n[0], v.n[1], v.n[2]);
}

/// Returns the normalized form of a vector (note: assumes non-zero length!)
template<typename T>
const T normalize(const T& Vector)
{
	return Vector / Vector.Length();
}

/// Returns the normalized form of a normal (note: assumes non-zero length!)
inline const normal3 normalize(const normal3& Normal)
{
	return Normal / Normal.length();
}

/// Returns the length of a vector
template<typename T>
const double length(const T& Vector)
{
	return Vector.Length();
}

/// Returns the length of a normal
inline const double length(const normal2& Normal)
{
	return Normal.length();
}

/// Returns the length of a normal
inline const double length(const normal3& Normal)
{
	return Normal.length();
}

} // namespace k3d

#endif // K3DSDK_VECTORS_H

