// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements Aqsis layered-shader plugins
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/iaqsis.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iplugin_factory.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/irenderman.h>
#include <k3dsdk/iuser_property.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/property.h>
#include <k3dsdk/property_group_collection.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/shaders.h>
#include <k3dsdk/string_modifiers.h>
#include <k3dsdk/user_properties.h>

using namespace k3d::xml;

namespace libk3daqsis
{

/////////////////////////////////////////////////////////////////////////////
// layer

/// Abstract base class that provides most of the implementation for our shader objects
class layer :
	public k3d::persistent<k3d::node>
{
	typedef k3d::persistent<k3d::node> base;

public:
	layer(k3d::idocument& Document, const k3d::sl::shader::type_t ShaderType) :
		base(Document),
		m_shader_name(init_owner(*this) + init_name("shader_name") + init_label(_("Shader Name")) + init_description(_("RenderMan Shader Name")) + init_value<std::string>("") + init_values(shader_values(ShaderType)))
	{
		m_shader_name_connection = m_shader_name.changed_signal().connect(sigc::mem_fun(*this, &layer::on_shader_changed));
	}

	void load(element& Element, const k3d::ipersistent::load_context& Context)
	{
		// Disable argument list updates while loading so we don't create the same arguments twice
		m_shader_name_connection.disconnect();
		base::load(Element, Context);
		m_shader_name_connection = m_shader_name.changed_signal().connect(sigc::mem_fun(*this, &layer::on_shader_changed));
	}

protected:
	const std::string shader_name()
	{
		return m_shader_name.value();
	}

	template<typename LayerT, typename ShaderT>
	void setup_connected_layers(const k3d::ri::render_state& State, void (ShaderT::*SetupFunction)(const k3d::ri::render_state&))
	{
		const iproperty_collection::properties_t& properties = this->properties();
		for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
		{
			k3d::aqsis::ilayer_connection_property* const layer_property = dynamic_cast<k3d::aqsis::ilayer_connection_property*>(*property);
			if(!layer_property)
				continue;

			if((*property)->property_type() != typeid(k3d::inode*))
			{
				k3d::log() << warning << "layer connection property has incorrect type" << std::endl;
				continue;
			}

			ShaderT* shader = dynamic_cast<ShaderT*>(dynamic_cast<LayerT*>(boost::any_cast<k3d::inode*>((*property)->property_value())));
			if(!shader)
				continue;

			(shader->*SetupFunction)(State);
		}
	}

	k3d::ri::parameter_list shader_arguments(const k3d::ri::render_state& State)
	{
		k3d::ri::parameter_list results;

		const k3d::iproperty_collection::properties_t& properties = base::properties();
		for(k3d::iproperty_collection::properties_t::const_iterator prop = properties.begin(); prop != properties.end(); ++prop)
		{
			if(dynamic_cast<k3d::iuser_property*>(*prop))
			{
				k3d::iproperty& property = **prop;

				if(property.property_type() == typeid(k3d::ri::real))
				{
					results.push_back(k3d::ri::parameter(property.property_name(), k3d::ri::CONSTANT, boost::any_cast<k3d::ri::real>(property.property_value())));
				}
				else if(property.property_type() == typeid(k3d::ri::string))
				{
					results.push_back(k3d::ri::parameter(property.property_name(), k3d::ri::CONSTANT, boost::any_cast<k3d::ri::string>(property.property_value())));
				}
				else if(property.property_type() == typeid(k3d::ri::color))
				{
					results.push_back(k3d::ri::parameter(property.property_name(), k3d::ri::CONSTANT, boost::any_cast<k3d::ri::color>(property.property_value())));
				}
				else if(property.property_type() == typeid(k3d::ri::itexture*))
				{
					k3d::ri::itexture* const texture = boost::any_cast<k3d::ri::itexture*>(property.property_value());
					if(texture)
						results.push_back(k3d::ri::parameter(property.property_name(), k3d::ri::CONSTANT, static_cast<k3d::ri::string>(texture->renderman_texture_path(State).native_file_string())));
				}
				else if(property.property_type() == typeid(k3d::inode*))
				{
				}
				else
				{
					k3d::log() << error << k3d_file_reference << ": unknown storage type for shader argument [" << property.property_name() << "] will be ignored" << std::endl;
				}
			}
		}

		return results;
	}

	template<typename LayerT>
	void setup_connections(const std::string& Type, const k3d::ri::render_state& State)
	{
		const iproperty_collection::properties_t& properties = this->properties();
		for(iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
		{
			k3d::aqsis::ilayer_connection_property* const layer_property = dynamic_cast<k3d::aqsis::ilayer_connection_property*>(*property);
			if(!layer_property)
				continue;

			if((*property)->property_type() != typeid(k3d::inode*))
			{
				k3d::log() << warning << "layer connection property has incorrect type" << std::endl;
				continue;
			}


			k3d::inode* const node = boost::any_cast<k3d::inode*>((*property)->property_value());
			if(!node)
				continue;

			if(!dynamic_cast<LayerT*>(node))
				continue;

			const std::string source_name = node->name();
			const std::string source_variable = layer_property->get_source_variable();
			const std::string target_name = name();
			const std::string target_variable = layer_property->get_target_variable();

			State.engine.RiConnectShaderLayers(Type, source_name, source_variable, target_name, target_variable);
		}
	}

private:
	/// Helper functor for searching for shaders by name
	struct same_name
	{
		same_name(const std::string& Name) : name(Name) {}

		template<typename T>
		bool operator()(const T& LHS)
		{
			return LHS.name == name;
		}

		const std::string name;
	};

	void on_shader_changed()
	{
		setup_arguments();
	}

	void clear()
	{
		const k3d::iproperty_collection::properties_t properties = base::properties();
		for(k3d::iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
		{
			if(dynamic_cast<k3d::iuser_property*>(*property))
			{
				if(k3d::ipersistent* const persistent = dynamic_cast<k3d::ipersistent*>(*property))
					disable_serialization(*persistent);

				unregister_property(**property);
				delete dynamic_cast<k3d::ideletable*>(*property);
			}
		}
	}

	class pause_recording
	{
	public:
		pause_recording(k3d::idocument& Document) :
			m_document(Document)
		{
			if(m_document.state_recorder().current_change_set())
				m_change_set = m_document.state_recorder().stop_recording();
		}

		~pause_recording()
		{
			if(m_change_set.get())
				m_document.state_recorder().start_recording(m_change_set);
		}

	private:
		k3d::idocument& m_document;
		std::auto_ptr<k3d::state_change_set> m_change_set;
	};

	void setup_arguments()
	{
		// At the moment, we don't support undo-able changes to the argument-set, so temporarily disable any recording ...
		pause_recording pause(document());

		clear();

		// Try to find a shader description that matches the new shader name ...
		const std::string shader_name = m_shader_name.value();
		if(shader_name.empty())
			return;

		const k3d::sl::shaders_t::const_iterator shader = std::find_if(k3d::application().shaders().begin(), k3d::application().shaders().end(), same_name(shader_name));
		if(shader == k3d::application().shaders().end())
		{
			k3d::log() << error << "Unknown shader [" << shader_name << "]" << std::endl;
			return;
		}

		for(k3d::sl::shader::arguments_t::const_iterator argument = shader->arguments.begin(); argument != shader->arguments.end(); ++argument)
		{
			if(argument->output)
				continue;

			switch(argument->extended_type)
			{
				case k3d::sl::argument::EX_FLOAT:
				case k3d::sl::argument::EX_TIME:
				case k3d::sl::argument::EX_ANGLE:
				case k3d::sl::argument::EX_DISTANCE:
				case k3d::sl::argument::EX_AREA:
				case k3d::sl::argument::EX_VOLUME:
				case k3d::sl::argument::EX_MASS:
				case k3d::sl::argument::EX_FORCE:
				case k3d::sl::argument::EX_PRESSURE:
				{
					new k3d::user::double_property(
						init_owner(*this)
						+ init_name(argument->name.c_str())
						+ init_label(argument->label.c_str())
						+ init_description(argument->description.c_str())
						+ init_value(k3d::from_string<double>(argument->default_value, 0.0)));
					break;
				}
				case k3d::sl::argument::EX_COLOR:
				{
					new k3d::user::color_property(
						init_owner(*this)
						+ init_name(argument->name.c_str())
						+ init_label(argument->label.c_str())
						+ init_description(argument->description.c_str())
						+ init_value(k3d::from_string<k3d::color>(argument->default_value, k3d::color(1, 1, 1))));
					break;
				}
				case k3d::sl::argument::EX_STRING:
				case k3d::sl::argument::EX_SPACE:
				{
					new k3d::user::string_property(
						init_owner(*this)
						+ init_name(argument->name.c_str())
						+ init_label(argument->label.c_str())
						+ init_description(argument->description.c_str())
						+ init_value(argument->default_value));
					break;
				}
				case k3d::sl::argument::EX_TEXTURE:
/*
				{
					texture_storage_t* const texture_storage = new texture_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value<k3d::ri::itexture*>(0));

					m_arguments.push_back(argument_t(argument->name, boost::any(texture_storage)));
					break;
				}
*/
				case k3d::sl::argument::EX_POINT:
				case k3d::sl::argument::EX_VECTOR:
				case k3d::sl::argument::EX_NORMAL:
				case k3d::sl::argument::EX_MATRIX:
				{
					break;
				}
				default:
				{
					k3d::log() << error << k3d_file_reference << " unknown extended argument type for [" << argument->name << "] will not receive storage" << std::endl;
				}
			}
		}
	}

	/// Storage for the shader name
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, list_property, with_serialization) m_shader_name;
	sigc::connection m_shader_name_connection;

	/// Returns lists of available shaders of the given type
	const k3d::ilist_property<std::string>::values_t& shader_values(const k3d::sl::shader::type_t Type)
	{
		static bool initialized = false;
		static k3d::ilist_property<std::string>::values_t deformation_shader_values;
		static k3d::ilist_property<std::string>::values_t imager_shader_values;
		static k3d::ilist_property<std::string>::values_t light_shader_values;
		static k3d::ilist_property<std::string>::values_t surface_shader_values;
		static k3d::ilist_property<std::string>::values_t transformation_shader_values;
		static k3d::ilist_property<std::string>::values_t volume_shader_values;
		static k3d::ilist_property<std::string>::values_t empty_shader_values;

		if(!initialized)
		{
			for(k3d::sl::shaders_t::const_iterator shader = k3d::application().shaders().begin(); shader != k3d::application().shaders().end(); ++shader)
			{
				switch(shader->type)
				{
					case k3d::sl::shader::DISPLACEMENT:
						deformation_shader_values.push_back(shader->name);
						break;

					case k3d::sl::shader::IMAGER:
						imager_shader_values.push_back(shader->name);
						break;

					case k3d::sl::shader::LIGHT:
						light_shader_values.push_back(shader->name);
						break;

					case k3d::sl::shader::SURFACE:
						surface_shader_values.push_back(shader->name);
						break;

					case k3d::sl::shader::TRANSFORMATION:
						transformation_shader_values.push_back(shader->name);
						break;

					case k3d::sl::shader::VOLUME:
						volume_shader_values.push_back(shader->name);
						break;
				}
			}

			deformation_shader_values.push_back(std::string());
			imager_shader_values.push_back(std::string());
			light_shader_values.push_back(std::string());
			surface_shader_values.push_back(std::string());
			transformation_shader_values.push_back(std::string());
			volume_shader_values.push_back(std::string());

			std::sort(deformation_shader_values.begin(), deformation_shader_values.end());
			std::sort(imager_shader_values.begin(), imager_shader_values.end());
			std::sort(light_shader_values.begin(), light_shader_values.end());
			std::sort(surface_shader_values.begin(), surface_shader_values.end());
			std::sort(transformation_shader_values.begin(), transformation_shader_values.end());
			std::sort(volume_shader_values.begin(), volume_shader_values.end());

			initialized = true;
		}

		switch(Type)
		{
			case k3d::sl::shader::DISPLACEMENT:
				return deformation_shader_values;

			case k3d::sl::shader::IMAGER:
				return imager_shader_values;

			case k3d::sl::shader::LIGHT:
				return light_shader_values;

			case k3d::sl::shader::SURFACE:
				return surface_shader_values;

			case k3d::sl::shader::TRANSFORMATION:
				return transformation_shader_values;

			case k3d::sl::shader::VOLUME:
				return volume_shader_values;
		}

		return empty_shader_values;
	}
};

/////////////////////////////////////////////////////////////////////////////
// displacement_shader_layer

/// Implements an Aqsis displacement shader layer
class displacement_shader_layer :
	public layer,
	public k3d::ri::idisplacement_shader,
	public k3d::aqsis::idisplacement_layer
{
	typedef layer base;

public:
	displacement_shader_layer(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::DISPLACEMENT)
	{
	}

	void setup_renderman_displacement_shader(const k3d::ri::render_state& State)
	{
		setup_connected_layers<k3d::aqsis::idisplacement_layer, k3d::ri::idisplacement_shader>(State, &k3d::ri::idisplacement_shader::setup_renderman_displacement_shader);
		State.engine.RiShaderLayerV("displacement", shader_name(), name(), shader_arguments(State));
		setup_connections<k3d::aqsis::idisplacement_layer>("displacement", State);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<displacement_shader_layer>,
				k3d::interface_list<k3d::ri::idisplacement_shader,
				k3d::interface_list<k3d::aqsis::idisplacement_layer> > > factory(
			k3d::classes::AqsisDisplacementShaderLayer(),
			"AqsisDisplacementShaderLayer",
			"Encapsulates an Aqsis displacement shader layer",
			"Aqsis",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// surface_shader_layer

/// Implements an Aqsis surface shader layer
class surface_shader_layer :
	public layer,
	public k3d::ri::isurface_shader,
	public k3d::aqsis::isurface_layer
{
	typedef layer base;

public:
	surface_shader_layer(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::SURFACE)
	{
	}

	void setup_renderman_surface_shader(const k3d::ri::render_state& State)
	{
		setup_connected_layers<k3d::aqsis::isurface_layer, k3d::ri::isurface_shader>(State, &k3d::ri::isurface_shader::setup_renderman_surface_shader);
		State.engine.RiShaderLayerV("surface", shader_name(), name(), shader_arguments(State));
		setup_connections<k3d::aqsis::isurface_layer>("surface", State);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<surface_shader_layer>,
				k3d::interface_list<k3d::ri::isurface_shader,
				k3d::interface_list<k3d::aqsis::isurface_layer> > > factory(
					k3d::classes::AqsisSurfaceShaderLayer(),
					"AqsisSurfaceShaderLayer",
					"Encapsulates an Aqsis surface shader layer",
					"Aqsis",
					k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// layer factories

k3d::iplugin_factory& displacement_shader_layer_factory()
{
	return displacement_shader_layer::get_factory();
}

k3d::iplugin_factory& surface_shader_layer_factory()
{
	return surface_shader_layer::get_factory();
}

} // namespace libk3daqsis

