#ifndef MODULE_BITMAP_BITMAP_COMPOSITE_SIMPLE_H
#define MODULE_BITMAP_BITMAP_COMPOSITE_SIMPLE_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/bitmap.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

#include "bitmap_composite.h"
#include "bitmap_element.h"

namespace libk3dbitmap
{

class bitmap_composite_simple :
	public bitmap_composite<k3d::persistent<k3d::node> >
{
	typedef bitmap_composite<k3d::persistent<k3d::node> > base;

public:
	bitmap_composite_simple(k3d::idocument& Document) :
		base(Document)
	{
		m_bitmap_a.changed_signal().connect(sigc::mem_fun(*this, &bitmap_composite_simple::on_reset_bitmap));
		m_bitmap_b.changed_signal().connect(sigc::mem_fun(*this, &bitmap_composite_simple::on_reset_bitmap));
		m_bitmap_o.need_data_signal().connect(sigc::mem_fun(*this, &bitmap_composite_simple::on_create_bitmap));
	}

	void on_reset_bitmap()
	{
		m_bitmap_o.reset();
	}

	k3d::bitmap* on_create_bitmap()
	{
		// If we don't have any input bitmap, we're done ...
		k3d::bitmap* const A_bitmap = m_bitmap_a.value();
		k3d::bitmap* const B_bitmap = m_bitmap_b.value();
		if(!A_bitmap && !B_bitmap)
			return 0;
		else if(!B_bitmap)
			return new k3d::bitmap(*A_bitmap);
		else if(!A_bitmap)
			return new k3d::bitmap(*B_bitmap);

		// Update values
		on_value_change();

		// Composite ...
		bitmap_element* A_element = new bitmap_element(A_bitmap);
		bitmap_element* B_element = new bitmap_element(B_bitmap);

		const int max_x = std::max(A_element->max_x, B_element->max_x);
		const int min_x = std::min(A_element->min_x, B_element->min_x);
		const int max_y = std::max(A_element->max_y, B_element->max_y);
		const int min_y = std::min(A_element->min_y, B_element->min_y);

		k3d::bitmap* const output = new k3d::bitmap(max_x-min_x, max_y-min_y);
		k3d::bitmap::iterator O = output->begin();

		for(int y = max_y; y > min_y; --y)
		{
			for(int x = min_x; x < max_x; ++x)
			{
				k3d::pixel A = A_element->get_pixel(x, y);
				k3d::pixel B = B_element->get_pixel(x, y);
				on_composite(A, B, *O);
				++O;
			}
		}

		return output;
	}

	k3d::pixel::sample_type fix_alpha(const k3d::pixel& A, const k3d::pixel& B)
	{
		if(A.alpha == k3d::pixel::sample_traits::transparent() ||
		    B.alpha == k3d::pixel::sample_traits::transparent())
			return  k3d::pixel::sample_traits::transparent();

		return k3d::pixel::sample_traits::opaque();
	}

	virtual void on_value_change() = 0;
	virtual void on_composite(const k3d::pixel& A, const k3d::pixel& B, k3d::pixel& O) = 0;
};

} // namespace libk3dbitmap

#endif // !MODULE_BITMAP_BITMAP_COMPOSITE_SIMPLE_H


