// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "basic_bitmap_modifier.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_threshold

class bitmap_threshold :
	public basic_bitmap_modifier
{
	typedef basic_bitmap_modifier base;

public:
	bitmap_threshold(k3d::idocument& Document) :
		base(Document),
		m_red_threshold(init_owner(*this) + init_name("red_threshold") + init_label(_("Red threshold")) + init_description(_("Clamp Red channel to threshold")) + init_value(0.0)),
		m_green_threshold(init_owner(*this) + init_name("geen_threshold") + init_label(_("Green threshold")) + init_description(_("Clamp Green channel to threshold")) + init_value(0.0)),
		m_blue_threshold(init_owner(*this) + init_name("blue_threshold") + init_label(_("Blue threshold")) + init_description(_("Clamp Blue channel to threshold")) + init_value(0.0)),
		m_alpha_threshold(init_owner(*this) + init_name("alpha_threshold") + init_label(_("Alpha threshold")) + init_description(_("Clamp Alpha channel to threshold")) + init_value(0.0))
	{
		m_red_threshold.changed_signal().connect(sigc::mem_fun(*this, &bitmap_threshold::on_value_change));
		m_green_threshold.changed_signal().connect(sigc::mem_fun(*this, &bitmap_threshold::on_value_change));
		m_blue_threshold.changed_signal().connect(sigc::mem_fun(*this, &bitmap_threshold::on_value_change));
		m_alpha_threshold.changed_signal().connect(sigc::mem_fun(*this, &bitmap_threshold::on_value_change));
	}

	void on_value_change()
	{
		m_red_cache   = m_red_threshold.value();
		m_green_cache = m_green_threshold.value();
		m_blue_cache  = m_blue_threshold.value();
		m_alpha_cache = m_alpha_threshold.value();
		on_reset_bitmap();
	}

        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		double I_red   = k3d::color_traits<double>::convert(I.red);
		double I_green = k3d::color_traits<double>::convert(I.green);
		double I_blue  = k3d::color_traits<double>::convert(I.blue);
		double I_alpha = k3d::color_traits<double>::convert(I.alpha);

		// Matte divide ...
		if (I_alpha != 0)
		{
			I_red   = I_red   / I_alpha;
			I_green = I_green / I_alpha;
			I_blue  = I_blue  / I_alpha;
		}

		// Color operation ...
		const double O_red   = std::max(I_red,   m_red_cache);
		const double O_green = std::max(I_green, m_green_cache);
		const double O_blue  = std::max(I_blue,  m_blue_cache);
		const double O_alpha = std::max(I_alpha, m_alpha_cache);

		// Matte multiply ...
		O.red   = k3d::bitmap::pixel_type::sample_traits::convert(O_red   * I_alpha);
		O.green = k3d::bitmap::pixel_type::sample_traits::convert(O_green * I_alpha);
		O.blue  = k3d::bitmap::pixel_type::sample_traits::convert(O_blue  * I_alpha);
		O.alpha = O_alpha;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_threshold>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xcc8d625b, 0x970d4293, 0xa304d906, 0x81201d28),
				"BitmapThreshold",
				_("Cut off chanels at a certain value"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_red_threshold;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_green_threshold;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_blue_threshold;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_alpha_threshold;

	double m_red_cache;
	double m_green_cache;
	double m_blue_cache;
	double m_alpha_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_threshold_factory

k3d::iplugin_factory& bitmap_threshold_factory()
{
	return bitmap_threshold::get_factory();
}

} // namespace libk3dbitmap


