// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// cylindrical_wave_points

class cylindrical_wave_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	cylindrical_wave_points(k3d::idocument& Document) :
		base(Document),
		m_along(init_owner(*this) + init_name("along") + init_label(_("Along")) + init_description(_("Wave points along given axis")) + init_value(k3d::X) + init_enumeration(k3d::axis_values())),
		m_amplitude(init_owner(*this) + init_name("amplitude") + init_label(_("Amplitude")) + init_description(_("Wave amplitude")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_wavelength(init_owner(*this) + init_name("wavelength") + init_label(_("Wavelength")) + init_description(_("Wavelength")) + init_value(10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_phase(init_owner(*this) + init_name("phase") + init_label(_("Phase")) + init_description(_("Wave phase")) + init_value(0.0) + init_precision(2) + init_step_increment(k3d::radians(1.0)) + init_units(typeid(k3d::measurement::angle)))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reset_geometry));

		m_along.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reshape_geometry));
		m_amplitude.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reshape_geometry));
		m_wavelength.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reshape_geometry));
		m_phase.changed_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_reshape_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &cylindrical_wave_points::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.internal_value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::axis along = m_along.value();
		const double amplitude = m_amplitude.value();
		const double wavelength = m_wavelength.value();
		const double phase = m_phase.value();

		// Filter-out infinite frequencies ...
		if(0 == wavelength)
			return;

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				k3d::vector3 position = Source.points[i]->position;

				const double wave_position = phase + (k3d::pi_times_2() * position[along] / wavelength);
				const double offset = amplitude * sin(wave_position);

				const k3d::vector3 direction((k3d::X != along) * position[0], (k3d::Y != along) * position[1], (k3d::Z != along) * position[2]);
				if(0 == direction.Length2())
					continue;

				Target.points[i]->position = k3d::mix(Source.points[i]->position, position + offset * k3d::normalize(direction), Target.points[i]->selection_weight);
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<cylindrical_wave_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xfb3c4caf, 0xec534256, 0x96705718, 0xf6e5ec41),
				"CylindricalWavePoints",
				_("Applies a wave function to points along an axis"),
				"Deformation",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::axis, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_along;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_amplitude;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_wavelength;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_phase;
};

/////////////////////////////////////////////////////////////////////////////
// cylindrical_wave_points_factory

k3d::iplugin_factory& cylindrical_wave_points_factory()
{
	return cylindrical_wave_points::get_factory();
}

} // namespace libk3ddeformation


