// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/imesh_storage.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/serialization.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/string_modifiers.h>

using namespace k3d::xml;

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// frozen_mesh_implementation

class frozen_mesh_implementation :
	public k3d::mesh_modifier<k3d::persistent<k3d::node> >,
	public k3d::imesh_storage
{
	typedef k3d::mesh_modifier<k3d::persistent<k3d::node> > base;

public:
	frozen_mesh_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &frozen_mesh_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &frozen_mesh_implementation::on_create_geometry));
	}

	void reset_mesh(k3d::mesh* const Mesh)
	{
		m_output_mesh.reset(Mesh);
	}

	void on_reset_geometry()
	{
//		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input data, we're done ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create mutable output geometry ...
		k3d::mesh* const output = new k3d::mesh();

		// Make a copy of the input geometry ...
		k3d::deep_copy(*input, *output);

		return output;
	}

	void load_legacy_data(element& Element, k3d::mesh& Mesh)
	{
		// Load points ...
		std::map<unsigned long, k3d::point*> point_map;
		element* const xml_points = find_element(Element, "points");
		if(xml_points)
			{
				for(element::elements_t::const_iterator xml_point = xml_points->children.begin(); xml_point != xml_points->children.end(); ++xml_point)
					{
						if(xml_point->name != "point")
							continue;

						const unsigned long id = attribute_value<unsigned long>(*xml_point, "id", 0);
						return_if_fail(id);

						Mesh.points.push_back(new k3d::point(attribute_value<k3d::vector3>(*xml_point, "position", k3d::vector3(0, 0, 0))));
						point_map[id] = Mesh.points.back();
					}
			}

		// Load polygons into a single, non-solid polyhedron ...
		element* const xml_paths = find_element(Element, "paths");
		if(xml_paths)
			{
				k3d::polyhedron* const polyhedron = new k3d::polyhedron();
				Mesh.polyhedra.push_back(polyhedron);

				const std::string xml_rendermode = attribute_text(Element.safe_element("variables").safe_element(element("variable", attribute("name", "rendermode"))), "value", std::string("normal"));
				polyhedron->type = ("subdivision" == xml_rendermode) ? k3d::polyhedron::CATMULL_CLARK_SUBDIVISION_MESH : k3d::polyhedron::POLYGONS;

				for(element::elements_t::iterator xml_path = xml_paths->children.begin(); xml_path != xml_paths->children.end(); ++xml_path)
					{
						if(xml_path->name != "path")
							continue;

						k3d::imaterial* const material = reinterpret_cast<k3d::imaterial*>(attribute_value<unsigned long>(*xml_path, "material", 0UL));

						element* const xml_pathpoints = find_element(*xml_path, "pathpoints");
						if(xml_pathpoints)
							{
								std::vector<k3d::split_edge*> new_edges;
								for(element::elements_t::iterator xml_pathpoint = xml_pathpoints->children.begin(); xml_pathpoint != xml_pathpoints->children.end(); ++xml_pathpoint)
									{
										if(xml_pathpoint->name != "pathpoint")
											continue;

										const unsigned long point_id = attribute_value<unsigned long>(*xml_pathpoint, "pointid", 0UL);
										return_if_fail(point_map.count(point_id));

										new_edges.push_back(new k3d::split_edge(point_map[point_id]));
									}

								k3d::loop_edges(new_edges.begin(), new_edges.end());

								polyhedron->faces.push_back(new k3d::face(new_edges.front(), material));
							}
					}
			}
	}

	void save(element& Element, const k3d::ipersistent::save_context& Context)
	{
		base::save(Element, Context);

		// If we aren't storing any data, we're done ...
		if(m_output_mesh.empty())
			return;

		// Save our stored mesh data ...
		const k3d::mesh& mesh = *m_output_mesh.internal_value();
		element& xml_mesh = Element.append(element("mesh"));

		k3d::save_mesh(mesh, xml_mesh, Context);
	}

	void load(element& Element, const k3d::ipersistent::load_context& Context)
	{
		base::load(Element, Context);

		// Create a new mesh ...
		k3d::mesh* const mesh = new k3d::mesh();
		m_output_mesh.reset(mesh);

		// Load legacy data from the original Mesh object
		load_legacy_data(Element, *mesh);

		// Load the stored mesh data ...
		element* const xml_mesh = find_element(Element, "mesh");
		if(!xml_mesh)
			return;

		load_mesh(*mesh, *xml_mesh, Context);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<frozen_mesh_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
			k3d::classes::FrozenMesh(),
			"FrozenMesh",
			_("Freezes its input for manual editing"),
			"Mesh",
			k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// frozen_mesh_factory

k3d::iplugin_factory& frozen_mesh_factory()
{
	return frozen_mesh_implementation::get_factory();
}

} // namespace libk3dmesh


