// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/basic_math.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/renderman.h>

#include <boost/random/lagged_fibonacci.hpp>
#include <iterator>

namespace libk3dprimitives
{

/////////////////////////////////////////////////////////////////////////////
// lightning_implementation

class lightning_implementation :
	public k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > >
{
	typedef k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > > base;

public:
	lightning_implementation(k3d::idocument& Document) :
		base(Document),
		m_segments(init_owner(*this) + init_name("segments") + init_label(_("Segments")) + init_description(_("Lightning segments")) + init_value(20) + init_constraint(constraint::minimum(0L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_width(init_owner(*this) + init_name("width") + init_label(_("Width")) + init_description(_("Lightning width")) + init_value(0.2) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::distance))),
		m_radius(init_owner(*this) + init_name("radius") + init_label(_("Radius")) + init_description(_("Radius")) + init_value(0.2) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::distance)))
	{
		m_material.changed_signal().connect(sigc::mem_fun(*this, &lightning_implementation::on_reset_geometry));

		m_segments.changed_signal().connect(sigc::mem_fun(*this, &lightning_implementation::on_reset_geometry));
		m_width.changed_signal().connect(sigc::mem_fun(*this, &lightning_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(sigc::mem_fun(*this, &lightning_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &lightning_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		k3d::mesh* const mesh = new k3d::mesh();

		const unsigned long segments = m_segments.value();
		const double width = m_width.value();
		const double radius= m_radius.value();

		k3d::linear_curve_group* const group = new k3d::linear_curve_group();
		group->material = m_material.value();
		group->constant_data["constantwidth"] = static_cast<k3d::ri::real>(width);

		k3d::linear_curve* const curve = new k3d::linear_curve();

		boost::lagged_fibonacci607 random;
		mesh->points.resize(segments+1);
		for(unsigned long i = 0; i <= segments; ++i)
			{
				mesh->points[i] = new k3d::point(radius * (random() - 0.5), k3d::mix(0.5, -0.5, static_cast<double>(i) / static_cast<double>(segments)), radius * (random() - 0.5));
			}
		curve->control_points = mesh->points;

		group->curves.push_back(curve);
		mesh->linear_curve_groups.push_back(group);

		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<lightning_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x7bc13682, 0xfb5f44ad, 0x97a524ad, 0x5f3d5c6f),
			"Lightning",
			_("Simulates a lightning stroke using curves"),
			"Curves",
			k3d::iplugin_factory::DEPRECATED);

		return factory;
	}

private:
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_segments;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_width;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_radius;
};

/////////////////////////////////////////////////////////////////////////////
// lightning_factory

k3d::iplugin_factory& lightning_factory()
{
	return lightning_implementation::get_factory();
}

} // namespace libk3dprimitives


