// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements a scripting object model in Python
		\author Anders Dahnielson (anders@dahnielson.com)
		\author Romain Behar (romainbehar@yahoo.com)
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "object_model.h"

#include <k3dsdk/algebra.h>
#include <k3dsdk/application.h>
#include <k3dsdk/auto_ptr.h>
#include <k3dsdk/bitmap.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/command_node.h>
#include <k3dsdk/command_tree.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/iapplication_plugin_factory.h>
#include <k3dsdk/icommand_tree.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_plugin_factory.h>
#include <k3dsdk/idocument_read_format.h>
#include <k3dsdk/idocument_write_format.h>
#include <k3dsdk/ienumeration_property.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/imeasurement_property.h>
#include <k3dsdk/imesh_storage.h>
#include <k3dsdk/inode.h>
#include <k3dsdk/inode_collection.h>
#include <k3dsdk/iplugin_factory.h>
#include <k3dsdk/iproperty_collection.h>
#include <k3dsdk/irenderman.h>
#include <k3dsdk/iselectable.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/log.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_selection.h>
#include <k3dsdk/nodes.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/property.h>
#include <k3dsdk/renderman_properties.h>
#include <k3dsdk/result.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/tokens.h>
#include <k3dsdk/user_properties.h>
#include <k3dsdk/vectors.h>

#include <boost/filesystem/path.hpp>
#include <boost/static_assert.hpp>

#include <iostream>

namespace libk3dpython
{

/// Well known Python values
#define PYVAL_NONE	Py_BuildValue("")
#define PYVAL_FALSE	Py_BuildValue("i", 0)
#define PYVAL_TRUE	Py_BuildValue("i", 1)

template<typename T>
T python_cast(PyObject* Object)
{
//	BOOST_STATIC_ASSERT(sizeof(T) == 0);
	assert_not_reached();
	return T();
}

// Using macros to do this is a guilty pleasure ;)

/// Wraps a Python object around a K-3D object by reference
#define DECLARE_OBJECT_REF_WRAPPER(name, label, description, k3d_type) \
	PyObject* name##_new(PyTypeObject*, PyObject*, PyObject*); \
	PyObject* name##_getattr(PyObject*, char*); \
	int name##_setattr(PyObject*, char*, PyObject*); \
	extern PyTypeObject name##_type; \
	struct name \
	{ \
		PyObject_HEAD \
		k3d_type* object; \
	}; \
	PyObject* python_wrap(k3d_type* Object) \
	{ \
		return_val_if_fail(Object, 0); \
		name* const result = PyObject_New(name, &name##_type); \
		return_val_if_fail(result, 0); \
		result->object = Object; \
		return reinterpret_cast<PyObject*>(result); \
	} \
	template<> \
	k3d_type* python_cast(PyObject* Object) \
	{ \
		if(!Object) \
			return 0; \
		return_val_if_fail(PyObject_TypeCheck(Object, &name##_type), 0); \
		return reinterpret_cast<name*>(Object)->object; \
	} \
	PyObject* name##_new(PyTypeObject* type, PyObject* args, PyObject* kwds) \
	{ \
		PyObject* self = type->tp_alloc(type, 0); \
		return_val_if_fail(self, 0); \
		reinterpret_cast<name*>(self)->object = 0; \
		return self; \
	} \
	int name##_compare(PyObject* LHS, PyObject* RHS) \
	{ \
		k3d_type* const lhs = python_cast<k3d_type*>(LHS); \
		k3d_type* const rhs = python_cast<k3d_type*>(RHS); \
		if(lhs < rhs) \
			return -1; \
		if(lhs > rhs) \
			return 1; \
		return 0; \
	} \
	long name##_hash(PyObject* Object) \
	{ \
		k3d_type* const object = python_cast<k3d_type*>(Object); \
		return_val_if_fail(object, 0); \
		return reinterpret_cast<long>(object); \
	} \
	PyObject* python_wrap(k3d_type& Object) \
	{ \
		name* const result = PyObject_New(name, &name##_type); \
		return_val_if_fail(result, 0); \
		result->object = &Object; \
		return reinterpret_cast<PyObject*>(result); \
	} \
	PyTypeObject name##_type = \
	{ \
		PyObject_HEAD_INIT(NULL) \
		0, \
		label, /* tp_name */ \
		sizeof(name), /* tp_basicsize */ \
		0, /* tp_itemsize */ \
		0, /* tp_dealloc */ \
		0, /* tp_print */ \
		name##_getattr, /* tp_getattr */ \
		name##_setattr, /* tp_setattr */ \
		name##_compare, /* tp_compare */ \
		0, /* tp_repr */ \
		0, /* tp_as_number */ \
		0, /* tp_as_sequence */ \
		0, /* tp_as_mapping */ \
		name##_hash, /* tp_hash */ \
		0, \
		0, \
		0, \
		0, \
		0, \
		Py_TPFLAGS_DEFAULT & ~Py_TPFLAGS_HAVE_RICHCOMPARE, /* tp_flags */ \
		description, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		name##_new, \
	};

/// Wraps a resizable STL container with a new Python type
#define DECLARE_CONTAINER_WRAPPER(name, label, description, container_type) \
	PyObject* name##_new(PyTypeObject*, PyObject*, PyObject*); \
	extern PyTypeObject name##_type; \
	extern PySequenceMethods name##_sequence_methods; \
	struct name \
	{ \
		PyObject_HEAD \
		container_type* container; \
		container_type::iterator iterator; \
	}; \
	PyObject* name##_new(PyTypeObject* type, PyObject* args, PyObject* kwds) \
	{ \
		PyObject* self = type->tp_alloc(type, 0); \
		return_val_if_fail(self, 0); \
		reinterpret_cast<name*>(self)->container = 0; \
		return self; \
	} \
	PyObject* python_wrap(container_type& Container) \
	{ \
		name* const result = PyObject_New(name, &name##_type); \
		return_val_if_fail(result, 0); \
		result->container = &Container; \
		result->iterator = Container.end(); \
		return reinterpret_cast<PyObject*>(result); \
	} \
	template<> \
	container_type* python_cast(PyObject* Object) \
	{ \
		if(!Object) \
			return 0; \
		return_val_if_fail(PyObject_TypeCheck(Object, &name##_type), 0); \
		return reinterpret_cast<name*>(Object)->container; \
	} \
	int name##_length(PyObject* Object) \
	{ \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, -1); \
		return container->size(); \
	} \
	PyObject* name##_item(PyObject* Object, int Index) \
	{ \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, 0); \
		return_val_if_fail(0 <= Index, 0); \
		if(static_cast<unsigned int>(Index) >= container->size()) \
			container->resize(Index + 1); \
		return python_wrap(container->at(Index)); \
	} \
	int name##_assign_item(PyObject* Object, int Index, PyObject* Value) \
	{ \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, -1); \
		return_val_if_fail(0 <= Index, -1); \
		if(static_cast<unsigned int>(Index) >= container->size()) \
			container->resize(Index + 1); \
		if(Value) \
		{ \
			container->at(Index) = python_cast<container_type::value_type>(Value); \
		} \
		else \
		{ \
			container_type::iterator i = container->begin(); \
			std::advance(i, Index); \
			container->erase(i); \
		} \
		return 0; \
	} \
	PyObject* name##_getiter(PyObject* Object) \
	{ \
		name* const object = reinterpret_cast<name*>(Object); \
		return_val_if_fail(object, 0); \
		return_val_if_fail(object->container, 0); \
		object->iterator = object->container->begin(); \
		Py_INCREF(Object); \
		return Object; \
	} \
	PyObject* name##_iternext(PyObject* Object) \
	{ \
		name* const object = reinterpret_cast<name*>(Object); \
		return_val_if_fail(object, 0); \
		return_val_if_fail(object->container, 0); \
		if(object->iterator != object->container->end()) \
			return python_wrap(*object->iterator++); \
		PyErr_SetNone(PyExc_StopIteration); \
		return 0; \
	} \
	PyObject* name##_append(PyObject* Object, PyObject* Value) \
	{ \
		name* const object = reinterpret_cast<name*>(Object); \
		return_val_if_fail(object, 0); \
		return_val_if_fail(object->container, 0); \
		PyObject* value = 0; \
		if(!PyArg_ParseTuple(Value, "O", &value)) \
			return 0; \
		object->container->push_back(python_cast<container_type::value_type>(value)); \
		return PYVAL_NONE; \
	} \
	PyMethodDef name##_methods[] = \
	{ \
		{"append", name##_append, METH_VARARGS, "Append an object."}, \
		{NULL, NULL, 0, NULL} \
	}; \
	PyObject* name##_getattr(PyObject* Object, char* Name) \
	{ \
		return Py_FindMethod(name##_methods, Object, Name); \
	} \
	PySequenceMethods name##_sequence_methods = \
	{ \
		name##_length, /* sq_length */ \
		0, /* sq_concat */ \
		0, /* sq_repeat */ \
		name##_item, /* sq_item */ \
		0, /* sq_slice */ \
		name##_assign_item, /* sq_ass_item */ \
		0, /* sq_ass_slice */ \
		0, /* sq_containers */ \
		0, /* sq_inplace_concat */ \
		0, /* sq_inplace_repeat */ \
	}; \
	PyTypeObject name##_type = \
	{ \
		PyObject_HEAD_INIT(NULL) \
		0, \
		label, /* tp_name */ \
		sizeof(name), /* tp_basicsize */ \
		0, /* tp_itemsize */ \
		0, /* tp_dealloc */ \
		0, /* tp_print */ \
		&name##_getattr, /* tp_getattr */ \
		0, /* tp_setattr */ \
		0, /* tp_compare */ \
		0, /* tp_repr */ \
		0, /* tp_as_number */ \
		&name##_sequence_methods, /* tp_as_sequence */ \
		0, /* tp_as_mapping */ \
		0, /* tp_hash */ \
		0, /* tp_call */ \
		0, /* tp_str */ \
		0, /* tp_getattro */ \
		0, /* tp_setattro */ \
		0, /* tp_as_buffer */ \
		Py_TPFLAGS_DEFAULT, /* tp_flags */ \
		description, /* tp_doc */ \
		0, /* tp_traverse */ \
		0, /* tp_clear */ \
		0, /* tp_richcompare */ \
		0, /* tp_weaklistoffset */ \
		name##_getiter, /* tp_iter */ \
		name##_iternext, /* tp_iternext */ \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		name##_new, \
	};

/// Wraps a fixed-size STL-compatible container with a new Python type
#define DECLARE_FIXED_CONTAINER_WRAPPER(name, label, description, container_type) \
	PyObject* name##_new(PyTypeObject*, PyObject*, PyObject*); \
	extern PyTypeObject name##_type; \
	extern PySequenceMethods name##_sequence_methods; \
	struct name \
	{ \
		PyObject_HEAD \
		container_type* container; \
		container_type::iterator iterator; \
	}; \
	PyObject* name##_new(PyTypeObject* type, PyObject* args, PyObject* kwds) \
	{ \
		PyObject* self = type->tp_alloc(type, 0); \
		return_val_if_fail(self, 0); \
		reinterpret_cast<name*>(self)->container = 0; \
		return self; \
	} \
	PyObject* python_wrap(container_type& Container) \
	{ \
		name* const result = PyObject_New(name, &name##_type); \
		return_val_if_fail(result, 0); \
		result->container = &Container; \
		return reinterpret_cast<PyObject*>(result); \
	} \
	template<> \
	container_type* python_cast(PyObject* Object) \
	{ \
		if(!Object) \
			return 0; \
		return_val_if_fail(PyObject_TypeCheck(Object, &name##_type), 0); \
		return reinterpret_cast<name*>(Object)->container; \
	} \
	int name##_length(PyObject* Object) \
	{ \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, -1); \
		return container->size(); \
	} \
	PyObject* name##_item(PyObject* Object, int Index) \
	{ \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, 0); \
		return_val_if_fail(0 <= Index && static_cast<unsigned int>(Index) < container->size(), 0); \
		return python_wrap(container->at(Index)); \
	} \
	int name##_assign_item(PyObject* Object, int Index, PyObject* Value) \
	{ \
		return_val_if_fail(Value, -1); \
		container_type* const container = python_cast<container_type*>(Object); \
		return_val_if_fail(container, -1); \
		return_val_if_fail(0 <= Index, -1); \
		return_val_if_fail(0 <= Index && static_cast<unsigned int>(Index) < container->size(), -1); \
		container->at(Index) = python_cast<container_type::value_type>(Value); \
		return 0; \
	} \
	PyObject* name##_getiter(PyObject* Object) \
	{ \
		name* const object = reinterpret_cast<name*>(Object); \
		return_val_if_fail(object, 0); \
		return_val_if_fail(object->container, 0); \
		object->iterator = object->container->begin(); \
		Py_INCREF(Object); \
		return Object; \
	} \
	PyObject* name##_iternext(PyObject* Object) \
	{ \
		name* const object = reinterpret_cast<name*>(Object); \
		return_val_if_fail(object, 0); \
		return_val_if_fail(object->container, 0); \
		if(object->iterator != object->container->end()) \
			return python_wrap(*object->iterator++); \
		PyErr_SetNone(PyExc_StopIteration); \
		return 0; \
	} \
	PySequenceMethods name##_sequence_methods = \
	{ \
		name##_length, /* sq_length */ \
		0, /* sq_concat */ \
		0, /* sq_repeat */ \
		name##_item, /* sq_item */ \
		0, /* sq_slice */ \
		name##_assign_item, /* sq_ass_item */ \
		0, /* sq_ass_slice */ \
		0, /* sq_containers */ \
		0, /* sq_inplace_concat */ \
		0, /* sq_inplace_repeat */ \
	}; \
	PyTypeObject name##_type = \
	{ \
		PyObject_HEAD_INIT(NULL) \
		0, \
		label, /* tp_name */ \
		sizeof(name), /* tp_basicsize */ \
		0, /* tp_itemsize */ \
		0, /* tp_dealloc */ \
		0, /* tp_print */ \
		0, /* tp_getattr */ \
		0, /* tp_setattr */ \
		0, /* tp_compare */ \
		0, /* tp_repr */ \
		0, /* tp_as_number */ \
		&name##_sequence_methods, /* tp_as_sequence */ \
		0, /* tp_as_mapping */ \
		0, /* tp_hash */ \
		0, /* tp_call */ \
		0, /* tp_str */ \
		0, /* tp_getattro */ \
		0, /* tp_setattro */ \
		0, /* tp_as_buffer */ \
		Py_TPFLAGS_DEFAULT, /* tp_flags */ \
		description, /* tp_doc */ \
		0, /* tp_traverse */ \
		0, /* tp_clear */ \
		0, /* tp_richcompare */ \
		0, /* tp_weaklistoffset */ \
		name##_getiter, /* tp_iter */ \
		name##_iternext, /* tp_iternext */ \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		0, \
		name##_new, \
	};

DECLARE_OBJECT_REF_WRAPPER(k3d_bicubic_patch, "k3d.bicubic_patch", "K-3D Bicubic Patch", k3d::bicubic_patch);
DECLARE_OBJECT_REF_WRAPPER(k3d_bilinear_patch, "k3d.bilinear_patch", "K-3D Bilinear Patch", k3d::bilinear_patch);
DECLARE_OBJECT_REF_WRAPPER(k3d_blobby, "k3d.blobby", "K-3D Blobby", k3d::blobby);
DECLARE_OBJECT_REF_WRAPPER(k3d_blobby_opcode, "k3d.blobby_opcode", "K-3D Blobby Opcode", k3d::blobby::opcode);
DECLARE_OBJECT_REF_WRAPPER(k3d_cubic_curve, "k3d.cubic_curve", "K-3D Cubic Curve", k3d::cubic_curve);
DECLARE_OBJECT_REF_WRAPPER(k3d_cubic_curve_group, "k3d.cubic_curve_group", "K-3D Cubic Curve Group", k3d::cubic_curve_group);
DECLARE_OBJECT_REF_WRAPPER(k3d_face, "k3d.face", "K-3D Face", k3d::face);
DECLARE_OBJECT_REF_WRAPPER(k3d_iplugin_factory, "k3d.plugin_factory", "K-3D Plugin Factory", k3d::iplugin_factory);
DECLARE_OBJECT_REF_WRAPPER(k3d_iproperty, "k3d.property", "K-3D Property", k3d::iproperty);
DECLARE_OBJECT_REF_WRAPPER(k3d_iunknown, "k3d.unknown", "K-3D Unknown", k3d::iunknown);
DECLARE_OBJECT_REF_WRAPPER(k3d_iuser_interface, "k3d.user_interface", "K-3D User Interface", k3d::iuser_interface);
DECLARE_OBJECT_REF_WRAPPER(k3d_linear_curve, "k3d.linear_curve", "K-3D Linear Curve", k3d::linear_curve);
DECLARE_OBJECT_REF_WRAPPER(k3d_linear_curve_group, "k3d.linear_curve_group", "K-3D Linear Curve Group", k3d::linear_curve_group);
DECLARE_OBJECT_REF_WRAPPER(k3d_mesh, "k3d.mesh", "K-3D Mesh", k3d::mesh);
DECLARE_OBJECT_REF_WRAPPER(k3d_parameters, "k3d.parameter", "K-3D Parameters", k3d::parameters_t);
DECLARE_OBJECT_REF_WRAPPER(k3d_point, "k3d.point", "K-3D Point", k3d::point);
DECLARE_OBJECT_REF_WRAPPER(k3d_point_group, "k3d.point_group", "K-3D Point Group", k3d::point_group);
DECLARE_OBJECT_REF_WRAPPER(k3d_polyhedron, "k3d.polyhedron", "K-3D Polyhedron", k3d::polyhedron);
DECLARE_OBJECT_REF_WRAPPER(k3d_split_edge, "k3d.split_edge", "K-3D Split Edge", k3d::split_edge);

DECLARE_CONTAINER_WRAPPER(k3d_bicubic_patch_collection, "k3d.bicubic_patch_collection", "K-3D Bicubic Patch Collection", k3d::mesh::bicubic_patches_t);
DECLARE_CONTAINER_WRAPPER(k3d_bilinear_patch_collection, "k3d.bilinear_patch_collection", "K-3D Bilinear Patch Collection", k3d::mesh::bilinear_patches_t);
DECLARE_CONTAINER_WRAPPER(k3d_blobby_collection, "k3d.blobby_collection", "K-3D Blobby Collection", k3d::mesh::blobbies_t);
DECLARE_CONTAINER_WRAPPER(k3d_cubic_curve_collection, "k3d.cubic_curve_collection", "K-3D Cubic Curve Collection", k3d::cubic_curve_group::curves_t);
DECLARE_CONTAINER_WRAPPER(k3d_cubic_curve_group_collection, "k3d.cubic_curve_group_collection", "K-3D Cubic Curve Group Collection", k3d::mesh::cubic_curve_groups_t);
DECLARE_CONTAINER_WRAPPER(k3d_face_collection, "k3d.face_collection", "K-3D Face Collection", k3d::polyhedron::faces_t);
DECLARE_CONTAINER_WRAPPER(k3d_linear_curve_collection, "k3d.linear_curve_collection", "K-3D Linear Curve Collection", k3d::linear_curve_group::curves_t);
DECLARE_CONTAINER_WRAPPER(k3d_linear_curve_group_collection, "k3d.linear_curve_group_collection", "K-3D Linear Curve Group Collection", k3d::mesh::linear_curve_groups_t);
DECLARE_CONTAINER_WRAPPER(k3d_parameters_collection, "k3d.parameters_collection", "K-3D Parameters Collection", std::vector<k3d::parameters_t>);
DECLARE_CONTAINER_WRAPPER(k3d_point_collection, "k3d.point_collection", "K-3D Point Collection", k3d::mesh::points_t);
DECLARE_CONTAINER_WRAPPER(k3d_point_group_collection, "k3d.point_group_collection", "K-3D Point Group Collection", k3d::mesh::point_groups_t);
DECLARE_CONTAINER_WRAPPER(k3d_polyhedron_collection, "k3d.polyhedron_collection", "K-3D Polyhedron Collection", k3d::mesh::polyhedra_t);

DECLARE_FIXED_CONTAINER_WRAPPER(k3d_bicubic_patch_control_point_collection, "k3d.bicubic_patch_control_point_collection", "K-3D Bicubic Patch Control Point Collection", k3d::bicubic_patch::control_points_t);
DECLARE_FIXED_CONTAINER_WRAPPER(k3d_bilinear_patch_control_point_collection, "k3d.bilinear_patch_control_point_collection", "K-3D Bilinear Patch Control Point Collection", k3d::bilinear_patch::control_points_t);

template<> k3d::icommand_node* python_cast(PyObject* Object);
template<> k3d::idocument* python_cast(PyObject* Object);
template<> k3d::imesh_storage* python_cast(PyObject* Object);
template<> k3d::inode* python_cast(PyObject* Object);
template<> k3d::iproperty_collection* python_cast(PyObject* Object);
template<> k3d::vector3* python_cast(PyObject* Object);
template<> k3d::normal3* python_cast(PyObject* Object);
template<> k3d::angle_axis* python_cast(PyObject* Object);
template<> k3d::euler_angles* python_cast(PyObject* Object);
template<> k3d::vector4* python_cast(PyObject* Object);
template<> k3d::matrix4* python_cast(PyObject* Object);
template<> k3d::color* python_cast(PyObject* Object);
template<> k3d::bitmap* python_cast(PyObject* Object);

template<>
double python_cast(PyObject* Object)
{
	return PyFloat_AsDouble(Object);
}

PyObject* python_wrap(const k3d::vector3&);
PyObject* python_wrap(const k3d::normal3&);
PyObject* python_wrap(const k3d::angle_axis&);
PyObject* python_wrap(const k3d::euler_angles&);
PyObject* python_wrap(const k3d::vector4&);
PyObject* python_wrap(const k3d::matrix4&);
PyObject* python_wrap(const k3d::color&);
PyObject* python_wrap(k3d::bitmap&);
PyObject* python_wrap(k3d::bitmap*);


PyObject* python_wrap(const bool Value)
{
	return Value ? PYVAL_TRUE : PYVAL_FALSE;
}

PyObject* python_wrap(const unsigned long Value)
{
	return Py_BuildValue("l", Value);
}

PyObject* python_wrap(const int Value)
{
	return Py_BuildValue("i", Value);
}

PyObject* python_wrap(const long Value)
{
	return Py_BuildValue("l", Value);
}

PyObject* python_wrap(const half Value)
{
	return Py_BuildValue("d", static_cast<double>(Value));
}

PyObject* python_wrap(const double Value)
{
	return Py_BuildValue("d", Value);
}

PyObject* python_wrap(const char* const Value)
{
	return Py_BuildValue("s", Value);
}

PyObject* python_wrap(const std::string& Value)
{
	return Py_BuildValue("s", const_cast<char*>(Value.c_str()));
}

PyObject* python_wrap(const boost::filesystem::path& Value)
{
	return Py_BuildValue("s", const_cast<char*>(Value.native_file_string().c_str()));
}

PyObject* python_wrap(const k3d::vector2& Value)
{
	return Py_BuildValue("(dd)", Value[0], Value[1]);
}

/////////////////////////////////////////////////////////////////////////////
// units_to_string

/// Returns a string representation of a C++ measurement units type
PyObject* units_to_string(const std::type_info& Units)
{
	if(Units == typeid(void))
		return python_wrap("scalar");

	if(Units == typeid(k3d::measurement::angle))
		return python_wrap("angle");

	if(Units == typeid(k3d::measurement::area))
		return python_wrap("area");

	if(Units == typeid(k3d::measurement::distance))
		return python_wrap("distance");

	if(Units == typeid(k3d::measurement::force))
		return python_wrap("force");

	if(Units == typeid(k3d::measurement::mass))
		return python_wrap("mass");

	if(Units == typeid(k3d::measurement::pressure))
		return python_wrap("pressure");

	if(Units == typeid(k3d::measurement::time))
		return python_wrap("time");

	if(Units == typeid(k3d::measurement::volume))
		return python_wrap("volume");

	return python_wrap("unknown");
}

/////////////////////////////////////////////////////////////////////////////
// type_to_string

/// Returns a string representation of a C++ type
PyObject* type_to_string(const std::type_info& Type)
{
	if(Type == typeid(bool))
		return python_wrap("boolean");

	if(Type == typeid(double))
		return python_wrap("double");

	if(Type == typeid(std::string))
		return python_wrap("string");

	if(Type == typeid(boost::filesystem::path))
		return python_wrap("path");

	if(Type == typeid(k3d::color))
		return python_wrap("color");

	if(Type == typeid(k3d::vector2))
		return python_wrap("vector2");

	if(Type == typeid(k3d::vector3))
		return python_wrap("vector3");

	if(Type == typeid(k3d::normal3))
		return python_wrap("normal3");

	if(Type == typeid(k3d::vector4))
		return python_wrap("vector4");

	if(Type == typeid(k3d::angle_axis))
		return python_wrap("angle_axis");

	if(Type == typeid(k3d::euler_angles))
		return python_wrap("euler_angles");

	if(Type == typeid(k3d::matrix4))
		return python_wrap("matrix4");

	if(Type == typeid(unsigned long))
		return python_wrap("unsigned long");

	if(Type == typeid(int))
		return python_wrap("int");

	if(Type == typeid(long))
		return python_wrap("long");

	if(Type == typeid(k3d::bounding_box3))
		return python_wrap("bounding_box");

	if(Type == typeid(k3d::mesh*))
		return python_wrap("mesh");

	if(Type == typeid(k3d::inode*))
		return python_wrap("object");

	if(Type == typeid(k3d::bitmap*))
		return python_wrap("bitmap");

	if(Type == typeid(k3d::mesh_selection))
		return python_wrap("mesh_selection");

	k3d::log() << warning << k3d_file_reference << ": unknown type: [" << Type.name() << "]" << std::endl;

	return python_wrap("unknown");
}

/// Converts a boost::any object to a Python value
PyObject* any_to_python(const boost::any Value)
{
	// We put the likely types up front for efficiency ...
	const std::type_info& type = Value.type();

	if(type == typeid(bool))
		return python_wrap(boost::any_cast<bool>(Value));

	if(type == typeid(double))
		return python_wrap(boost::any_cast<double>(Value));

	if(type == typeid(std::string))
		return python_wrap(boost::any_cast<std::string>(Value));

	if(type == typeid(boost::filesystem::path))
		return python_wrap(boost::any_cast<boost::filesystem::path>(Value));

	if(type == typeid(k3d::color))
		return python_wrap(boost::any_cast<k3d::color>(Value));

	if(type == typeid(k3d::vector2))
		return python_wrap(boost::any_cast<k3d::vector2>(Value));

	if(type == typeid(k3d::vector3))
		return python_wrap(boost::any_cast<k3d::vector3>(Value));

	if(type == typeid(k3d::normal3))
		return python_wrap(boost::any_cast<k3d::normal3>(Value));

	if(type == typeid(k3d::vector4))
		return python_wrap(boost::any_cast<k3d::vector4>(Value));

	if(type == typeid(k3d::matrix4))
		return python_wrap(boost::any_cast<k3d::matrix4>(Value));

	if(type == typeid(k3d::angle_axis))
		return python_wrap(boost::any_cast<k3d::angle_axis>(Value));

	if(type == typeid(k3d::euler_angles))
		return python_wrap(boost::any_cast<k3d::euler_angles>(Value));

	if(type == typeid(int))
		return python_wrap(boost::any_cast<int>(Value));

	if(type == typeid(long))
		return python_wrap(boost::any_cast<long>(Value));

	if(type == typeid(unsigned long))
		return python_wrap(boost::any_cast<unsigned long>(Value));

	if(type == typeid(k3d::mesh*))
	{
		k3d::mesh* const mesh = boost::any_cast<k3d::mesh*>(Value);
		return mesh ? python_wrap(*mesh) : PYVAL_NONE;
	}

	if(type == typeid(k3d::bitmap*))
	{
		k3d::bitmap* const bitmap = boost::any_cast<k3d::bitmap*>(Value);
		return bitmap ? python_wrap(*bitmap) : PYVAL_NONE;
	}

	if(type == typeid(k3d::inode*))
	{
		k3d::inode* const object = boost::any_cast<k3d::inode*>(Value);
		return object ? python_wrap(*object) : PYVAL_NONE;
	}

	k3d::log() << error << k3d_file_reference << ": unrecognized type [" << type.name() << "]" << std::endl;
	return PYVAL_NONE;
}

/// Converts a Python value to a boost::any object, with the target type explicitly specified
boost::any python_to_any(PyObject* Value, const std::type_info& TargetType)
{
	if(TargetType == typeid(bool))
	{
		return boost::any(PyObject_IsTrue(Value) ? true : false);
	}

	if(TargetType == typeid(double))
	{
		return_val_if_fail(PyFloat_Check(Value), boost::any());
		return boost::any(PyFloat_AsDouble(Value));
	}

	if(TargetType == typeid(std::string))
	{
		return_val_if_fail(PyString_Check(Value), boost::any());
		return boost::any(std::string(PyString_AsString(Value)));
	}

	if(TargetType == typeid(boost::filesystem::path))
	{
		return_val_if_fail(PyString_Check(Value), boost::any());
		return boost::any(boost::filesystem::path(PyString_AsString(Value), boost::filesystem::native));
	}

	if(TargetType == typeid(k3d::color))
	{
		k3d::color* const value = python_cast<k3d::color*>(Value);
		if(value)
			return boost::any(*value);

		return_val_if_fail(PyTuple_Check(Value), boost::any());
		return_val_if_fail(3 == PyTuple_Size(Value), boost::any());

		return boost::any(k3d::color(
			PyFloat_AsDouble(PyTuple_GetItem(Value, 0)),
			PyFloat_AsDouble(PyTuple_GetItem(Value, 1)),
			PyFloat_AsDouble(PyTuple_GetItem(Value, 2))));
	}

	if(TargetType == typeid(k3d::vector3))
	{
		k3d::vector3* const value = python_cast<k3d::vector3*>(Value);
		if(value)
			return boost::any(*value);

		return_val_if_fail(PyTuple_Check(Value), boost::any());
		return_val_if_fail(3 == PyTuple_Size(Value), boost::any());

		return boost::any(k3d::vector3(
			PyFloat_AsDouble(PyTuple_GetItem(Value, 0)),
			PyFloat_AsDouble(PyTuple_GetItem(Value, 1)),
			PyFloat_AsDouble(PyTuple_GetItem(Value, 2))));
	}

	if(TargetType == typeid(k3d::matrix4))
	{
		k3d::matrix4* const value = python_cast<k3d::matrix4*>(Value);
		if(value)
			return boost::any(*value);

		if(PyTuple_Check(Value))
		{
			return_val_if_fail(16 == PyTuple_Size(Value), boost::any());

			return boost::any(k3d::matrix4(
				k3d::vector4(
					PyFloat_AsDouble(PyTuple_GetItem(Value, 0)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 1)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 2)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 3))),
				k3d::vector4(
					PyFloat_AsDouble(PyTuple_GetItem(Value, 4)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 5)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 6)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 7))),
				k3d::vector4(
					PyFloat_AsDouble(PyTuple_GetItem(Value, 8)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 9)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 10)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 11))),
				k3d::vector4(
					PyFloat_AsDouble(PyTuple_GetItem(Value, 12)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 13)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 14)),
					PyFloat_AsDouble(PyTuple_GetItem(Value, 15)))));
		}
		if(PyList_Check(Value))
		{
			return_val_if_fail(16 == PyList_Size(Value), boost::any());

			return boost::any(k3d::matrix4(
				k3d::vector4(
					PyFloat_AsDouble(PyList_GetItem(Value, 0)),
					PyFloat_AsDouble(PyList_GetItem(Value, 1)),
					PyFloat_AsDouble(PyList_GetItem(Value, 2)),
					PyFloat_AsDouble(PyList_GetItem(Value, 3))),
				k3d::vector4(
					PyFloat_AsDouble(PyList_GetItem(Value, 4)),
					PyFloat_AsDouble(PyList_GetItem(Value, 5)),
					PyFloat_AsDouble(PyList_GetItem(Value, 6)),
					PyFloat_AsDouble(PyList_GetItem(Value, 7))),
				k3d::vector4(
					PyFloat_AsDouble(PyList_GetItem(Value, 8)),
					PyFloat_AsDouble(PyList_GetItem(Value, 9)),
					PyFloat_AsDouble(PyList_GetItem(Value, 10)),
					PyFloat_AsDouble(PyList_GetItem(Value, 11))),
				k3d::vector4(
					PyFloat_AsDouble(PyList_GetItem(Value, 12)),
					PyFloat_AsDouble(PyList_GetItem(Value, 13)),
					PyFloat_AsDouble(PyList_GetItem(Value, 14)),
					PyFloat_AsDouble(PyList_GetItem(Value, 15)))));
		}
		assert_warning(0);
		return boost::any();
	}

	if(TargetType == typeid(k3d::angle_axis))
	{
		if(k3d::angle_axis* const value = python_cast<k3d::angle_axis*>(Value))
			return boost::any(*value);

		if(k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(Value))
			return boost::any(*value);

		return_val_if_fail(PyTuple_Check(Value), boost::any());
		const int length = PyTuple_Size(Value);

		if(3 == length)
		{
			k3d::euler_angles euler(0, 0, 0, k3d::euler_angles::ZXYstatic);

			euler.n[1] = PyFloat_AsDouble(PyTuple_GetItem(Value, 0));
			euler.n[2] = PyFloat_AsDouble(PyTuple_GetItem(Value, 1));
			euler.n[0] = PyFloat_AsDouble(PyTuple_GetItem(Value, 2));

			euler.n[0] = k3d::radians(euler.n[0]);
			euler.n[1] = k3d::radians(euler.n[1]);
			euler.n[2] = k3d::radians(euler.n[2]);

			return boost::any(k3d::angle_axis(euler));
		}

		if(4 == length)
		{
			k3d::angle_axis result;

			result.angle = PyFloat_AsDouble(PyTuple_GetItem(Value, 0));
			result.axis[0] = PyFloat_AsDouble(PyTuple_GetItem(Value, 1));
			result.axis[1] = PyFloat_AsDouble(PyTuple_GetItem(Value, 2));
			result.axis[2] = PyFloat_AsDouble(PyTuple_GetItem(Value, 3));

			result.angle = k3d::radians(result.angle);

			return boost::any(result);
		}

		return_val_if_fail(0, boost::any());
	}

	if(TargetType == typeid(int))
	{
		return_val_if_fail(PyInt_Check(Value), boost::any());
		return boost::any(static_cast<int>(PyInt_AsLong(Value)));
	}

	if(TargetType == typeid(long))
	{
		return_val_if_fail(PyLong_Check(Value), boost::any());
		return boost::any(static_cast<long>(PyLong_AsLong(Value)));
	}

	if(TargetType == typeid(unsigned long))
	{
		return_val_if_fail(PyLong_Check(Value), boost::any());
		return boost::any(static_cast<unsigned long>(PyLong_AsLong(Value)));
	}

	if(TargetType == typeid(k3d::inode*))
	{
		return boost::any(python_cast<k3d::inode*>(Value));
	}

	if(TargetType == typeid(k3d::bitmap*))
	{
		return boost::any(python_cast<k3d::bitmap*>(Value));
	}

	k3d::log() << error << k3d_file_reference << ": unrecognized type: [" << TargetType.name() << "]" << std::endl;
	return boost::any();
}

/////////////////////////////////////////////////////////////////////////////
// k3d::iapplication

PyObject* k3d_iapplication_exit(PyObject* self, PyObject* args)
{
	k3d::application().exit();
	return PYVAL_NONE;
}

PyObject* k3d_iapplication_new_document(PyObject* self, PyObject* args)
{
	if(k3d::idocument* const document = k3d::application().create_document())
		return python_wrap(*document);

	return 0;
}

PyObject* k3d_iapplication_open_document(PyObject* self, PyObject* args)
{
	const char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;
	const boost::filesystem::path document_path(string1, boost::filesystem::native);

	k3d::auto_ptr<k3d::idocument_read_format> filter(k3d::create_plugin<k3d::idocument_read_format>(k3d::classes::DocumentReader()));
	return_val_if_fail(filter.get(), 0);
	k3d::idocument* const document = k3d::application().create_document();
	return_val_if_fail(document, 0);
	return_val_if_fail(filter->read_file(*document, document_path), 0);

	return python_wrap(*document);
}

PyObject* k3d_iapplication_close_document(PyObject* self, PyObject* args)
{
	PyObject* object;
	if(!PyArg_ParseTuple(args, "O", &object))
		return 0;

	k3d::idocument* document = python_cast<k3d::idocument*>(object);
	return_val_if_fail(document, 0);

	k3d::application().close_document(*document);

	return PYVAL_TRUE;
}

PyObject* k3d_iapplication_get_command_node(PyObject* self, PyObject* args)
{
	char *string1;
	if(!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	const std::string nodepath(string1);
	if(nodepath.empty())
	{
		k3d::log() << error << "Empty command node path";
		return 0;
	}

	k3d::icommand_node* node = k3d::command_node::lookup(nodepath);
	if(!node)
	{
		k3d::log() << error << "Could not find command node [" << nodepath << "]" << std::endl;
		return 0;
	}

	return python_wrap(*node);
}

PyMethodDef k3d_iapplication_methods[] =
{
	{"exit", k3d_iapplication_exit, METH_VARARGS, "Requests that K-3D be closed."},
	{"new_document", k3d_iapplication_new_document, METH_VARARGS, "Creates a new K-3D document."},
	{"open_document", k3d_iapplication_open_document, METH_VARARGS, "Opens a K-3D document."},
	{"close_document", k3d_iapplication_close_document, METH_VARARGS, "Closes an open K-3D document."},
	{"get_command_node", k3d_iapplication_get_command_node, METH_VARARGS, "Return a command-node."},
	{NULL, NULL, 0, NULL}
};

/////////////////////////////////////////////////////////////////////////////
// k3d_icommand_node

PyObject* k3d_icommand_node_execute_command(PyObject* self, PyObject* args)
{
	k3d::icommand_node* const command_node = python_cast<k3d::icommand_node*>(self);
	return_val_if_fail(command_node, 0);

	char* string1;
	char* string2;
	if(!PyArg_ParseTuple(args, "ss", &string1, &string2))
		return 0;

	const std::string command(string1);
	const std::string arguments(string2);
	return_val_if_fail(!command.empty(), 0);

	if(command_node->execute_command(command, arguments))
		return PYVAL_TRUE;

	return 0;
}

PyObject* k3d_icommand_node_get_child(PyObject* self, PyObject* args)
{
	k3d::icommand_node* const command_node = python_cast<k3d::icommand_node*>(self);
	return_val_if_fail(command_node, 0);

	PyObject *object1;

	if(!PyArg_ParseTuple(args, "O", &object1))
		return 0;

	if(!PyString_Check(object1))
		return 0;

	const std::string child_name = PyString_AsString(object1);

	k3d::icommand_tree::nodes_t children = k3d::command_tree().children(command_node);
	for(k3d::icommand_tree::nodes_t::iterator child = children.begin(); child != children.end(); ++child)
	{
		if(child_name == k3d::command_tree().name(**child))
			return python_wrap(**child);
	}

	return 0;
}

PyMethodDef k3d_icommand_node_methods[] =
{
	{"execute_command", k3d_icommand_node_execute_command, METH_VARARGS, "Execute a command on the command-node."},
	{"get_child", k3d_icommand_node_get_child, METH_VARARGS, "Return command-node child by name."},
	{NULL, NULL, 0, NULL}
};

/////////////////////////////////////////////////////////////////////////////
// k3d::iproperty_collection

PyObject* k3d_iproperty_collection_get_property(PyObject* self, PyObject* args)
{
	k3d::iproperty_collection* const property_collection = python_cast<k3d::iproperty_collection*>(self);
	return_val_if_fail(property_collection, 0);

	char* string1;
	if(!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	if(k3d::iproperty* const property = k3d::get_property(*property_collection, string1))
		return python_wrap(*property);

	return 0;
}

template<typename PropertyT, typename ValueT>
PyObject* create_user_property(const char* const Name, const char* const Label, const char* const Description, k3d::idocument& Document, k3d::iproperty_collection& PropertyCollection, k3d::ipersistent_container& PersistentContainer, k3d::inode* Object, const ValueT& Value)
{
	PropertyT* const property = k3d::user::create_property<PropertyT, ValueT>(Name, Label, Description, Document, PropertyCollection, PersistentContainer, Object, Value);
	return python_wrap(static_cast<k3d::iproperty*>(property));
}

PyObject* k3d_iproperty_collection_add_user_property(PyObject* self, PyObject* args)
{
	k3d::iproperty_collection* const property_collection = python_cast<k3d::iproperty_collection*>(self);
	return_val_if_fail(property_collection, 0);

	k3d::ipersistent_container* const persistent_container = dynamic_cast<k3d::ipersistent_container*>(property_collection);
	return_val_if_fail(persistent_container, 0);

	k3d::inode* const object = dynamic_cast<k3d::inode*>(property_collection);
	return_val_if_fail(object, 0);

	char* t;
	char* name;
	char* label;
	char* description;
	if(!PyArg_ParseTuple(args, "ssss", &t, &name, &label, &description))
		return 0;

	std::string type(t);
	if(type == "bool")
		return create_user_property<k3d::user::bool_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, false);
	else if(type == "color")
		return create_user_property<k3d::user::color_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::color(1, 1, 1));
	else if(type == "double")
		return create_user_property<k3d::user::double_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, 0.0);
	else if(type == "matrix4")
		return create_user_property<k3d::user::matrix4_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::identity3D());
	else if(type == "string")
		return create_user_property<k3d::user::string_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, std::string());
	else if(type == "vector3")
		return create_user_property<k3d::user::vector3_property>(name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::vector3(0, 0, 0));

	k3d::log() << error << "unknown user property type: [" << type << "]" << std::endl;
	return 0;

}

template<typename PropertyT, typename ValueT>
PyObject* create_renderman_property(const k3d::irenderman_property::parameter_type_t Type, const char* const List, const char* const Name, const char* const Label, const char* const Description, k3d::idocument& Document, k3d::iproperty_collection& PropertyCollection, k3d::ipersistent_container& PersistentContainer, k3d::inode* Object, const ValueT& Value)
{
	PropertyT* const property = k3d::ri::create_property<PropertyT, ValueT>(Type, List, Name, Label, Description, Document, PropertyCollection, PersistentContainer, Object, Value);
	return python_wrap(static_cast<k3d::iproperty*>(property));
}

PyObject* k3d_iproperty_collection_add_ri_attribute(PyObject* self, PyObject* args)
{
	k3d::iproperty_collection* const property_collection = python_cast<k3d::iproperty_collection*>(self);
	return_val_if_fail(property_collection, 0);

	k3d::ipersistent_container* const persistent_container = dynamic_cast<k3d::ipersistent_container*>(property_collection);
	return_val_if_fail(persistent_container, 0);

	k3d::inode* const object = dynamic_cast<k3d::inode*>(property_collection);
	return_val_if_fail(object, 0);

	char* t;
	char* list;
	char* name;
	char* label;
	char* description;
	if(!PyArg_ParseTuple(args, "sssss", &t, &list, &name, &label, &description))
		return 0;

	std::string type(t);
	if(type == "integer")
		return create_renderman_property<k3d::ri::integer_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, 0);
	else if(type == "real")
		return create_renderman_property<k3d::ri::real_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, 0.0);
	else if(type == "string")
		return create_renderman_property<k3d::ri::string_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::string(""));
	else if(type == "point")
		return create_renderman_property<k3d::ri::point_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::point());
	else if(type == "vector")
		return create_renderman_property<k3d::ri::vector_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::vector());
	else if(type == "color")
		return create_renderman_property<k3d::ri::color_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::color(0, 0, 0));
	else if(type == "hpoint")
		return create_renderman_property<k3d::ri::hpoint_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::hpoint());
	else if(type == "matrix")
		return create_renderman_property<k3d::ri::matrix_property>(k3d::irenderman_property::ATTRIBUTE, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::matrix());

	k3d::log() << error << "unknown user property type: [" << type << "]" << std::endl;
	return 0;

}

PyObject* k3d_iproperty_collection_add_ri_option(PyObject* self, PyObject* args)
{
	k3d::iproperty_collection* const property_collection = python_cast<k3d::iproperty_collection*>(self);
	return_val_if_fail(property_collection, 0);

	k3d::ipersistent_container* const persistent_container = dynamic_cast<k3d::ipersistent_container*>(property_collection);
	return_val_if_fail(persistent_container, 0);

	k3d::inode* const object = dynamic_cast<k3d::inode*>(property_collection);
	return_val_if_fail(object, 0);

	char* t;
	char* list;
	char* name;
	char* label;
	char* description;
	if(!PyArg_ParseTuple(args, "sssss", &t, &list, &name, &label, &description))
		return 0;

	std::string type(t);
	if(type == "integer")
		return create_renderman_property<k3d::ri::integer_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, 0);
	else if(type == "real")
		return create_renderman_property<k3d::ri::real_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, 0.0);
	else if(type == "string")
		return create_renderman_property<k3d::ri::string_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::string(""));
	else if(type == "point")
		return create_renderman_property<k3d::ri::point_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::point());
	else if(type == "vector")
		return create_renderman_property<k3d::ri::vector_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::vector());
	else if(type == "color")
		return create_renderman_property<k3d::ri::color_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::color(0, 0, 0));
	else if(type == "hpoint")
		return create_renderman_property<k3d::ri::hpoint_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::hpoint());
	else if(type == "matrix")
		return create_renderman_property<k3d::ri::matrix_property>(k3d::irenderman_property::OPTION, list, name, label, description, object->document(), *property_collection, *persistent_container, object, k3d::ri::matrix());

	k3d::log() << error << "unknown user property type: [" << type << "]" << std::endl;
	return 0;

}

PyMethodDef k3d_iproperty_collection_methods[] =
{
	{"get_property", k3d_iproperty_collection_get_property, METH_VARARGS, "Returns a property by name"},
	{"add_user_property", k3d_iproperty_collection_add_user_property, METH_VARARGS, "Adds a custom property to an existing object"},
	{"add_ri_attribute", k3d_iproperty_collection_add_ri_attribute, METH_VARARGS, "Adds a custom RenderMan attribute to an existing object"},
	{"add_ri_option", k3d_iproperty_collection_add_ri_option, METH_VARARGS, "Adds a custom RenderMan option to an existing object"},
	{NULL, NULL, 0, NULL}
};

/////////////////////////////////////////////////////////////////////////////
// k3d::idocument

PyObject* k3d_idocument_save(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	k3d::auto_ptr<k3d::idocument_write_format> filter(k3d::create_plugin<k3d::idocument_write_format>(k3d::classes::DocumentWriter()));
	return_val_if_fail(filter.get(), false);
	if(filter->write_file(*document, boost::filesystem::path(string1, boost::filesystem::native)))
		return PYVAL_TRUE;

	return 0;
}

PyObject* k3d_idocument_start_change_set(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	k3d::start_state_change_set(*document);
	return PYVAL_TRUE;
}

PyObject* k3d_idocument_cancel_change_set(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	k3d::cancel_state_change_set(*document);
	return PYVAL_TRUE;
}

PyObject* k3d_idocument_finish_change_set(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	k3d::finish_state_change_set(*document, string1);
	return PYVAL_TRUE;
}

PyObject* k3d_idocument_redraw_all(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	k3d::gl::redraw_all(*document, k3d::gl::irender_engine::ASYNCHRONOUS);
	return PYVAL_TRUE;
}

PyObject* k3d_idocument_new_object(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	PyObject *object1;
	if(!PyArg_ParseTuple(args, "O", &object1))
		return 0;

	if(PyString_Check(object1))
	{
		k3d::iplugin_factory_collection::factories_t factories = k3d::plugins(PyString_AsString(object1));
		return_val_if_fail(1 == factories.size(), 0);

		k3d::inode* const object = k3d::create_plugin<k3d::inode>(**factories.begin(), *document);
		return_val_if_fail(object, 0);

		return python_wrap(*object);
	}
	else if(k3d::iplugin_factory* const factory = python_cast<k3d::iplugin_factory*>(object1))
	{
		k3d::inode* const object = k3d::create_plugin<k3d::inode>(*factory, *document);
		return_val_if_fail(object, 0);

		return python_wrap(*object);
	}

	return 0;
}

PyObject* k3d_idocument_get_object(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	k3d::inode* const object = k3d::find_node(document->nodes(), string1);
	return_val_if_fail(object, 0);

	return python_wrap(*object);
}

PyObject* k3d_idocument_delete_object(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	PyObject *object1;
	if (!PyArg_ParseTuple(args, "O", &object1))
		return 0;

	k3d::inode* const object = python_cast<k3d::inode*>(object1);
	return_val_if_fail(object, 0);

	k3d::delete_nodes(*document, k3d::make_collection<k3d::nodes_t>(object));
	return PYVAL_TRUE;
}

PyObject* k3d_idocument_set_dependency(PyObject* self, PyObject* args)
{
	k3d::idocument* const document = python_cast<k3d::idocument*>(self);
	return_val_if_fail(document, 0);

	PyObject *object1, *object2;
	if (!PyArg_ParseTuple(args, "OO", &object1, &object2))
		return 0;

	k3d::iproperty* from = python_cast<k3d::iproperty*>(object1);
	return_val_if_fail(from, 0);

	k3d::iproperty* to = python_cast<k3d::iproperty*>(object2);

	if(from && to)
		return_val_if_fail(from->property_type() == to->property_type(), 0);

	k3d::idag::dependencies_t dependencies;
	dependencies[from] = to;
	document->dag().set_dependencies(dependencies);

	return PYVAL_TRUE;
}

PyMethodDef k3d_idocument_methods[] =
{
	{"new_object", k3d_idocument_new_object, METH_VARARGS, "Create a new K-3D object."},
	{"save", k3d_idocument_save, METH_VARARGS, "Save document to a given file."},
	{"start_change_set", k3d_idocument_start_change_set, METH_VARARGS, "Starts recording an atomic state change for undo/redo."},
	{"cancel_change_set", k3d_idocument_cancel_change_set, METH_VARARGS, "Cancels recording an atomic state change for undo/redo."},
	{"finish_change_set", k3d_idocument_finish_change_set, METH_VARARGS, "Completes recording an atomic state change for undo/redo."},
	{"redraw_all", k3d_idocument_redraw_all, METH_VARARGS, "Redraws all cameras."},
	{"get_object", k3d_idocument_get_object, METH_VARARGS, "Return named K-3D object."},
	{"delete_object", k3d_idocument_delete_object, METH_VARARGS, "Delete named K-3D object."},
	{"set_dependency", k3d_idocument_set_dependency, METH_VARARGS, "Set a dependency between two properties."},
	{NULL, NULL, 0, NULL}
};

/////////////////////////////////////////////////////////////////////////////
// k3d::imesh_storage

PyObject* k3d_imesh_storage_new_mesh(PyObject* self, PyObject* args)
{
	k3d::imesh_storage* const mesh_storage = python_cast<k3d::imesh_storage*>(self);
	return_val_if_fail(mesh_storage, 0);

	k3d::mesh* const mesh = new k3d::mesh();
	mesh_storage->reset_mesh(mesh);

	return python_wrap(*mesh);
}

PyMethodDef k3d_imesh_storage_methods[] =
{
	{"new_mesh", k3d_imesh_storage_new_mesh, METH_VARARGS, "Create a new mesh."},
	{NULL, NULL, 0, NULL}
};

/////////////////////////////////////////////////////////////////////////////
// k3d_iunknown

PyObject* k3d_iunknown_getattr(PyObject* Object, char* Name)
{
	k3d::iunknown* const unknown = python_cast<k3d::iunknown*>(Object);
	return_val_if_fail(unknown, 0);

	const std::string name = Name;

	if(name == "is_application")
		return dynamic_cast<k3d::iapplication*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_command_node")
		return dynamic_cast<k3d::icommand_node*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_property_collection")
		return dynamic_cast<k3d::iproperty_collection*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_document")
		return dynamic_cast<k3d::idocument*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_object")
		return dynamic_cast<k3d::inode*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_selectable")
		return dynamic_cast<k3d::iselectable*>(unknown) ? PYVAL_TRUE : PYVAL_FALSE;

	if(k3d::iselectable* const selectable = dynamic_cast<k3d::iselectable*>(unknown))
	{
		if(name == "visible_selection")
			return python_wrap(selectable->get_visible_selection());

		if(name == "selection_weight")
			return python_wrap(selectable->get_selection_weight());
	}

	if(k3d::iapplication* const application = dynamic_cast<k3d::iapplication*>(unknown))
	{
		if(name == "ui")
			return python_wrap(application->user_interface());

		if(name == "plugins")
		{
			const k3d::iplugin_factory_collection::factories_t& factories = k3d::application().plugins();

			PyObject* const list = PyList_New(0);
			for(k3d::iplugin_factory_collection::factories_t::const_iterator factory = factories.begin(); factory != factories.end(); ++factory)
				PyList_Append(list, python_wrap(**factory));
			return list;
		}

		if(name == "share_path")
			return python_wrap(application->share_path());

		if(PyObject* const method = Py_FindMethod(k3d_iapplication_methods, Object, Name))
			return method;
		PyErr_Clear();
	}

	if(k3d::icommand_node* const command_node = dynamic_cast<k3d::icommand_node*>(unknown))
	{
		if(name == "command_node_name")
			return python_wrap(k3d::command_tree().name(*command_node));

		if(name == "children")
		{
			PyObject* const list = PyList_New(0);
			k3d::icommand_tree::nodes_t children = k3d::command_tree().children(command_node);
			for(k3d::icommand_tree::nodes_t::iterator child = children.begin(); child != children.end(); ++child)
				PyList_Append(list, python_wrap(**child));
			return list;
		}

		if(PyObject* const method = Py_FindMethod(k3d_icommand_node_methods, Object, Name))
			return method;
		PyErr_Clear();
	}

	if(k3d::iproperty_collection* const property_collection = dynamic_cast<k3d::iproperty_collection*>(unknown))
	{
		if(name == "properties")
		{
			PyObject* const list = PyList_New(0);
			const k3d::iproperty_collection::properties_t properties = property_collection->properties();
			for(k3d::iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
				PyList_Append(list, python_wrap(**property));
			return list;
		}

		if(k3d::iproperty* property = k3d::get_property(*property_collection, name))
		{
			k3d::idag* dag = property->property_node() ? &property->property_node()->document().dag() : 0;
			if(dag)
			{
				for(k3d::iproperty* dependency = dag->dependency(*property); dependency; dependency = dag->dependency(*dependency))
					property = dependency;
			}
			return any_to_python(property->property_value());
		}

		if(PyObject* const method = Py_FindMethod(k3d_iproperty_collection_methods, Object, Name))
			return method;
		PyErr_Clear();
	}

	if(k3d::idocument* const document = dynamic_cast<k3d::idocument*>(unknown))
	{
		if(name == "objects")
		{
			const k3d::inode_collection::nodes_t objects = document->nodes().collection();

			PyObject* const list = PyList_New(0);
			for(k3d::inode_collection::nodes_t::const_iterator object = objects.begin(); object != objects.end(); ++object)
				PyList_Append(list, python_wrap(**object));
			return list;
		}

		if(PyObject* const method = Py_FindMethod(k3d_idocument_methods, Object, Name))
			return method;
		PyErr_Clear();
	}

	if(k3d::inode* const object = dynamic_cast<k3d::inode*>(unknown))
	{
		if(name == "name")
			return python_wrap(object->name());
		if(name == "document")
			return python_wrap(object->document());
		if(name == "factory")
			return python_wrap(object->factory());
	}

	if(dynamic_cast<k3d::imesh_storage*>(unknown))
	{
		if(PyObject* const method = Py_FindMethod(k3d_imesh_storage_methods, Object, Name))
			return method;
		PyErr_Clear();
	}

	const std::string error_message = "Unknown attribute: " + name;
	k3d::log() << error << error_message << std::endl;
	PyErr_SetString(PyExc_AttributeError, error_message.c_str());
	return 0;
}

int k3d_iunknown_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::iunknown* const unknown = python_cast<k3d::iunknown*>(Object);
	const std::string name = Name;

	if(k3d::iselectable* const selectable = dynamic_cast<k3d::iselectable*>(unknown))
	{
		if(name == "visible_selection")
		{
			selectable->set_visible_selection(PyObject_IsTrue(Value));
			return 0;
		}

		if(name == "selection_weight")
		{
			return_val_if_fail(PyFloat_Check(Value), -1);
			selectable->set_selection_weight(PyFloat_AsDouble(Value));
			return 0;
		}
	}

	if(k3d::inode* const object = dynamic_cast<k3d::inode*>(unknown))
	{
		if(name == "name")
		{
			return_val_if_fail(PyString_Check(Value), -1);
			object->set_name(PyString_AsString(Value));
			return 0;
		}
	}

	if(k3d::iproperty_collection* const property_collection = dynamic_cast<k3d::iproperty_collection*>(unknown))
	{
		if(k3d::iproperty* const property = k3d::get_property(*property_collection, name))
		{
			if(k3d::iwritable_property* const writable = dynamic_cast<k3d::iwritable_property*>(property))
			{
				writable->property_set_value(python_to_any(Value, property->property_type()));
				return 0;
			}

			const std::string error_message = "Read-only attribute: " + name;
			k3d::log() << error << error_message << std::endl;
			PyErr_SetString(PyExc_AttributeError, error_message.c_str());
			return -1;
		}
	}

	const std::string error_message = "Unknown attribute: " + name;
	k3d::log() << error << error_message << std::endl;
	PyErr_SetString(PyExc_AttributeError, error_message.c_str());
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// python_cast

template<>
k3d::icommand_node* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;
	return_val_if_fail(PyObject_TypeCheck(Object, &k3d_iunknown_type), 0);
	return dynamic_cast<k3d::icommand_node*>(reinterpret_cast<k3d_iunknown*>(Object)->object);
}

template<>
k3d::idocument* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;
	return_val_if_fail(PyObject_TypeCheck(Object, &k3d_iunknown_type), 0);
	return dynamic_cast<k3d::idocument*>(reinterpret_cast<k3d_iunknown*>(Object)->object);
}

template<>
k3d::imesh_storage* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;
	return_val_if_fail(PyObject_TypeCheck(Object, &k3d_iunknown_type), 0);
	return dynamic_cast<k3d::imesh_storage*>(reinterpret_cast<k3d_iunknown*>(Object)->object);
}

template<>
k3d::inode* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;
	return_val_if_fail(PyObject_TypeCheck(Object, &k3d_iunknown_type), 0);
	return dynamic_cast<k3d::inode*>(reinterpret_cast<k3d_iunknown*>(Object)->object);
}

template<>
k3d::iproperty_collection* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;
	return_val_if_fail(PyObject_TypeCheck(Object, &k3d_iunknown_type), 0);
	return dynamic_cast<k3d::iproperty_collection*>(reinterpret_cast<k3d_iunknown*>(Object)->object);
}

/////////////////////////////////////////////////////////////////////////////
// k3d_iuser_interface

PyObject* k3d_iuser_interface_message(PyObject* self, PyObject* args)
{
	char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	python_cast<k3d::iuser_interface*>(self)->message(string1, "");
	return PYVAL_TRUE;
}

PyObject* k3d_iuser_interface_error_message(PyObject* self, PyObject* args)
{
	char *string1;
	if (!PyArg_ParseTuple(args, "s", &string1))
		return 0;

	python_cast<k3d::iuser_interface*>(self)->error_message(string1, "");
	return PYVAL_TRUE;
}

PyObject* k3d_iuser_interface_query_message(PyObject* self, PyObject* args)
{
	PyObject *list1;
	char *string1;
	if (!PyArg_ParseTuple(args, "sO", &string1, &list1))
		return 0;

	std::vector<std::string> buttons;
	for(int i = 0; i < PyList_Size(list1); i++)
	{
		buttons.push_back(std::string(PyString_AsString(PyList_GetItem(list1, i))));
	}

	return python_wrap(static_cast<unsigned long>(python_cast<k3d::iuser_interface*>(self)->query_message(string1, "", 0, buttons)));
}

PyObject* k3d_iuser_interface_get_file_path(PyObject* self, PyObject* args)
{
	k3d::iuser_interface* const user_interface = python_cast<k3d::iuser_interface*>(self);
	return_val_if_fail(user_interface, 0);

	char *string1, *string2, *string3, *string4;
	if(!PyArg_ParseTuple(args, "ssss", &string1, &string2, &string3, &string4))
		return 0;

	const std::string direction = string1;
	const std::string path_type = string2;
	const std::string prompt = string3;
	const boost::filesystem::path old_path(string4, boost::filesystem::native);
	boost::filesystem::path new_path;

	if(direction == "r" || direction == "read")
		user_interface->get_file_path(k3d::ipath_property::READ, path_type, prompt, old_path, new_path);
	else if(direction == "w" || direction == "write")
		user_interface->get_file_path(k3d::ipath_property::WRITE, path_type, prompt, old_path, new_path);
	else
		return_val_if_fail(0, 0);

	return python_wrap(new_path);
}

PyMethodDef k3d_iuser_interface_methods[] =
{
	{"message", k3d_iuser_interface_message, METH_VARARGS, "Displays an informational message in a modal dialog box."},
	{"error_message", k3d_iuser_interface_error_message, METH_VARARGS, "Displays an error in a modal dialog box."},
	{"query_message", k3d_iuser_interface_query_message, METH_VARARGS, "Prompts the user to choose one of selections in a modal dialog box."},
	{"get_file_path", k3d_iuser_interface_get_file_path, METH_VARARGS, "Prompt user for filepath."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_iuser_interface_getattr(PyObject* Object, char* Name)
{
/*
	k3d::iuser_interface* const user_interface = python_cast<k3d::iuser_interface*>(Object);
	return_val_if_fail(user_interface, 0);
	const std::string name = Name;
*/
	return Py_FindMethod(k3d_iuser_interface_methods, Object, Name);
}

int k3d_iuser_interface_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_iplugin_factory

PyObject* k3d_iplugin_factory_getattr(PyObject* Object, char* Name)
{
	k3d::iplugin_factory* const plugin_factory = python_cast<k3d::iplugin_factory*>(Object);
	return_val_if_fail(plugin_factory, 0);
	const std::string name = Name;

	if(name == "class_id")
	{
		const k3d::uuid& class_id = plugin_factory->class_id();
		return Py_BuildValue(
			"(OOOO)",
			PyLong_FromUnsignedLong(class_id.data1),
			PyLong_FromUnsignedLong(class_id.data2),
			PyLong_FromUnsignedLong(class_id.data3),
			PyLong_FromUnsignedLong(class_id.data4));
	}
	if(name == "name")
		return python_wrap(plugin_factory->name());
	if(name == "short_description")
		return python_wrap(plugin_factory->short_description());
	if(name == "is_application_plugin")
		return dynamic_cast<k3d::iapplication_plugin_factory*>(plugin_factory) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_document_plugin")
		return dynamic_cast<k3d::idocument_plugin_factory*>(plugin_factory) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "categories")
	{
		const k3d::iplugin_factory::categories_t& categories = plugin_factory->categories();

		PyObject* const list = PyList_New(0);
		for(k3d::iplugin_factory::categories_t::const_iterator category = categories.begin(); category != categories.end(); ++category)
			PyList_Append(list, python_wrap(*category));
		return list;
	}
	if(name == "quality")
	{
		switch(plugin_factory->quality())
		{
			case k3d::iplugin_factory::STABLE:
				return python_wrap("stable");
			case k3d::iplugin_factory::EXPERIMENTAL:
				return python_wrap("experimental");
			case k3d::iplugin_factory::DEPRECATED:
				return python_wrap("deprecated");
		}
	}

	return 0;
}

int k3d_iplugin_factory_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_iproperty

PyObject* k3d_iproperty_getattr(PyObject* Object, char* Name)
{
	k3d::iproperty* const property = python_cast<k3d::iproperty*>(Object);
	const std::string name = Name;

	if(name == "name")
		return python_wrap(property->property_name());
	if(name == "label")
		return python_wrap(property->property_label());
	if(name == "description")
		return python_wrap(property->property_description());
	if(name == "type")
		return type_to_string(property->property_type());
	if(name == "internal_value")
		return any_to_python(property->property_value());
	if(name == "value")
	{
		k3d::iproperty* source = property;
		k3d::idag* dag = property->property_node() ? &property->property_node()->document().dag() : 0;
		if(dag)
		{
			for(k3d::iproperty* dependency = dag->dependency(*source); dependency; dependency = dag->dependency(*dependency))
				source = dependency;
		}
		return any_to_python(source->property_value());
	}
	if(name == "object")
		return property->property_node() ? python_wrap(*property->property_node()) : PYVAL_NONE;
	if(name == "is_writable")
		return dynamic_cast<k3d::iwritable_property*>(property) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "is_enumeration")
		return dynamic_cast<k3d::ienumeration_property*>(property) ? PYVAL_TRUE : PYVAL_FALSE;
	if(name == "enumeration_values")
	{
		if(k3d::ienumeration_property* const enumeration = dynamic_cast<k3d::ienumeration_property*>(property))
		{
			const k3d::ienumeration_property::enumeration_values_t values = enumeration->property_values();

			PyObject* const list = PyList_New(0);
			for(k3d::ienumeration_property::enumeration_values_t::const_iterator value = values.begin(); value != values.end(); ++value)
				PyList_Append(list, python_wrap(value->value));
			return list;
		}

		return 0;
	}
	if(name == "units")
	{
		if(k3d::imeasurement_property* const measurement = dynamic_cast<k3d::imeasurement_property*>(property))
			return units_to_string(measurement->property_units());
		return PYVAL_NONE;
	}

	return 0;
}

int k3d_iproperty_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::iproperty* const property = python_cast<k3d::iproperty*>(Object);
	const std::string name = Name;

	if(name == "value")
	{
		if(k3d::iwritable_property* const writable = dynamic_cast<k3d::iwritable_property*>(property))
		{
			writable->property_set_value(python_to_any(Value, property->property_type()));
			return 0;
		}
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_parameters

PyObject* k3d_parameters_set_integer(PyObject* self, PyObject* args)
{
	char* name = 0;
	k3d::ri::integer value = 0;
	if(!PyArg_ParseTuple(args, "si", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), boost::any(value)));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_real(PyObject* self, PyObject* args)
{
	char* name = 0;
	k3d::ri::real value = 0;
	if(!PyArg_ParseTuple(args, "sd", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), boost::any(value)));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_string(PyObject* self, PyObject* args)
{
	char* name = 0;
	char* value = 0;
	if(!PyArg_ParseTuple(args, "ss", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), boost::any(k3d::ri::string(value))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_point(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), python_to_any(value, typeid(k3d::ri::point))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_vector(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), python_to_any(value, typeid(k3d::ri::vector))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_normal(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), python_to_any(value, typeid(k3d::ri::normal))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_color(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), python_to_any(value, typeid(k3d::ri::color))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_hpoint(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), python_to_any(value, typeid(k3d::ri::hpoint))));
	return PYVAL_NONE;
}

PyObject* k3d_parameters_set_bool(PyObject* self, PyObject* args)
{
	char* name = 0;
	PyObject* value = 0;
	if(!PyArg_ParseTuple(args, "sO", &name, &value))
		return 0;

	python_cast<k3d::parameters_t*>(self)->insert(std::make_pair(std::string(name), boost::any(PyObject_IsTrue(value) ? true : false)));
	return PYVAL_NONE;
}

PyMethodDef k3d_parameters_methods[] =
{
	{"set_integer", k3d_parameters_set_integer, METH_VARARGS, "Sets a named integer parameter."},
	{"set_real", k3d_parameters_set_real, METH_VARARGS, "Sets a named real parameter."},
	{"set_string", k3d_parameters_set_string, METH_VARARGS, "Sets a named string parameter."},
	{"set_point", k3d_parameters_set_point, METH_VARARGS, "Sets a named point parameter."},
	{"set_vector", k3d_parameters_set_vector, METH_VARARGS, "Sets a named vector parameter."},
	{"set_normal", k3d_parameters_set_normal, METH_VARARGS, "Sets a named normal parameter."},
	{"set_color", k3d_parameters_set_color, METH_VARARGS, "Sets a named color parameter."},
	{"set_hpoint", k3d_parameters_set_hpoint, METH_VARARGS, "Sets a named homogeneous point parameter."},
	{"set_bool", k3d_parameters_set_bool, METH_VARARGS, "Sets a named boolean parameter."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_parameters_getattr(PyObject* Object, char* Name)
{
	k3d::parameters_t* const parameters = python_cast<k3d::parameters_t*>(Object);
	const std::string name = Name;

	k3d::parameters_t::iterator parameter = parameters->find(name);
	if(parameter != parameters->end())
		return any_to_python(parameter->second);

	return Py_FindMethod(k3d_parameters_methods, Object, Name);
}

int k3d_parameters_setattr(PyObject* Object, char* Name, PyObject* Value)
{
/*
	k3d::parameters_t* const parameters = python_cast<k3d::parameters_t*>(Object);
	const std::string name = Name;
	(*parameters)[name] = python_to_any(Value, typeid(k3d::color));
*/

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_point

PyObject* k3d_point_getattr(PyObject* Object, char* Name)
{
	k3d::point* const point = python_cast<k3d::point*>(Object);
	const std::string name = Name;

	if(name == "position")
		return python_wrap(point->position);
	if(name == "vertex_data")
		return python_wrap(point->vertex_data);
	if(name == "tags")
		return python_wrap(point->tags);

	return 0;
}

int k3d_point_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::point* const point = python_cast<k3d::point*>(Object);
	const std::string name = Name;

	if(name == "position")
	{
		k3d::vector3* const vector3 = python_cast<k3d::vector3*>(Value);
		if(vector3)
		{
			point->position = *vector3;
			return 0;
		}

		return_val_if_fail(PyTuple_Check(Value), -1);
		return_val_if_fail(3 == PyTuple_Size(Value), -1);

		point->position[0] = PyFloat_AsDouble(PyTuple_GetItem(Value, 0));
		point->position[1] = PyFloat_AsDouble(PyTuple_GetItem(Value, 1));
		point->position[2] = PyFloat_AsDouble(PyTuple_GetItem(Value, 2));

		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_point_group

PyMethodDef k3d_point_group_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_point_group_getattr(PyObject* Object, char* Name)
{
	k3d::point_group* const group = python_cast<k3d::point_group*>(Object);
	const std::string name = Name;

	if(name == "points")
		return python_wrap(group->points);
	if(name == "constant_data")
		return python_wrap(group->constant_data);
	if(name == "material")
		return group->material ? python_wrap(*group->material) : PYVAL_NONE;

	return Py_FindMethod(k3d_point_group_methods, Object, Name);
}

int k3d_point_group_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::point_group* const group = python_cast<k3d::point_group*>(Object);
	const std::string name = Name;

	if(name == "material")
	{
		group->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_split_edge

PyMethodDef k3d_split_edge_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_split_edge_getattr(PyObject* Object, char* Name)
{
	k3d::split_edge* const edge = python_cast<k3d::split_edge*>(Object);
	const std::string name = Name;

	if(name == "vertex")
		return edge->vertex ? python_wrap(*edge->vertex) : PYVAL_NONE;
	if(name == "face_clockwise")
		return edge->face_clockwise ? python_wrap(*edge->face_clockwise) : PYVAL_NONE;
	if(name == "companion")
		return edge->companion ? python_wrap(*edge->companion) : PYVAL_NONE;
	if(name == "facevarying_data")
		return python_wrap(edge->facevarying_data);
	if(name == "tags")
		return python_wrap(edge->tags);

	return 0;
}

int k3d_split_edge_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::split_edge* const edge = python_cast<k3d::split_edge*>(Object);
	const std::string name = Name;

	if(name == "vertex")
	{
		edge->vertex = python_cast<k3d::point*>(Value);
		return 0;
	}
	if(name == "face_clockwise")
	{
		edge->face_clockwise = python_cast<k3d::split_edge*>(Value);
		return 0;
	}
	if(name == "companion")
	{
		edge->companion = python_cast<k3d::split_edge*>(Value);
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_face

PyMethodDef k3d_face_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_face_getattr(PyObject* Object, char* Name)
{
	k3d::face* const face = python_cast<k3d::face*>(Object);
	const std::string name = Name;

	if(name == "first_edge")
		return face->first_edge ? python_wrap(*face->first_edge) : PYVAL_NONE;
	if(name == "material")
		return face->material ? python_wrap(*face->material) : PYVAL_NONE;
	if(name == "uniform_data")
		return python_wrap(face->uniform_data);
	if(name == "tags")
		return python_wrap(face->tags);

	return Py_FindMethod(k3d_face_methods, Object, Name);
}

int k3d_face_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::face* const face = python_cast<k3d::face*>(Object);
	const std::string name = Name;

	if(name == "first_edge")
	{
		face->first_edge = python_cast<k3d::split_edge*>(Value);
		return 0;
	}

	if(name == "material")
	{
		face->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_polyhedron

PyObject* k3d_polyhedron_new_edge(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	PyObject* object2 = 0;
	PyObject* object3 = 0;
	if(!PyArg_ParseTuple(args, "O|OO", &object1, &object2, &object3))
		return 0;

	k3d::point* const point = python_cast<k3d::point*>(object1);
	k3d::split_edge* const face_clockwise = python_cast<k3d::split_edge*>(object2);
	k3d::split_edge* const companion = python_cast<k3d::split_edge*>(object3);

	k3d::split_edge* const edge = new k3d::split_edge(point, face_clockwise, companion);

	return python_wrap(*edge);
}

PyObject* k3d_polyhedron_new_face(PyObject* self, PyObject* args)
{
	PyObject* object;
	if(!PyArg_ParseTuple(args, "O", &object))
		return 0;

	k3d::split_edge* edge = python_cast<k3d::split_edge*>(object);
	return_val_if_fail(edge, 0);

	k3d::face* const face = new k3d::face(edge, 0);
	python_cast<k3d::polyhedron*>(self)->faces.push_back(face);

	return python_wrap(*face);
}

PyMethodDef k3d_polyhedron_methods[] =
{
	{"new_edge", k3d_polyhedron_new_edge, METH_VARARGS, "Create a new edge."},
	{"new_face", k3d_polyhedron_new_face, METH_VARARGS, "Create a new face."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_polyhedron_getattr(PyObject* Object, char* Name)
{
	k3d::polyhedron* const polyhedron = python_cast<k3d::polyhedron*>(Object);
	const std::string name = Name;

	if(name == "faces")
		return python_wrap(polyhedron->faces);
	if(name == "constant_data")
		return python_wrap(polyhedron->constant_data);
	if(name == "tags")
		return python_wrap(polyhedron->tags);

	return Py_FindMethod(k3d_polyhedron_methods, Object, Name);
}

int k3d_polyhedron_setattr(PyObject* Object, char* Name, PyObject* Value)
{
/*
	k3d::polyhedron* const polyhedron = python_cast<k3d::polyhedron*>(Object);
	const std::string name = Name;
*/

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_linear_curve

PyMethodDef k3d_linear_curve_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_linear_curve_getattr(PyObject* Object, char* Name)
{
	k3d::linear_curve* const curve = python_cast<k3d::linear_curve*>(Object);
	const std::string name = Name;

	if(name == "control_points")
		return python_wrap(curve->control_points);
	if(name == "uniform_data")
		return python_wrap(curve->uniform_data);
	if(name == "varying_data")
		return python_wrap(curve->varying_data);

	return 0;
}

int k3d_linear_curve_setattr(PyObject* Object, char* Name, PyObject* Value)
{
/*
	k3d::linear_curve* const curve = python_cast<k3d::linear_curve*>(Object);
	const std::string name = Name;
*/
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_linear_curve_group

PyObject* k3d_linear_curve_group_new_curve(PyObject* self, PyObject* args)
{
	k3d::linear_curve* const curve = new k3d::linear_curve();
	python_cast<k3d::linear_curve_group*>(self)->curves.push_back(curve);
	return python_wrap(*curve);
}

PyMethodDef k3d_linear_curve_group_methods[] =
{
	{"new_curve", k3d_linear_curve_group_new_curve, METH_VARARGS, "Create a new linear curve."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_linear_curve_group_getattr(PyObject* Object, char* Name)
{
	k3d::linear_curve_group* const group = python_cast<k3d::linear_curve_group*>(Object);
	const std::string name = Name;

	if(name == "curves")
		return python_wrap(group->curves);
	if(name == "constant_data")
		return python_wrap(group->constant_data);
	if(name == "material")
		return group->material ? python_wrap(*group->material) : PYVAL_NONE;

	return Py_FindMethod(k3d_linear_curve_group_methods, Object, Name);
}

int k3d_linear_curve_group_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::linear_curve_group* const group = python_cast<k3d::linear_curve_group*>(Object);
	const std::string name = Name;

	if(name == "material")
	{
		group->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_cubic_curve

PyMethodDef k3d_cubic_curve_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_cubic_curve_getattr(PyObject* Object, char* Name)
{
	k3d::cubic_curve* const curve = python_cast<k3d::cubic_curve*>(Object);
	const std::string name = Name;

	if(name == "control_points")
		return python_wrap(curve->control_points);
	if(name == "uniform_data")
		return python_wrap(curve->uniform_data);
	if(name == "varying_data")
		return python_wrap(curve->varying_data);

	return 0;
}

int k3d_cubic_curve_setattr(PyObject* Object, char* Name, PyObject* Value)
{
/*
	k3d::cubic_curve* const curve = python_cast<k3d::cubic_curve*>(Object);
	const std::string name = Name;
*/
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_cubic_curve_group

PyObject* k3d_cubic_curve_group_new_curve(PyObject* self, PyObject* args)
{
	k3d::cubic_curve* const curve = new k3d::cubic_curve();
	python_cast<k3d::cubic_curve_group*>(self)->curves.push_back(curve);
	return python_wrap(*curve);
}

PyMethodDef k3d_cubic_curve_group_methods[] =
{
	{"new_curve", k3d_cubic_curve_group_new_curve, METH_VARARGS, "Create a new cubic curve."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_cubic_curve_group_getattr(PyObject* Object, char* Name)
{
	k3d::cubic_curve_group* const group = python_cast<k3d::cubic_curve_group*>(Object);
	const std::string name = Name;

	if(name == "curves")
		return python_wrap(group->curves);
	if(name == "constant_data")
		return python_wrap(group->constant_data);
	if(name == "material")
		return group->material ? python_wrap(*group->material) : PYVAL_NONE;

	return Py_FindMethod(k3d_cubic_curve_group_methods, Object, Name);
}

int k3d_cubic_curve_group_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::cubic_curve_group* const group = python_cast<k3d::cubic_curve_group*>(Object);
	const std::string name = Name;

	if(name == "material")
	{
		group->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_bilinear_patch

PyMethodDef k3d_bilinear_patch_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_bilinear_patch_getattr(PyObject* Object, char* Name)
{
	k3d::bilinear_patch* const patch = python_cast<k3d::bilinear_patch*>(Object);
	const std::string name = Name;

	if(name == "control_points")
		return python_wrap(patch->control_points);
	if(name == "material")
		return patch->material ? python_wrap(*patch->material) : PYVAL_NONE;
	if(name == "uniform_data")
		return python_wrap(patch->uniform_data);
	if(name == "varying_data")
	{
		PyObject* const list = PyList_New(0);
		for(k3d::bilinear_patch::varying_t::iterator data = patch->varying_data.begin(); data != patch->varying_data.end(); ++data)
			PyList_Append(list, python_wrap(*data));
		return list;
	}

	return 0;
}

int k3d_bilinear_patch_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::bilinear_patch* const patch = python_cast<k3d::bilinear_patch*>(Object);
	const std::string name = Name;

	if(name == "material")
	{
		patch->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_bicubic_patch

PyMethodDef k3d_bicubic_patch_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_bicubic_patch_getattr(PyObject* Object, char* Name)
{
	k3d::bicubic_patch* const patch = python_cast<k3d::bicubic_patch*>(Object);
	const std::string name = Name;

	if(name == "control_points")
		return python_wrap(patch->control_points);
	if(name == "material")
		return patch->material ? python_wrap(*patch->material) : PYVAL_NONE;
	if(name == "uniform_data")
		return python_wrap(patch->uniform_data);
	if(name == "varying_data")
	{
		PyObject* const list = PyList_New(0);
		for(k3d::bicubic_patch::varying_t::iterator data = patch->varying_data.begin(); data != patch->varying_data.end(); ++data)
			PyList_Append(list, python_wrap(*data));
		return list;
	}

	return 0;
}

int k3d_bicubic_patch_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::bicubic_patch* const patch = python_cast<k3d::bicubic_patch*>(Object);
	const std::string name = Name;

	if(name == "material")
		{
			patch->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
			return 0;
		}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_blobby

typedef std::vector<PyObject*> python_objects_t;

void flatten_sequences(PyObject* const Item, python_objects_t& Results)
{
	if(PySequence_Check(Item))
	{
		const int count = PySequence_Length(Item);
		for(int i = 0; i < count; ++i)
			flatten_sequences(PySequence_GetItem(Item, i), Results);
	}
	else
	{
		Results.push_back(Item);
	}
}

typedef std::vector<k3d::blobby::opcode*> opcodes_t;

opcodes_t get_opcodes(PyObject* Args)
{
	opcodes_t opcodes;

	python_objects_t objects;
	flatten_sequences(Args, objects);
	for(python_objects_t::iterator object = objects.begin(); object != objects.end(); ++object)
	{
		k3d::blobby::opcode* const opcode = python_cast<k3d::blobby::opcode*>(*object);
		if(opcode)
			opcodes.push_back(opcode);
	}

	return opcodes;
}

PyObject* k3d_blobby_new_constant(PyObject* self, PyObject* args)
{
	double value;
	if(!PyArg_ParseTuple(args, "f", &value))
		return 0;

	return python_wrap(*new k3d::blobby::constant(value));
}

PyObject* k3d_blobby_new_ellipsoid(PyObject* self, PyObject* args)
{
	PyObject *object1 = 0;
	if(!PyArg_ParseTuple(args, "O", &object1))
		return 0;

	k3d::point* const origin = python_cast<k3d::point*>(object1);

	return python_wrap(*new k3d::blobby::ellipsoid(origin, k3d::identity3D()));
}

PyObject* k3d_blobby_new_segment(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	PyObject* object2 = 0;
	double radius = 0;
	if(!PyArg_ParseTuple(args, "OOf", &object1, &object2, &radius))
		return 0;

	k3d::point* const start = python_cast<k3d::point*>(object1);
	k3d::point* const end = python_cast<k3d::point*>(object2);

	return python_wrap(*new k3d::blobby::segment(start, end, radius, k3d::identity3D()));
}

PyObject* k3d_blobby_new_add(PyObject* self, PyObject* args)
{
	k3d::blobby::add* add = new k3d::blobby::add();
	add->operands = get_opcodes(args);
	return python_wrap(*add);
}

PyObject* k3d_blobby_new_subtract(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	PyObject* object2 = 0;
	if(!PyArg_ParseTuple(args, "OO", &object1, &object2))
		return 0;

	k3d::blobby::opcode* subtrahend = python_cast<k3d::blobby::opcode*>(object1);
	k3d::blobby::opcode* minuend = python_cast<k3d::blobby::opcode*>(object2);

	return python_wrap(*new k3d::blobby::subtract(subtrahend, minuend));
}

PyObject* k3d_blobby_new_multiply(PyObject* self, PyObject* args)
{
	k3d::blobby::multiply* multiply = new k3d::blobby::multiply();
	multiply->operands = get_opcodes(args);
	return python_wrap(*multiply);
}

PyObject* k3d_blobby_new_divide(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	PyObject* object2 = 0;
	if(!PyArg_ParseTuple(args, "OO", &object1, &object2))
		return 0;

	k3d::blobby::opcode* dividend = python_cast<k3d::blobby::opcode*>(object1);
	k3d::blobby::opcode* divisor = python_cast<k3d::blobby::opcode*>(object2);

	return python_wrap(*new k3d::blobby::divide(dividend, divisor));
}

PyObject* k3d_blobby_new_min(PyObject* self, PyObject* args)
{
	k3d::blobby::min* min = new k3d::blobby::min();
	min->operands = get_opcodes(args);
	return python_wrap(*min);
}

PyObject* k3d_blobby_new_max(PyObject* self, PyObject* args)
{
	k3d::blobby::max* max = new k3d::blobby::max();
	max->operands = get_opcodes(args);
	return python_wrap(*max);
}

PyMethodDef k3d_blobby_methods[] =
{
	{"new_constant", k3d_blobby_new_constant, METH_VARARGS, "Create a new blobby constant opcode."},
	{"new_ellipsoid", k3d_blobby_new_ellipsoid, METH_VARARGS, "Create a new blobby constant opcode."},
	{"new_segment", k3d_blobby_new_segment, METH_VARARGS, "Create a new blobby segment opcode."},
	{"new_add", k3d_blobby_new_add, METH_VARARGS, "Create a new blobby add opcode."},
	{"new_subtract", k3d_blobby_new_subtract, METH_VARARGS, "Create a new blobby subtract opcode."},
	{"new_multiply", k3d_blobby_new_multiply, METH_VARARGS, "Create a new blobby multiply opcode."},
	{"new_divide", k3d_blobby_new_divide, METH_VARARGS, "Create a new blobby divide opcode."},
	{"new_min", k3d_blobby_new_min, METH_VARARGS, "Create a new blobby min opcode."},
	{"new_max", k3d_blobby_new_max, METH_VARARGS, "Create a new blobby max opcode."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_blobby_getattr(PyObject* Object, char* Name)
{
	k3d::blobby* const blobby = python_cast<k3d::blobby*>(Object);
	const std::string name = Name;

	if(name == "material")
		return blobby->material ? python_wrap(*blobby->material) : PYVAL_NONE;
	if(name == "root")
		return blobby->root ? python_wrap(*blobby->root) : PYVAL_NONE;

	return Py_FindMethod(k3d_blobby_methods, Object, Name);
}

int k3d_blobby_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	k3d::blobby* const blobby = python_cast<k3d::blobby*>(Object);
	const std::string name = Name;

	if(name == "material")
	{
		blobby->material = dynamic_cast<k3d::imaterial*>(python_cast<k3d::inode*>(Value));
		return 0;
	}

	if(name == "root")
	{
		blobby->root = python_cast<k3d::blobby::opcode*>(Value);
		return 0;
	}

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_blobby_opcode

PyMethodDef k3d_blobby_opcode_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_blobby_opcode_getattr(PyObject* Object, char* Name)
{
/*
	k3d::blobby::opcode* const opcode = python_cast<k3d::blobby::opcode*>(Object);
	const std::string name = Name;
*/
	return 0;
}

int k3d_blobby_opcode_setattr(PyObject* Object, char* Name, PyObject* Value)
{
/*
	k3d::blobby::opcode* const opcode = python_cast<k3d::blobby::opcode*>(Object);
	const std::string name = Name;
*/

	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_mesh

PyObject* k3d_mesh_new_point(PyObject* self, PyObject* args)
{
	k3d::vector3 position;
	if(!PyArg_ParseTuple(args, "(ddd)", &position[0], &position[1], &position[2]))
		return 0;

	k3d::point* const point = new k3d::point(position);
	python_cast<k3d::mesh*>(self)->points.push_back(point);
	return python_wrap(*point);
}

PyObject* k3d_mesh_new_point_group(PyObject* self, PyObject* args)
{
	k3d::point_group* const group = new k3d::point_group();
	python_cast<k3d::mesh*>(self)->point_groups.push_back(group);
	return python_wrap(*group);
}

PyObject* k3d_mesh_new_polyhedron(PyObject* self, PyObject* args)
{
	k3d::polyhedron* const polyhedron = new k3d::polyhedron();
	python_cast<k3d::mesh*>(self)->polyhedra.push_back(polyhedron);
	return python_wrap(*polyhedron);
}

PyObject* k3d_mesh_new_linear_curve_group(PyObject* self, PyObject* args)
{
	k3d::linear_curve_group* const group = new k3d::linear_curve_group();
	python_cast<k3d::mesh*>(self)->linear_curve_groups.push_back(group);
	return python_wrap(*group);
}

PyObject* k3d_mesh_new_cubic_curve_group(PyObject* self, PyObject* args)
{
	k3d::cubic_curve_group* const group = new k3d::cubic_curve_group();
	python_cast<k3d::mesh*>(self)->cubic_curve_groups.push_back(group);
	return python_wrap(*group);
}

PyObject* k3d_mesh_new_bilinear_patch(PyObject* self, PyObject* args)
{
	k3d::bilinear_patch* const patch = new k3d::bilinear_patch();
	python_cast<k3d::mesh*>(self)->bilinear_patches.push_back(patch);
	return python_wrap(*patch);
}

PyObject* k3d_mesh_new_bicubic_patch(PyObject* self, PyObject* args)
{
	k3d::bicubic_patch* const patch = new k3d::bicubic_patch();
	python_cast<k3d::mesh*>(self)->bicubic_patches.push_back(patch);

	return python_wrap(*patch);
}

PyObject* k3d_mesh_new_blobby(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	if(!PyArg_ParseTuple(args, "|O", &object1))
		return 0;

	k3d::blobby::opcode* const root = python_cast<k3d::blobby::opcode*>(object1);

	k3d::blobby* const blobby = new k3d::blobby(root);
	python_cast<k3d::mesh*>(self)->blobbies.push_back(blobby);

	return python_wrap(*blobby);
}

PyMethodDef k3d_mesh_methods[] =
{
	{"new_point", k3d_mesh_new_point, METH_VARARGS, "Create a new point."},
	{"new_point_group", k3d_mesh_new_point_group, METH_VARARGS, "Create a new point group."},
	{"new_polyhedron", k3d_mesh_new_polyhedron, METH_VARARGS, "Create a new polyhedron."},
	{"new_linear_curve_group", k3d_mesh_new_linear_curve_group, METH_VARARGS, "Create a new linear curve group."},
	{"new_cubic_curve_group", k3d_mesh_new_cubic_curve_group, METH_VARARGS, "Create a new cubic curve group."},
	{"new_bilinear_patch", k3d_mesh_new_bilinear_patch, METH_VARARGS, "Create a new bilinear patch."},
	{"new_bicubic_patch", k3d_mesh_new_bicubic_patch, METH_VARARGS, "Create a new bicubic patch."},
	{"new_blobby", k3d_mesh_new_blobby, METH_VARARGS, "Create a new blobby."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_mesh_getattr(PyObject* Object, char* Name)
{
	k3d::mesh* const mesh = python_cast<k3d::mesh*>(Object);
	const std::string name = Name;

	if(name == "points")
		return python_wrap(mesh->points);
	if(name == "point_groups")
		return python_wrap(mesh->point_groups);
	if(name == "polyhedra")
		return python_wrap(mesh->polyhedra);
	if(name == "linear_curve_groups")
		return python_wrap(mesh->linear_curve_groups);
	if(name == "cubic_curve_groups")
		return python_wrap(mesh->cubic_curve_groups);
	if(name == "bilinear_patches")
		return python_wrap(mesh->bilinear_patches);
	if(name == "bicubic_patches")
		return python_wrap(mesh->bicubic_patches);
	if(name == "blobbies")
		return python_wrap(mesh->blobbies);

	return Py_FindMethod(k3d_mesh_methods, Object, Name);
}

int k3d_mesh_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

/////////////////////////////////////////////////////////////////////////////
// k3d_bitmap

extern PyTypeObject k3d_bitmap_type;

struct k3d_bitmap
{
	PyObject_HEAD
	k3d::bitmap* object;
	k3d::bitmap::iterator iterator;
};

PyObject* k3d_bitmap_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_bitmap*>(self)->object = 0;
	return reinterpret_cast<PyObject*>(self);
}

int k3d_bitmap_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::bitmap* const value = python_cast<k3d::bitmap*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%ld, %ld)", value->width(), value->height());

	return 0;
}

PyObject* k3d_bitmap_assign(PyObject* self, PyObject* args)
{
	k3d::bitmap* const lhs = python_cast<k3d::bitmap*>(self);
	return_val_if_fail(lhs, 0);

	PyObject *object1;
	if (!PyArg_ParseTuple(args, "O", &object1))
		return 0;

	k3d::bitmap* const rhs = python_cast<k3d::bitmap*>(object1);
	return_val_if_fail(rhs, 0);

	*lhs = *rhs;

	return PYVAL_NONE;
}

PyObject* k3d_bitmap_reset(PyObject* self, PyObject* args)
{
	k3d::bitmap* const lhs = python_cast<k3d::bitmap*>(self);
	return_val_if_fail(lhs, 0);

	int width = 0;
	int height = 0;
	if (!PyArg_ParseTuple(args, "ii", &width, &height))
		return 0;

	lhs->reset(width, height);

	return PYVAL_NONE;
}

PyObject* k3d_bitmap_get_pixel(PyObject* self, PyObject* args)
{
	k3d::bitmap* const lhs = python_cast<k3d::bitmap*>(self);
	return_val_if_fail(lhs, 0);

	int x = 0;
	int y = 0;
	if (!PyArg_ParseTuple(args, "ii", &x, &y))
		return 0;

	if(x >= static_cast<long>(lhs->width()))
		return 0;

	if(y >= static_cast<long>(lhs->height()))
		return 0;

	return python_wrap(*(lhs->begin() + (lhs->width() * y) + x));
}

PyObject* k3d_bitmap_set_pixel(PyObject* self, PyObject* args)
{
	k3d::bitmap* const lhs = python_cast<k3d::bitmap*>(self);
	return_val_if_fail(lhs, 0);

	int x = 0;
	int y = 0;
	PyObject* object1 = 0;
	if (!PyArg_ParseTuple(args, "iiO", &x, &y, &object1))
		return 0;

	if(x >= static_cast<long>(lhs->width()))
		return 0;

	if(y >= static_cast<long>(lhs->height()))
		return 0;

	k3d::color* const rhs = python_cast<k3d::color*>(object1);
	return_val_if_fail(rhs, 0);

	(*(lhs->begin() + (lhs->width() * y) + x)) = *rhs;

	return PYVAL_NONE;
}

PyMethodDef k3d_bitmap_methods[] =
{
	{"assign", k3d_bitmap_assign, METH_VARARGS, "Make a copy of a bitmap."},
	{"reset", k3d_bitmap_reset, METH_VARARGS, "Resize a bitmap, discarding the previous contents."},
	{"get_pixel", k3d_bitmap_get_pixel, METH_VARARGS, "Return a bitmap pixel using cartesian coordinates."},
	{"set_pixel", k3d_bitmap_set_pixel, METH_VARARGS, "Set a bitmap pixel using cartesian coordinates and a color value."},
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_bitmap_getattr(PyObject* Object, char* Name)
{
	k3d::bitmap* const bitmap = python_cast<k3d::bitmap*>(Object);
	return_val_if_fail(bitmap, 0);

	const std::string name = Name;
	if(name == "width")
		return any_to_python(bitmap->width());
	if(name == "height")
		return any_to_python(bitmap->height());

	return Py_FindMethod(k3d_bitmap_methods, Object, Name);
}

int k3d_bitmap_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* python_wrap(k3d::bitmap& Object)
{
	k3d_bitmap* const result = PyObject_New(k3d_bitmap, &k3d_bitmap_type);
	return_val_if_fail(result, 0);
	result->object = &Object;
	return reinterpret_cast<PyObject*>(result);
}

PyObject* python_wrap(k3d::bitmap* Object)
{
	return_val_if_fail(Object, 0);
	k3d_bitmap* const result = PyObject_New(k3d_bitmap, &k3d_bitmap_type);
	return_val_if_fail(result, 0);
	result->object = Object;
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::bitmap* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_bitmap_type))
		return 0;

	return reinterpret_cast<k3d_bitmap*>(Object)->object;
}

PyObject* k3d_bitmap_item(PyObject* Object, int Index)
{
	k3d::bitmap* const value = python_cast<k3d::bitmap*>(Object);
	return_val_if_fail(value, 0);

	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(static_cast<unsigned long>(Index) <= value->size(), 0);

	return python_wrap(*(value->begin() + Index));
}

int k3d_bitmap_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	k3d::bitmap* const lhs = python_cast<k3d::bitmap*>(Object);
	return_val_if_fail(lhs, -1);

	k3d::color* const rhs = python_cast<k3d::color*>(Value);
	return_val_if_fail(rhs, -1);

	return_val_if_fail(0 <= Index, -1);
	return_val_if_fail(static_cast<unsigned long>(Index) <= lhs->size(), -1);

	(*(lhs->begin() + Index)) = *rhs;

	return 0;
}

PyObject* k3d_bitmap_getiter(PyObject* Object)
{
	k3d_bitmap* const object = reinterpret_cast<k3d_bitmap*>(Object);
	return_val_if_fail(object, 0);
	return_val_if_fail(object->object, 0);
	object->iterator = object->object->begin();
	Py_INCREF(Object);
	return Object;
}

PyObject* k3d_bitmap_iternext(PyObject* Object)
{
	k3d_bitmap* const object = reinterpret_cast<k3d_bitmap*>(Object);
	return_val_if_fail(object, 0);
	return_val_if_fail(object->object, 0);
	if(object->iterator != object->object->end())
		return python_wrap(*object->iterator++);
	PyErr_SetNone(PyExc_StopIteration);
	return 0;
}

PySequenceMethods k3d_bitmap_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_bitmap_item, /* sq_item */
	0, /* sq_slice */
	k3d_bitmap_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyTypeObject k3d_bitmap_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.bitmap", /* tp_name */
	sizeof(k3d_bitmap), /* tp_basicsize */
	0, /* tp_itemsize */
	0, /* tp_dealloc */
	k3d_bitmap_print, /* tp_print */
	k3d_bitmap_getattr, /* tp_getattr */
	k3d_bitmap_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	0, /* tp_as_number */
	&k3d_bitmap_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_CHECKTYPES,
	"K-3D bitmap",
	0, /* tp_traverse */
	0, /* tp_clear */
	0, /* tp_richcompare */
	0, /* tp_weaklistoffset */
	k3d_bitmap_getiter, /* tp_iter */
	k3d_bitmap_iternext, /* tp_iternext */
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_bitmap_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_vector3

extern PyTypeObject k3d_vector3_type;

struct k3d_vector3
{
	PyObject_HEAD
	k3d::vector3* object;
};

int k3d_vector3_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::vector3* const value = python_cast<k3d::vector3*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, %f, %f)", (*value)[0], (*value)[1], (*value)[2]);

	return 0;
}

PyObject* k3d_vector3_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_vector3*>(self)->object = new k3d::vector3();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_vector3_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_vector3*>(Object)->object;
	reinterpret_cast<k3d_vector3*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_vector3_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_vector3_getattr(PyObject* Object, char* Name)
{
	return Py_FindMethod(k3d_vector3_methods, Object, Name);
}

int k3d_vector3_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_vector3_item(PyObject* Object, int Index)
{
	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(Index <= 2, 0);

	k3d::vector3* const value = python_cast<k3d::vector3*>(Object);
	return_val_if_fail(value, 0);

	return python_wrap((*value)[Index]);
}

int k3d_vector3_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	return_val_if_fail(0 <= Index, -1);
	return_val_if_fail(Index <= 2, -1);
	return_val_if_fail(Value, -1);

	k3d::vector3* const value = python_cast<k3d::vector3*>(Object);
	return_val_if_fail(value, -1);

	(*value)[Index] = PyFloat_AsDouble(Value);

	return 0;
}

PyObject* k3d_vector3_add(PyObject* LHS, PyObject* RHS)
{
	if(k3d::vector3* const lhs = python_cast<k3d::vector3*>(LHS))
	{
		if(k3d::vector3* const rhs = python_cast<k3d::vector3*>(RHS))
			return python_wrap((*lhs) + (*rhs));

		if(k3d::normal3* const rhs = python_cast<k3d::normal3*>(RHS))
			return python_wrap((*lhs) + (*rhs));
	}

	assert_not_reached();
	return 0;
}

PyObject* k3d_vector3_subtract(PyObject* LHS, PyObject* RHS)
{
	if(k3d::vector3* const lhs = python_cast<k3d::vector3*>(LHS))
	{
		if(k3d::vector3* const rhs = python_cast<k3d::vector3*>(RHS))
			return python_wrap(k3d::to_normal((*lhs) - (*rhs)));
	}

	assert_not_reached();
	return 0;
}

PyObject* k3d_vector3_multiply(PyObject* LHS, PyObject* RHS)
{
	if(k3d::vector3* const lhs = python_cast<k3d::vector3*>(LHS))
		return python_wrap((*lhs) * python_cast<double>(RHS));
	if(k3d::vector3* const rhs = python_cast<k3d::vector3*>(RHS))
		return python_wrap(python_cast<double>(LHS) * (*rhs));

	return 0;
}

PyObject* python_wrap(const k3d::vector3& Object)
{
	k3d_vector3* const result = PyObject_New(k3d_vector3, &k3d_vector3_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::vector3(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::vector3* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_vector3_type))
		return 0;

	return reinterpret_cast<k3d_vector3*>(Object)->object;
}

PySequenceMethods k3d_vector3_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_vector3_item, /* sq_item */
	0, /* sq_slice */
	k3d_vector3_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyNumberMethods k3d_vector3_number_methods =
{
	k3d_vector3_add, // binaryfunc nb_add;
	k3d_vector3_subtract, // binaryfunc nb_subtract;
	k3d_vector3_multiply, // binaryfunc nb_multiply;
	0, // binaryfunc nb_divide;
	0, // binaryfunc nb_remainder;
	0, // binaryfunc nb_divmod;
	0, // ternaryfunc nb_power;
	0, // unaryfunc nb_negative;
	0, // unaryfunc nb_positive;
	0, // unaryfunc nb_absolute;
	0, // inquiry nb_nonzero;
	0, // unaryfunc nb_invert;
	0, // binaryfunc nb_lshift;
	0, // binaryfunc nb_rshift;
	0, // binaryfunc nb_and;
	0, // binaryfunc nb_xor;
	0, // binaryfunc nb_or;
	0, // coercion nb_coerce;
	0, // unaryfunc nb_int;
	0, // unaryfunc nb_long;
	0, // unaryfunc nb_float;
	0, // unaryfunc nb_oct;
	0, // unaryfunc nb_hex;
	0, // binaryfunc nb_inplace_add;
	0, // binaryfunc nb_inplace_subtract;
	0, // binaryfunc nb_inplace_multiply;
	0, // binaryfunc nb_inplace_divide;
	0, // binaryfunc nb_inplace_remainder;
	0, // ternaryfunc nb_inplace_power;
	0, // binaryfunc nb_inplace_lshift;
	0, // binaryfunc nb_inplace_rshift;
	0, // binaryfunc nb_inplace_and;
	0, // binaryfunc nb_inplace_xor;
	0, // binaryfunc nb_inplace_or;
	0, // binaryfunc nb_floor_divide;
	0, // binaryfunc nb_true_divide;
	0, // binaryfunc nb_inplace_floor_divide;
	0, // binaryfunc nb_inplace_true_divide;
};

PyTypeObject k3d_vector3_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.vector3", /* tp_name */
	sizeof(k3d_vector3), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_vector3_dealloc, /* tp_dealloc */
	k3d_vector3_print, /* tp_print */
	k3d_vector3_getattr, /* tp_getattr */
	k3d_vector3_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	&k3d_vector3_number_methods, /* tp_as_number */
	&k3d_vector3_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_CHECKTYPES,
	"K-3D vector3",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_vector3_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_normal3

extern PyTypeObject k3d_normal3_type;

struct k3d_normal3
{
	PyObject_HEAD
	k3d::normal3* object;
};

int k3d_normal3_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::normal3* const value = python_cast<k3d::normal3*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, %f, %f)", (*value)[0], (*value)[1], (*value)[2]);

	return 0;
}

PyObject* k3d_normal3_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_normal3*>(self)->object = new k3d::normal3();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_normal3_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_normal3*>(Object)->object;
	reinterpret_cast<k3d_normal3*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_normal3_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_normal3_getattr(PyObject* Object, char* Name)
{
	return Py_FindMethod(k3d_normal3_methods, Object, Name);
}

int k3d_normal3_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_normal3_item(PyObject* Object, int Index)
{
	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(Index <= 2, 0);

	k3d::normal3* const value = python_cast<k3d::normal3*>(Object);
	return_val_if_fail(value, 0);

	return python_wrap((*value)[Index]);
}

int k3d_normal3_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	return_val_if_fail(0 <= Index, -1);
	return_val_if_fail(Index <= 2, -1);
	return_val_if_fail(Value, -1);

	k3d::normal3* const value = python_cast<k3d::normal3*>(Object);
	return_val_if_fail(value, -1);

	(*value)[Index] = PyFloat_AsDouble(Value);

	return 0;
}

PyObject* k3d_normal3_add(PyObject* LHS, PyObject* RHS)
{
	k3d::normal3* const lhs = python_cast<k3d::normal3*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::normal3* const rhs = python_cast<k3d::normal3*>(RHS);
	return_val_if_fail(rhs, 0);

	return python_wrap((*lhs) + (*rhs));
}

PyObject* k3d_normal3_subtract(PyObject* LHS, PyObject* RHS)
{
	k3d::normal3* const lhs = python_cast<k3d::normal3*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::normal3* const rhs = python_cast<k3d::normal3*>(RHS);
	return_val_if_fail(rhs, 0);

	return python_wrap((*lhs) - (*rhs));
}

PyObject* k3d_normal3_multiply(PyObject* LHS, PyObject* RHS)
{
	if(k3d::normal3* const lhs = python_cast<k3d::normal3*>(LHS))
		return python_wrap((*lhs) * python_cast<double>(RHS));
	if(k3d::normal3* const rhs = python_cast<k3d::normal3*>(RHS))
		return python_wrap(python_cast<double>(LHS) * (*rhs));

	return 0;
}

PyObject* python_wrap(const k3d::normal3& Object)
{
	k3d_normal3* const result = PyObject_New(k3d_normal3, &k3d_normal3_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::normal3(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::normal3* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_normal3_type))
		return 0;

	return reinterpret_cast<k3d_normal3*>(Object)->object;
}

PyNumberMethods k3d_normal3_number_methods =
{
	k3d_normal3_add, // binaryfunc nb_add;
	k3d_normal3_subtract, // binaryfunc nb_subtract;
	k3d_normal3_multiply, // binaryfunc nb_multiply;
	0, // binaryfunc nb_divide;
	0, // binaryfunc nb_remainder;
	0, // binaryfunc nb_divmod;
	0, // ternaryfunc nb_power;
	0, // unaryfunc nb_negative;
	0, // unaryfunc nb_positive;
	0, // unaryfunc nb_absolute;
	0, // inquiry nb_nonzero;
	0, // unaryfunc nb_invert;
	0, // binaryfunc nb_lshift;
	0, // binaryfunc nb_rshift;
	0, // binaryfunc nb_and;
	0, // binaryfunc nb_xor;
	0, // binaryfunc nb_or;
	0, // coercion nb_coerce;
	0, // unaryfunc nb_int;
	0, // unaryfunc nb_long;
	0, // unaryfunc nb_float;
	0, // unaryfunc nb_oct;
	0, // unaryfunc nb_hex;
	0, // binaryfunc nb_inplace_add;
	0, // binaryfunc nb_inplace_subtract;
	0, // binaryfunc nb_inplace_multiply;
	0, // binaryfunc nb_inplace_divide;
	0, // binaryfunc nb_inplace_remainder;
	0, // ternaryfunc nb_inplace_power;
	0, // binaryfunc nb_inplace_lshift;
	0, // binaryfunc nb_inplace_rshift;
	0, // binaryfunc nb_inplace_and;
	0, // binaryfunc nb_inplace_xor;
	0, // binaryfunc nb_inplace_or;
	0, // binaryfunc nb_floor_divide;
	0, // binaryfunc nb_true_divide;
	0, // binaryfunc nb_inplace_floor_divide;
	0, // binaryfunc nb_inplace_true_divide;
};

PySequenceMethods k3d_normal3_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_normal3_item, /* sq_item */
	0, /* sq_slice */
	k3d_normal3_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyTypeObject k3d_normal3_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.normal3", /* tp_name */
	sizeof(k3d_normal3), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_normal3_dealloc, /* tp_dealloc */
	k3d_normal3_print, /* tp_print */
	k3d_normal3_getattr, /* tp_getattr */
	k3d_normal3_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	&k3d_normal3_number_methods, /* tp_as_number */
	&k3d_normal3_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_CHECKTYPES,
	"K-3D normal3",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_normal3_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_angle_axis

extern PyTypeObject k3d_angle_axis_type;

struct k3d_angle_axis
{
	PyObject_HEAD
	k3d::angle_axis* object;
};

int k3d_angle_axis_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::angle_axis* const value = python_cast<k3d::angle_axis*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, (%f, %f, %f))", k3d::degrees(value->angle), value->axis[0], value->axis[1], value->axis[2]);

	return 0;
}

PyObject* k3d_angle_axis_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_angle_axis*>(self)->object = new k3d::angle_axis();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_angle_axis_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_angle_axis*>(Object)->object;
	reinterpret_cast<k3d_angle_axis*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_angle_axis_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_angle_axis_getattr(PyObject* Object, char* Name)
{
	k3d::angle_axis* const value = python_cast<k3d::angle_axis*>(Object);
	return_val_if_fail(value, 0);

	const std::string name = Name;

	if(name == "angle")
		return python_wrap(k3d::degrees(value->angle));

	if(name == "axis")
		return python_wrap(k3d::to_normal(value->axis));

	return Py_FindMethod(k3d_angle_axis_methods, Object, Name);
}

int k3d_angle_axis_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* python_wrap(const k3d::angle_axis& Object)
{
	k3d_angle_axis* const result = PyObject_New(k3d_angle_axis, &k3d_angle_axis_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::angle_axis(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::angle_axis* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_angle_axis_type))
		return 0;

	return reinterpret_cast<k3d_angle_axis*>(Object)->object;
}

PyTypeObject k3d_angle_axis_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.angle_axis", /* tp_name */
	sizeof(k3d_angle_axis), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_angle_axis_dealloc, /* tp_dealloc */
	k3d_angle_axis_print, /* tp_print */
	k3d_angle_axis_getattr, /* tp_getattr */
	k3d_angle_axis_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	0, /* tp_as_number */
	0, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT,
	"K-3D angle_axis",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_angle_axis_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_euler_angles

extern PyTypeObject k3d_euler_angles_type;

struct k3d_euler_angles
{
	PyObject_HEAD
	k3d::euler_angles* object;
};

int k3d_euler_angles_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(\"");

	switch(value->order)
	{
		case k3d::euler_angles::XYZstatic:
			fprintf(fp, "xyz");
			break;
		case k3d::euler_angles::XYXstatic:
			fprintf(fp, "xyx");
			break;
		case k3d::euler_angles::XZYstatic:
			fprintf(fp, "xzy");
			break;
		case k3d::euler_angles::XZXstatic:
			fprintf(fp, "xzx");
			break;
		case k3d::euler_angles::YZXstatic:
			fprintf(fp, "yzx");
			break;
		case k3d::euler_angles::YZYstatic:
			fprintf(fp, "yzy");
			break;
		case k3d::euler_angles::YXZstatic:
			fprintf(fp, "yxz");
			break;
		case k3d::euler_angles::YXYstatic:
			fprintf(fp, "yxy");
			break;
		case k3d::euler_angles::ZXYstatic:
			fprintf(fp, "zxy");
			break;
		case k3d::euler_angles::ZXZstatic:
			fprintf(fp, "zxz");
			break;
		case k3d::euler_angles::ZYXstatic:
			fprintf(fp, "zyx");
			break;
		case k3d::euler_angles::ZYZstatic:
			fprintf(fp, "zyz");
			break;
		case k3d::euler_angles::ZYXrotating:
			fprintf(fp, "zyx rotating");
			break;
		case k3d::euler_angles::XYXrotating :
			fprintf(fp, "xyx rotating");
			break;
		case k3d::euler_angles::YZXrotating:
			fprintf(fp, "yzx rotating");
			break;
		case k3d::euler_angles::XZXrotating :
			fprintf(fp, "xzx rotating");
			break;
		case k3d::euler_angles::XZYrotating:
			fprintf(fp, "xzy rotating");
			break;
		case k3d::euler_angles::YZYrotating :
			fprintf(fp, "yzy rotating");
			break;
		case k3d::euler_angles::ZXYrotating:
			fprintf(fp, "zxy rotating");
			break;
		case k3d::euler_angles::YXYrotating :
			fprintf(fp, "yxy rotating");
			break;
		case k3d::euler_angles::YXZrotating:
			fprintf(fp, "yxz rotating");
			break;
		case k3d::euler_angles::ZXZrotating :
			fprintf(fp, "zxz rotating");
			break;
		case k3d::euler_angles::XYZrotating:
			fprintf(fp, "xyz rotating");
			break;
		case k3d::euler_angles::ZYZrotating:
			fprintf(fp, "zyz rotating");
			break;
	}

	fprintf(fp, "\", %f, %f, %f", k3d::degrees((*value)[0]), k3d::degrees((*value)[1]), k3d::degrees((*value)[2]));

	fprintf(fp, ")");

	return 0;
}

PyObject* k3d_euler_angles_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_euler_angles*>(self)->object = new k3d::euler_angles();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_euler_angles_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_euler_angles*>(Object)->object;
	reinterpret_cast<k3d_euler_angles*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_euler_angles_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_euler_angles_getattr(PyObject* Object, char* Name)
{
	k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(Object);
	return_val_if_fail(value, 0);

	const std::string name = Name;

	return Py_FindMethod(k3d_euler_angles_methods, Object, Name);
}

int k3d_euler_angles_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_euler_angles_item(PyObject* Object, int Index)
{
	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(Index <= 2, 0);

	k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(Object);
	return_val_if_fail(value, 0);

	return python_wrap((*value)[Index]);
}

int k3d_euler_angles_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	return_val_if_fail(0 <= Index, -1);
	return_val_if_fail(Index <= 2, -1);
	return_val_if_fail(Value, -1);

	k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(Object);
	return_val_if_fail(value, -1);

	(*value)[Index] = PyFloat_AsDouble(Value);

	return 0;
}

PyObject* k3d_euler_angles_add(PyObject* LHS, PyObject* RHS)
{
	k3d::euler_angles* const lhs = python_cast<k3d::euler_angles*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::euler_angles* const rhs = python_cast<k3d::euler_angles*>(RHS);
	return_val_if_fail(rhs, 0);

	return_val_if_fail(lhs->order == rhs->order, 0);

	return python_wrap(k3d::euler_angles((*lhs)[0] + (*rhs)[0], (*lhs)[1] + (*rhs)[1], (*lhs)[2] + (*rhs)[2], lhs->order));
}

PyObject* k3d_euler_angles_multiply(PyObject* LHS, PyObject* RHS)
{
	if(k3d::euler_angles* const lhs = python_cast<k3d::euler_angles*>(LHS))
	{
		const double rhs = python_cast<double>(RHS);
		return python_wrap(k3d::euler_angles((*lhs)[0] * rhs, (*lhs)[1] * rhs, (*lhs)[2] * rhs, lhs->order));
	}
	if(k3d::euler_angles* const rhs = python_cast<k3d::euler_angles*>(RHS))
	{
		const double lhs = python_cast<double>(LHS);
		return python_wrap(k3d::euler_angles((*rhs)[0] * lhs, (*rhs)[1] * lhs, (*rhs)[2] * lhs, rhs->order));
	}

	return 0;
}

PyObject* python_wrap(const k3d::euler_angles& Object)
{
	k3d_euler_angles* const result = PyObject_New(k3d_euler_angles, &k3d_euler_angles_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::euler_angles(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::euler_angles* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_euler_angles_type))
		return 0;

	return reinterpret_cast<k3d_euler_angles*>(Object)->object;
}

PyNumberMethods k3d_euler_angles_number_methods =
{
	k3d_euler_angles_add, // binaryfunc nb_add;
	0, // binaryfunc nb_subtract;
	k3d_euler_angles_multiply, // binaryfunc nb_multiply;
	0, // binaryfunc nb_divide;
	0, // binaryfunc nb_remainder;
	0, // binaryfunc nb_divmod;
	0, // ternaryfunc nb_power;
	0, // unaryfunc nb_negative;
	0, // unaryfunc nb_positive;
	0, // unaryfunc nb_absolute;
	0, // inquiry nb_nonzero;
	0, // unaryfunc nb_invert;
	0, // binaryfunc nb_lshift;
	0, // binaryfunc nb_rshift;
	0, // binaryfunc nb_and;
	0, // binaryfunc nb_xor;
	0, // binaryfunc nb_or;
	0, // coercion nb_coerce;
	0, // unaryfunc nb_int;
	0, // unaryfunc nb_long;
	0, // unaryfunc nb_float;
	0, // unaryfunc nb_oct;
	0, // unaryfunc nb_hex;
	0, // binaryfunc nb_inplace_add;
	0, // binaryfunc nb_inplace_subtract;
	0, // binaryfunc nb_inplace_multiply;
	0, // binaryfunc nb_inplace_divide;
	0, // binaryfunc nb_inplace_remainder;
	0, // ternaryfunc nb_inplace_power;
	0, // binaryfunc nb_inplace_lshift;
	0, // binaryfunc nb_inplace_rshift;
	0, // binaryfunc nb_inplace_and;
	0, // binaryfunc nb_inplace_xor;
	0, // binaryfunc nb_inplace_or;
	0, // binaryfunc nb_floor_divide;
	0, // binaryfunc nb_true_divide;
	0, // binaryfunc nb_inplace_floor_divide;
	0, // binaryfunc nb_inplace_true_divide;
};

PySequenceMethods k3d_euler_angles_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_euler_angles_item, /* sq_item */
	0, /* sq_slice */
	k3d_euler_angles_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyTypeObject k3d_euler_angles_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.euler_angles", /* tp_name */
	sizeof(k3d_euler_angles), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_euler_angles_dealloc, /* tp_dealloc */
	k3d_euler_angles_print, /* tp_print */
	k3d_euler_angles_getattr, /* tp_getattr */
	k3d_euler_angles_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	&k3d_euler_angles_number_methods, /* tp_as_number */
	&k3d_euler_angles_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_CHECKTYPES,
	"K-3D euler_angles",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_euler_angles_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_vector4

extern PyTypeObject k3d_vector4_type;

struct k3d_vector4
{
	PyObject_HEAD
	k3d::vector4* object;
};

int k3d_vector4_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::vector4* const value = python_cast<k3d::vector4*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, %f, %f, %f)", (*value)[0], (*value)[1], (*value)[2], (*value)[3]);

	return 0;
}

PyObject* k3d_vector4_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_vector4*>(self)->object = new k3d::vector4();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_vector4_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_vector4*>(Object)->object;
	reinterpret_cast<k3d_vector4*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_vector4_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_vector4_getattr(PyObject* Object, char* Name)
{
	return Py_FindMethod(k3d_vector4_methods, Object, Name);
}

int k3d_vector4_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_vector4_item(PyObject* Object, int Index)
{
	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(Index <= 3, 0);

	k3d::vector4* const value = python_cast<k3d::vector4*>(Object);
	return_val_if_fail(value, 0);

	return python_wrap((*value)[Index]);
}

int k3d_vector4_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	return_val_if_fail(0 <= Index, -1);
	return_val_if_fail(Index <= 3, -1);
	return_val_if_fail(Value, -1);

	k3d::vector4* const value = python_cast<k3d::vector4*>(Object);
	return_val_if_fail(value, -1);

	(*value)[Index] = PyFloat_AsDouble(Value);

	return 0;
}

PyObject* python_wrap(const k3d::vector4& Object)
{
	k3d_vector4* const result = PyObject_New(k3d_vector4, &k3d_vector4_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::vector4(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::vector4* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_vector4_type))
		return 0;

	return reinterpret_cast<k3d_vector4*>(Object)->object;
}

PySequenceMethods k3d_vector4_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_vector4_item, /* sq_item */
	0, /* sq_slice */
	k3d_vector4_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyTypeObject k3d_vector4_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.vector4", /* tp_name */
	sizeof(k3d_vector4), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_vector4_dealloc, /* tp_dealloc */
	k3d_vector4_print, /* tp_print */
	k3d_vector4_getattr, /* tp_getattr */
	k3d_vector4_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	0, /* tp_as_number */
	&k3d_vector4_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT,
	"K-3D vector4",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_vector4_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_matrix4

extern PyTypeObject k3d_matrix4_type;

struct k3d_matrix4
{
	PyObject_HEAD
	k3d::matrix4* object;
};

int k3d_matrix4_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::matrix4* const value = python_cast<k3d::matrix4*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, %f, %f, %f)\n", (*value)[0][0], (*value)[0][1], (*value)[0][2], (*value)[0][3]);
	fprintf(fp, "(%f, %f, %f, %f)\n", (*value)[1][0], (*value)[1][1], (*value)[1][2], (*value)[1][3]);
	fprintf(fp, "(%f, %f, %f, %f)\n", (*value)[2][0], (*value)[2][1], (*value)[2][2], (*value)[2][3]);
	fprintf(fp, "(%f, %f, %f, %f)", (*value)[3][0], (*value)[3][1], (*value)[3][2], (*value)[3][3]);

	return 0;
}

PyObject* k3d_matrix4_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_matrix4*>(self)->object = new k3d::matrix4();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_matrix4_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_matrix4*>(Object)->object;
	reinterpret_cast<k3d_matrix4*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_matrix4_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_matrix4_getattr(PyObject* Object, char* Name)
{
	return Py_FindMethod(k3d_matrix4_methods, Object, Name);
}

int k3d_matrix4_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_matrix4_item(PyObject* Object, int Index)
{
	return_val_if_fail(0 <= Index, 0);
	return_val_if_fail(Index <= 3, 0);

	k3d::matrix4* const value = python_cast<k3d::matrix4*>(Object);
	return_val_if_fail(value, 0);

	return python_wrap((*value)[Index]);
}

int k3d_matrix4_assign_item(PyObject* Object, int Index, PyObject* Value)
{
	return -1;
}

PyObject* k3d_matrix4_multiply(PyObject* LHS, PyObject* RHS)
{
	k3d::matrix4* const lhs = python_cast<k3d::matrix4*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::matrix4* const rhs = python_cast<k3d::matrix4*>(RHS);
	return_val_if_fail(rhs, 0);

	return python_wrap((*lhs) * (*rhs));
}

PyObject* python_wrap(const k3d::matrix4& Object)
{
	k3d_matrix4* const result = PyObject_New(k3d_matrix4, &k3d_matrix4_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::matrix4(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::matrix4* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_matrix4_type))
		return 0;

	return reinterpret_cast<k3d_matrix4*>(Object)->object;
}

PyNumberMethods k3d_matrix4_number_methods =
{
	0, // binaryfunc nb_add;
	0, // binaryfunc nb_subtract;
	k3d_matrix4_multiply, // binaryfunc nb_multiply;
	0, // binaryfunc nb_divide;
	0, // binaryfunc nb_remainder;
	0, // binaryfunc nb_divmod;
	0, // ternaryfunc nb_power;
	0, // unaryfunc nb_negative;
	0, // unaryfunc nb_positive;
	0, // unaryfunc nb_absolute;
	0, // inquiry nb_nonzero;
	0, // unaryfunc nb_invert;
	0, // binaryfunc nb_lshift;
	0, // binaryfunc nb_rshift;
	0, // binaryfunc nb_and;
	0, // binaryfunc nb_xor;
	0, // binaryfunc nb_or;
	0, // coercion nb_coerce;
	0, // unaryfunc nb_int;
	0, // unaryfunc nb_long;
	0, // unaryfunc nb_float;
	0, // unaryfunc nb_oct;
	0, // unaryfunc nb_hex;
	0, // binaryfunc nb_inplace_add;
	0, // binaryfunc nb_inplace_subtract;
	0, // binaryfunc nb_inplace_multiply;
	0, // binaryfunc nb_inplace_divide;
	0, // binaryfunc nb_inplace_remainder;
	0, // ternaryfunc nb_inplace_power;
	0, // binaryfunc nb_inplace_lshift;
	0, // binaryfunc nb_inplace_rshift;
	0, // binaryfunc nb_inplace_and;
	0, // binaryfunc nb_inplace_xor;
	0, // binaryfunc nb_inplace_or;
	0, // binaryfunc nb_floor_divide;
	0, // binaryfunc nb_true_divide;
	0, // binaryfunc nb_inplace_floor_divide;
	0, // binaryfunc nb_inplace_true_divide;
};

PySequenceMethods k3d_matrix4_sequence_methods =
{
	0, /* sq_length */
	0, /* sq_concat */
	0, /* sq_repeat */
	k3d_matrix4_item, /* sq_item */
	0, /* sq_slice */
	k3d_matrix4_assign_item, /* sq_ass_item */
	0, /* sq_ass_slice */
	0, /* sq_containers */
	0, /* sq_inplace_concat */
	0, /* sq_inplace_repeat */
};

PyTypeObject k3d_matrix4_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.matrix4", /* tp_name */
	sizeof(k3d_matrix4), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_matrix4_dealloc, /* tp_dealloc */
	k3d_matrix4_print, /* tp_print */
	k3d_matrix4_getattr, /* tp_getattr */
	k3d_matrix4_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	&k3d_matrix4_number_methods, /* tp_as_number */
	&k3d_matrix4_sequence_methods, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT,
	"K-3D matrix4",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_matrix4_new,
};

/////////////////////////////////////////////////////////////////////////////
// k3d_color

extern PyTypeObject k3d_color_type;

struct k3d_color
{
	PyObject_HEAD
	k3d::color* object;
};

int k3d_color_print(PyObject* Object, FILE *fp, int flags)
{
	k3d::color* const value = python_cast<k3d::color*>(Object);
	return_val_if_fail(value, -1);

	fprintf(fp, "(%f, %f, %f)", value->red, value->green, value->blue);

	return 0;
}

PyObject* k3d_color_new(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
	PyObject* self = type->tp_alloc(type, 0);
	return_val_if_fail(self, 0);
	reinterpret_cast<k3d_color*>(self)->object = new k3d::color();
	return reinterpret_cast<PyObject*>(self);
}

void k3d_color_dealloc(PyObject* Object)
{
	delete reinterpret_cast<k3d_color*>(Object)->object;
	reinterpret_cast<k3d_color*>(Object)->ob_type->tp_free(Object);
}

PyMethodDef k3d_color_methods[] =
{
	{NULL, NULL, 0, NULL}
};

PyObject* k3d_color_getattr(PyObject* Object, char* Name)
{
	k3d::color* const value = python_cast<k3d::color*>(Object);
	return_val_if_fail(value, 0);

	const std::string name = Name;
	if(name == "red")
		return python_wrap(value->red);
	if(name == "green")
		return python_wrap(value->green);
	if(name == "blue")
		return python_wrap(value->blue);

	return Py_FindMethod(k3d_color_methods, Object, Name);
}

int k3d_color_setattr(PyObject* Object, char* Name, PyObject* Value)
{
	return -1;
}

PyObject* k3d_color_add(PyObject* LHS, PyObject* RHS)
{
	k3d::color* const lhs = python_cast<k3d::color*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::color* const rhs = python_cast<k3d::color*>(RHS);
	return_val_if_fail(rhs, 0);

	return python_wrap((*lhs) + (*rhs));
}

PyObject* k3d_color_subtract(PyObject* LHS, PyObject* RHS)
{
	k3d::color* const lhs = python_cast<k3d::color*>(LHS);
	return_val_if_fail(lhs, 0);

	k3d::color* const rhs = python_cast<k3d::color*>(RHS);
	return_val_if_fail(rhs, 0);

	return python_wrap((*lhs) - (*rhs));
}

PyObject* k3d_color_multiply(PyObject* LHS, PyObject* RHS)
{
	if(k3d::color* const lhs = python_cast<k3d::color*>(LHS))
		return python_wrap((*lhs) * python_cast<double>(RHS));
	if(k3d::color* const rhs = python_cast<k3d::color*>(RHS))
		return python_wrap(python_cast<double>(LHS) * (*rhs));

	return 0;
}

PyObject* python_wrap(const k3d::color& Object)
{
	k3d_color* const result = PyObject_New(k3d_color, &k3d_color_type);
	return_val_if_fail(result, 0);
	result->object = new k3d::color(Object);
	return reinterpret_cast<PyObject*>(result);
}

template<>
k3d::color* python_cast(PyObject* Object)
{
	if(!Object)
		return 0;

	if(!PyObject_TypeCheck(Object, &k3d_color_type))
		return 0;

	return reinterpret_cast<k3d_color*>(Object)->object;
}

PyNumberMethods k3d_color_number_methods =
{
	k3d_color_add, // binaryfunc nb_add;
	k3d_color_subtract, // binaryfunc nb_subtract;
	k3d_color_multiply, // binaryfunc nb_multiply;
	0, // binaryfunc nb_divide;
	0, // binaryfunc nb_remainder;
	0, // binaryfunc nb_divmod;
	0, // ternaryfunc nb_power;
	0, // unaryfunc nb_negative;
	0, // unaryfunc nb_positive;
	0, // unaryfunc nb_absolute;
	0, // inquiry nb_nonzero;
	0, // unaryfunc nb_invert;
	0, // binaryfunc nb_lshift;
	0, // binaryfunc nb_rshift;
	0, // binaryfunc nb_and;
	0, // binaryfunc nb_xor;
	0, // binaryfunc nb_or;
	0, // coercion nb_coerce;
	0, // unaryfunc nb_int;
	0, // unaryfunc nb_long;
	0, // unaryfunc nb_float;
	0, // unaryfunc nb_oct;
	0, // unaryfunc nb_hex;
	0, // binaryfunc nb_inplace_add;
	0, // binaryfunc nb_inplace_subtract;
	0, // binaryfunc nb_inplace_multiply;
	0, // binaryfunc nb_inplace_divide;
	0, // binaryfunc nb_inplace_remainder;
	0, // ternaryfunc nb_inplace_power;
	0, // binaryfunc nb_inplace_lshift;
	0, // binaryfunc nb_inplace_rshift;
	0, // binaryfunc nb_inplace_and;
	0, // binaryfunc nb_inplace_xor;
	0, // binaryfunc nb_inplace_or;
	0, // binaryfunc nb_floor_divide;
	0, // binaryfunc nb_true_divide;
	0, // binaryfunc nb_inplace_floor_divide;
	0, // binaryfunc nb_inplace_true_divide;
};

PyTypeObject k3d_color_type =
{
	PyObject_HEAD_INIT(NULL)
	0,
	"k3d.color", /* tp_name */
	sizeof(k3d_color), /* tp_basicsize */
	0, /* tp_itemsize */
	k3d_color_dealloc, /* tp_dealloc */
	k3d_color_print, /* tp_print */
	k3d_color_getattr, /* tp_getattr */
	k3d_color_setattr, /* tp_setattr */
	0, /* tp_compare */
	0, /* tp_repr */
	&k3d_color_number_methods, /* tp_as_number */
	0, /* tp_as_sequence */
	0, /* tp_as_mapping */
	0,
	0,
	0,
	0,
	0,
	0,
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_CHECKTYPES,
	"K-3D color",
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	k3d_color_new,
};

/////////////////////////////////////////////////////////////////////////////
// init_object_model

PyObject* k3d_module_deep_copy(PyObject* self, PyObject* args)
{
	PyObject* object1 = 0;
	PyObject* object2 = 0;
	if(!PyArg_ParseTuple(args, "OO", &object1, &object2))
		return 0;

	k3d::mesh* const input = python_cast<k3d::mesh*>(object1);
	k3d::mesh* const output = python_cast<k3d::mesh*>(object2);
	return_val_if_fail(input && output, 0);

	k3d::deep_copy(*input, *output);

	return PYVAL_NONE;
}

PyObject* k3d_module_vector3(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "ddd", &d1, &d2, &d3))
		return python_wrap(k3d::vector3(d1, d2, d3));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
		{
			k3d::vector3* const value = python_cast<k3d::vector3*>(o1);
			if(value)
				return python_wrap(*value);
		}
	PyErr_Clear();

	return python_wrap(k3d::vector3(0, 0, 0));
}

PyObject* k3d_module_normal3(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "ddd", &d1, &d2, &d3))
		return python_wrap(k3d::normal3(d1, d2, d3));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
		{
			k3d::normal3* const value = python_cast<k3d::normal3*>(o1);
			if(value)
				return python_wrap(*value);
		}
	PyErr_Clear();

	return python_wrap(k3d::normal3(0, 0, 0));
}

PyObject* k3d_module_angle_axis(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	double d4 = 0;
	if(PyArg_ParseTuple(args, "dddd", &d1, &d2, &d3, &d4))
		return python_wrap(k3d::angle_axis(k3d::radians(d1), k3d::vector3(d2, d3, d4)));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "dO", &d1, &o1))
	{
		k3d::normal3* const value = python_cast<k3d::normal3*>(o1);
		if(value)
			return python_wrap(k3d::angle_axis(k3d::radians(d1), k3d::to_vector(*value)));
	}
	PyErr_Clear();

	return python_wrap(k3d::angle_axis(0, k3d::vector3(0, 0, 1)));
}

PyObject* k3d_module_euler_angles(PyObject* self, PyObject* args)
{
	const char* s1 = 0;
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "sddd", &s1, &d1, &d2, &d3))
	{
		const std::string order = s1;

		if(order == "xyz")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::XYZstatic));
		if(order == "xyx")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::XYXstatic));
		if(order == "xzy")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::XZYstatic));
		if(order == "xzx")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::XZXstatic));
		if(order == "yzx")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::YZXstatic));
		if(order == "yzy")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::YZYstatic));
		if(order == "yxz")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::YXZstatic));
		if(order == "yxy")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::YXYstatic));
		if(order == "zxy")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::ZXYstatic));
		if(order == "zxz")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::ZXZstatic));
		if(order == "zyx")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::ZYXstatic));
		if(order == "zyz")
			return python_wrap(k3d::euler_angles(k3d::radians(d1), k3d::radians(d2), k3d::radians(d3), k3d::euler_angles::ZYZstatic));

		return_val_if_fail(0, 0);
	}
	PyErr_Clear();

	return python_wrap(k3d::euler_angles(0, 0, 0, k3d::euler_angles::ZXYstatic));
}

PyObject* k3d_module_vector4(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	double d4 = 0;
	if(PyArg_ParseTuple(args, "dddd", &d1, &d2, &d3, &d4))
		return python_wrap(k3d::vector4(d1, d2, d3, d4));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		k3d::vector4* const value = python_cast<k3d::vector4*>(o1);
		if(value)
			return python_wrap(k3d::vector4(*value));
	}
	PyErr_Clear();

	return python_wrap(k3d::vector4(0, 0, 0, 0));
}

PyObject* k3d_module_matrix4(PyObject* self, PyObject* args)
{
	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		k3d::matrix4* const value = python_cast<k3d::matrix4*>(o1);
		if(value)
			return python_wrap(k3d::matrix4(*value));
	}
	PyErr_Clear();

	return python_wrap(k3d::matrix4(
		k3d::vector4(0, 0, 0, 0),
		k3d::vector4(0, 0, 0, 0),
		k3d::vector4(0, 0, 0, 0),
		k3d::vector4(0, 0, 0, 0)));
}

PyObject* k3d_module_color(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "ddd", &d1, &d2, &d3))
		return python_wrap(k3d::color(d1, d2, d3));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		k3d::color* const value = python_cast<k3d::color*>(o1);
		if(value)
			return python_wrap(k3d::color(*value));
	}
	PyErr_Clear();

	return python_wrap(k3d::color(0, 0, 0));
}

PyObject* k3d_module_identity3(PyObject* self, PyObject* args)
{
	return python_wrap(k3d::identity3D());
}

PyObject* k3d_module_length(PyObject* self, PyObject* args)
{
	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		if(k3d::normal3* const normal3 = python_cast<k3d::normal3*>(o1))
			return python_wrap(k3d::length(*normal3));
	}

	return 0;
}

PyObject* k3d_module_translate3(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "ddd", &d1, &d2, &d3))
		return python_wrap(k3d::translation3D(k3d::normal3(d1, d2, d3)));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		if(k3d::normal3* const value = python_cast<k3d::normal3*>(o1))
			return python_wrap(k3d::translation3D(*value));

		if(k3d::vector3* const value = python_cast<k3d::vector3*>(o1))
			return python_wrap(k3d::translation3D(*value));
	}
	PyErr_Clear();

	return 0;
}

PyObject* k3d_module_rotate3(PyObject* self, PyObject* args)
{
	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		if(k3d::angle_axis* const value = python_cast<k3d::angle_axis*>(o1))
			return python_wrap(k3d::rotation3D(*value));

		if(k3d::euler_angles* const value = python_cast<k3d::euler_angles*>(o1))
			return python_wrap(k3d::rotation3D(k3d::quaternion(*value)));
	}
	return 0;
}

PyObject* k3d_module_scale3(PyObject* self, PyObject* args)
{
	double d1 = 0;
	double d2 = 0;
	double d3 = 0;
	if(PyArg_ParseTuple(args, "ddd", &d1, &d2, &d3))
		return python_wrap(k3d::scaling3D(k3d::vector3(d1, d2, d3)));
	PyErr_Clear();

	if(PyArg_ParseTuple(args, "d", &d1))
		return python_wrap(k3d::scaling3D(k3d::vector3(d1, d1, d1)));
	PyErr_Clear();

	PyObject* o1 = 0;
	if(PyArg_ParseTuple(args, "O", &o1))
	{
		k3d::vector3* const value = python_cast<k3d::vector3*>(o1);
		if(value)
			return python_wrap(k3d::scaling3D(*value));
	}
	PyErr_Clear();

	return 0;
}

PyMethodDef k3d_module_methods[] =
{
	{"angle_axis", k3d_module_angle_axis, METH_VARARGS, "Constructs an angle_axis object."},
	{"color", k3d_module_color, METH_VARARGS, "Constructs a color object."},
	{"deep_copy", k3d_module_deep_copy, METH_VARARGS, "Makes a deep copy from one mesh to another."},
	{"euler_angles", k3d_module_euler_angles, METH_VARARGS, "Constructs an euler_angles object."},
	{"identity3", k3d_module_identity3, METH_VARARGS, "Returns a 3-dimensional identity matrix."},
	{"length", k3d_module_length, METH_VARARGS, "Returns the length of a normal."},
	{"matrix4", k3d_module_matrix4, METH_VARARGS, "Constructs a matrix4 object."},
	{"normal3", k3d_module_normal3, METH_VARARGS, "Constructs a normal3 object."},
	{"rotate3", k3d_module_rotate3, METH_VARARGS, "Returns a 3-dimensional rotation matrix."},
	{"scale3", k3d_module_scale3, METH_VARARGS, "Returns a 3-dimensional scaling matrix."},
	{"translate3", k3d_module_translate3, METH_VARARGS, "Returns a 3-dimensional translation matrix."},
	{"vector3", k3d_module_vector3, METH_VARARGS, "Constructs a vector3 object."},
	{"vector4", k3d_module_vector4, METH_VARARGS, "Constructs a vector4 object."},
	{ NULL }
};

PyObject* init_object_model()
{
	if(PyType_Ready(&k3d_bicubic_patch_type) < 0) return 0;
	if(PyType_Ready(&k3d_bilinear_patch_type) < 0) return 0;
	if(PyType_Ready(&k3d_bitmap_type) < 0) return 0;
	if(PyType_Ready(&k3d_blobby_opcode_type) < 0) return 0;
	if(PyType_Ready(&k3d_blobby_type) < 0) return 0;
	if(PyType_Ready(&k3d_cubic_curve_group_type) < 0) return 0;
	if(PyType_Ready(&k3d_cubic_curve_type) < 0) return 0;
	if(PyType_Ready(&k3d_face_type) < 0) return 0;
	if(PyType_Ready(&k3d_iplugin_factory_type) < 0) return 0;
	if(PyType_Ready(&k3d_iproperty_type) < 0) return 0;
	if(PyType_Ready(&k3d_iunknown_type) < 0) return 0;
	if(PyType_Ready(&k3d_iuser_interface_type) < 0) return 0;
	if(PyType_Ready(&k3d_linear_curve_group_type) < 0) return 0;
	if(PyType_Ready(&k3d_linear_curve_type) < 0) return 0;
	if(PyType_Ready(&k3d_mesh_type) < 0) return 0;
	if(PyType_Ready(&k3d_parameters_type) < 0) return 0;
	if(PyType_Ready(&k3d_point_type) < 0) return 0;
	if(PyType_Ready(&k3d_point_group_type) < 0) return 0;
	if(PyType_Ready(&k3d_polyhedron_type) < 0) return 0;
	if(PyType_Ready(&k3d_split_edge_type) < 0) return 0;

	if(PyType_Ready(&k3d_vector3_type) < 0) return 0;
	if(PyType_Ready(&k3d_normal3_type) < 0) return 0;
	if(PyType_Ready(&k3d_angle_axis_type) < 0) return 0;
	if(PyType_Ready(&k3d_euler_angles_type) < 0) return 0;
	if(PyType_Ready(&k3d_vector4_type) < 0) return 0;
	if(PyType_Ready(&k3d_matrix4_type) < 0) return 0;
	if(PyType_Ready(&k3d_color_type) < 0) return 0;

	if(PyType_Ready(&k3d_bicubic_patch_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_bilinear_patch_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_blobby_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_cubic_curve_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_cubic_curve_group_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_face_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_linear_curve_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_linear_curve_group_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_parameters_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_point_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_polyhedron_collection_type) < 0) return 0;

	if(PyType_Ready(&k3d_bicubic_patch_control_point_collection_type) < 0) return 0;
	if(PyType_Ready(&k3d_bilinear_patch_control_point_collection_type) < 0) return 0;

	return Py_InitModule3("k3d", k3d_module_methods, "K-3D Integration Module");
}

/////////////////////////////////////////////////////////////////////////////
// init_context

void init_context(PyObject* Module, const k3d::iscript_engine::context_t& Context)
{
	PyModule_AddObject(Module, "Application", python_wrap(k3d::application()));

	for(k3d::iscript_engine::context_t::const_iterator context = Context.begin(); context != Context.end(); ++context)
	{
		k3d::iunknown* const unknown = context->second.type() == typeid(k3d::iunknown*) ? boost::any_cast<k3d::iunknown*>(context->second) : 0;

		if(k3d::idocument* const document = dynamic_cast<k3d::idocument*>(unknown))
		{
			PyModule_AddObject(Module, const_cast<char*>(context->first.c_str()), python_wrap(*document));
		}
		else if(k3d::inode* const object = dynamic_cast<k3d::inode*>(unknown))
		{
			PyModule_AddObject(Module, const_cast<char*>(context->first.c_str()), python_wrap(*object));
		}
		else if(k3d::mesh* const mesh = dynamic_cast<k3d::mesh*>(unknown))
		{
			PyModule_AddObject(Module, const_cast<char*>(context->first.c_str()), python_wrap(*mesh));
		}
		else
		{
			PyModule_AddObject(Module, const_cast<char*>(context->first.c_str()), any_to_python(context->second));
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// update_context

void update_context(PyObject* Module, k3d::iscript_engine::context_t& Context)
{
	for(k3d::iscript_engine::context_t::iterator context = Context.begin(); context != Context.end(); ++context)
	{
		k3d::iunknown* const unknown = context->second.type() == typeid(k3d::iunknown*) ? boost::any_cast<k3d::iunknown*>(context->second) : 0;

		if(dynamic_cast<k3d::idocument*>(unknown))
			continue;
		else if(dynamic_cast<k3d::inode*>(unknown))
			continue;
		else if(dynamic_cast<k3d::mesh*>(unknown))
			continue;
		else
		{
			context->second = python_to_any(
				PyObject_GetItem(PyModule_GetDict(Module), python_wrap(context->first.c_str())),
				context->second.type());
		}
	}
}

} // namespace libk3dpython

