// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

namespace libk3dselection
{

/////////////////////////////////////////////////////////////////////////////
// selected_point_number

class selected_point_number :
	public k3d::mesh_modifier<k3d::persistent<k3d::node> >
{
	typedef k3d::mesh_modifier<k3d::persistent<k3d::node> > base;

public:
	selected_point_number(k3d::idocument& Document) :
		base(Document),
		m_point_number(init_owner(*this) + init_name("point_number") + init_label(_("Point Number")) + init_description(_("Point number")) + init_slot(sigc::mem_fun(*this, &selected_point_number::get_selected_point_number)))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &selected_point_number::on_reset_geometry));
		m_input_mesh.changed_signal().connect(m_point_number.changed_signal().make_slot());
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &selected_point_number::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	unsigned long get_selected_point_number()
	{
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		for(unsigned long n = 0; n < input->points.size(); ++n)
		{
			if(input->points[n]->selection_weight)
				return n;
		}

		return 0;
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<selected_point_number>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xa98120b3, 0x35714945, 0xa328b2d9, 0xc1f5092e),
				"SelectedPointNumber",
				"Displays first selected point number in mesh point list",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_data(unsigned long, immutable_name, change_signal, no_undo, computed_storage, no_constraint, read_only_property, no_serialization) m_point_number;
};

/////////////////////////////////////////////////////////////////////////////
// selected_point_number_factory

k3d::iplugin_factory& selected_point_number_factory()
{
	return selected_point_number::get_factory();
}

} // namespace libk3dselection

