// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::edit_control::control class, which provides a UI for string quantities
		\author Tim Shead (tshead@k-3d.com)
*/

#include "entry.h"
#include "interactive.h"

#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/window.h>

namespace libk3dngui
{

namespace entry
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	ui_component(Name, &Parent),
	m_data(Data)
{
	set_name("k3d-entry");

	data_changed();
	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::data_changed));
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "set_value")
	{
		interactive::set_text(*this, Arguments);
		select_region(0, 0);
		set_value();
		return true;
	}

	return ui_component::execute_command(Command, Arguments);
}

bool control::on_focus_out_event(GdkEventFocus* Event)
{
	set_value();
	return base::on_focus_out_event(Event);
}

void control::on_activate()
{
	set_value();
	base::on_activate();
}

void control::set_value()
{
	if(!m_data.get())
		return;

	// If the value didn't change, we're done ...
	const std::string new_value = get_text();
	if(new_value == m_data->value())
		return;

	// Record the command for posterity (tutorials) ...
	record_command("set_value", new_value);

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(new_value);

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message + " " + new_value);
}

void control::data_changed()
{
	return_if_fail(m_data.get());
	set_text(m_data->value());
}

} // namespace edit_control

} // namespace libk3dngui


