// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "file_selection.h"
#include "messages.h"

#include <k3dsdk/application.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/options.h>
#include <k3dsdk/result.h>

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

#include <gtkmm/filechooserdialog.h>
#include <gtkmm/stock.h>

namespace libk3dngui
{

bool get_file_path(const k3d::ipath_property::mode_t Mode, const std::string& Type, const std::string& Prompt, const boost::filesystem::path& OldPath, boost::filesystem::path& Result)
{
	Gtk::FileChooserDialog chooser(Prompt);
	return get_file_path(chooser, Mode, Type, Prompt, OldPath, Result);
}

bool get_file_path(Gtk::FileChooserDialog& Chooser, const k3d::ipath_property::mode_t Mode, const std::string& Type, const std::string& Prompt, const boost::filesystem::path& OldPath, boost::filesystem::path& Result)
{
	// Sanity checks ...
	return_val_if_fail(!Type.empty(), false);
	return_val_if_fail(!Prompt.empty(), false);

	// Get the user's path option ...
	boost::filesystem::path startpath = k3d::options::get_path(Type);
	if(!OldPath.empty())
		startpath = OldPath;

	const std::string gtk_start_path = startpath.native_file_string();

	Chooser.set_title(Prompt);
	Chooser.set_filename(gtk_start_path);
	Chooser.set_select_multiple(false);
	Chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);

	switch(Mode)
	{
		case k3d::ipath_property::READ:
			Chooser.set_action(Gtk::FILE_CHOOSER_ACTION_OPEN);
			Chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
			Chooser.set_default_response(Gtk::RESPONSE_OK);
			break;
		case k3d::ipath_property::WRITE:
			Chooser.set_action(Gtk::FILE_CHOOSER_ACTION_SAVE);
			Chooser.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
			Chooser.set_default_response(Gtk::RESPONSE_OK);
			break;
	}
	
	if(Gtk::RESPONSE_OK != Chooser.run())
		return false;

	// Make sure we're not overwriting Precious Memories of The Way We Were ...
	Result = boost::filesystem::path(Chooser.get_filename(), boost::filesystem::native);
	if(Mode == k3d::ipath_property::WRITE)
		{
			if(boost::filesystem::exists(Result))
				{
					std::vector<std::string> buttons;
					buttons.push_back(_("Yes"));
					buttons.push_back(_("No"));
					if(1 != query_message("Overwrite " + Result.native_file_string() + "?", Prompt, 1, buttons))
						return false;
				}
		}

	// Store the path for posterity ...
	k3d::options::set_path(Type, Result);

	return true;
}

} // namespace libk3dngui

