// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <gdkmm/cursor.h>
#include <gtkmm/widget.h>

#include "basic_input_model.h"
#include "command_arguments.h"
#include "cursors.h"
#include "document_state.h"
#include "icons.h"
#include "interactive.h"
#include "keyboard.h"
#include "selection.h"
#include "knife_tool.h"
#include "tutorial_message.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/color.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/property.h>
#include <k3dsdk/property_collection.h>

namespace libk3dngui
{

namespace detail
{

/// Returns whether Edge belongs to Face
bool edge_in_loop(k3d::split_edge* Edge, k3d::split_edge* Loop)
{
	k3d::split_edge* edge = Loop;
	do
	{
		if(edge == Edge)
			return true;

		edge = edge->face_clockwise;
	}
	while(edge != Loop);

	return false;
}

/// Returns the face and polyhedron containing parameter edge
bool edge_face(const k3d::split_edge* Edge, const k3d::mesh& Mesh, k3d::face*& Face, k3d::polyhedron*& Polyhedron)
{
	for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
	{
		for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
		{
			k3d::split_edge* edge = (*face)->first_edge;
			do
			{
				if(edge == Edge)
				{
					Face = *face;
					Polyhedron = *polyhedron;
					return true;
				}

				edge = edge->face_clockwise;
			}
			while(edge != (*face)->first_edge);
		}
	}

	return false;
}

/// Inserts a new edge after Edge, returns new point at the middle
k3d::point* append_edge(k3d::split_edge* Edge)
{
	// Create new point
	k3d::point* new_point = new k3d::point((Edge->vertex->position + Edge->face_clockwise->vertex->position) / 2);

	// Subdivide edge
	k3d::split_edge* new_edge = new k3d::split_edge(new_point);
	new_edge->face_clockwise = Edge->face_clockwise;
	Edge->face_clockwise = new_edge;

	if(!Edge->companion)
		return new_point;

	// Subdivide companion
	k3d::split_edge* companion = Edge->companion;

	k3d::split_edge* new_companion = new k3d::split_edge(new_point);
	new_companion->face_clockwise = companion->face_clockwise;
	companion->face_clockwise = new_companion;

	// Join edges
	k3d::join_edges(*Edge, *new_companion);
	k3d::join_edges(*new_edge, *companion);

	return new_point;
}

/// Connects two vertices belonging to the same face
k3d::face* connect_vertices(k3d::face* Face, k3d::split_edge* Edge1, k3d::split_edge* Edge2)
{
	// Make new edge loops
	k3d::split_edge* new_edge1 = new k3d::split_edge(Edge1->vertex, Edge2);
	k3d::split_edge* new_edge2 = new k3d::split_edge(Edge2->vertex, Edge1);

	k3d::split_edge* anti_edge1 = k3d::face_anticlockwise(Edge1);
	k3d::split_edge* anti_edge2 = k3d::face_anticlockwise(Edge2);

	anti_edge1->face_clockwise = new_edge1;
	anti_edge2->face_clockwise = new_edge2;

	k3d::join_edges(*new_edge1, *new_edge2);

	// Create a second face for ear cut
	k3d::face* new_face = 0;
	if(edge_in_loop(Face->first_edge, Edge1))
	{
		new_face = new k3d::face(Edge2, Face->material);
	}
	else
	{
		new_face = new k3d::face(Edge1, Face->material);
	}

	return new_face;
}

/// Subdivides edges and connect new points
k3d::face* connect_edges(k3d::face* Face, k3d::split_edge* Edge1, k3d::split_edge* Edge2, k3d::mesh& Mesh)
{
	k3d::point* point1 = append_edge(Edge1);
	k3d::point* point2 = append_edge(Edge2);
	Mesh.points.push_back(point1);
	Mesh.points.push_back(point2);

	return connect_vertices(Face, Edge1->face_clockwise, Edge2->face_clockwise);
}

/// Connect an already subdivided edge to a new one
k3d::face* connect_point_to_edge(k3d::face* Face, k3d::split_edge* SplitEdge, k3d::split_edge* Edge2, k3d::mesh& Mesh)
{
	k3d::point* point2 = append_edge(Edge2);
	Mesh.points.push_back(point2);

	return connect_vertices(Face, SplitEdge, Edge2->face_clockwise);
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// knife_tool::implementation

struct knife_tool::implementation :
	public k3d::property_collection
{
	struct split_edge_t
	{
		split_edge_t() :
			edge1(0), edge2(0), mesh(0),
			subdivided(false)
		{
		}

		void reset()
		{
			edge1 = edge2 = 0;
			mesh = 0;
			subdivided = false;
		}

		k3d::split_edge* edge1;
		k3d::split_edge* edge2;
		k3d::mesh* mesh;
		bool subdivided;
	};

	implementation(document_state& DocumentState, knife_tool& Tool) :
		m_document_state(DocumentState),
		m_tool(Tool)
	{
		m_previous_edge = 0;
		m_previous_edge_mesh = 0;
	}

	k3d::idocument& document()
	{
		return m_document_state.document();
	}

	void on_lbutton_down(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		lbutton_down(Viewport, coordinates, convert(Event.state));
	}

	void on_lbutton_click(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		lbutton_click(Viewport);

		// Record the command for tutorials
		assert_not_implemented();
//		m_tool.record_command("selection_click", command_arguments(m_document_state.document(), Viewport, widget_to_ndc(Viewport, coordinates), convert(Event.state)));
	}

	bool execute_command(const std::string& Command, const std::string& Arguments)
	{
		try
		{
			if(Command == "selection_click")
			{
				command_arguments arguments(m_document_state.document(), Arguments);

				tutorial_mouse_message("Click select", k3d::iuser_interface::LMB_CLICK, k3d::key_modifiers());
				interactive::move_pointer(arguments.get_viewport(), arguments.get_viewport_vector2("mouse"));

				assert_not_implemented();
/*
				lbutton_down(arguments.get_viewport(), arguments.get_viewport_vector2("mouse"), arguments.key_modifiers());
				lbutton_click(arguments.get_viewport());
*/
				return true;
			}
		}
		catch(std::exception& e)
		{
			k3d::log() << k3d_file_reference << ": caught exception: " << e.what() << std::endl;
		}

		return false;
	}

	k3d::iunknown* interactive_target(viewport::control& Viewport)
	{
		return Viewport.camera() ? &Viewport.camera()->navigation_target() : 0;
	}

	void lbutton_down(viewport::control& Viewport, const k3d::vector2& NDC, const k3d::key_modifiers& Modifiers)
	{
		// Find an edge under the mouse cursor
		m_start_selection = k3d::selection::record();
		m_start_selection = Viewport.pick_line(NDC);
	}

	void lbutton_click(viewport::control& Viewport)
	{
		assert_not_implemented();
/*
		if(!m_start_selection.empty())
		{
			// Update selection
			if(m_start_selection && !m_start_selection.is_selected())
				m_document_state.select(make_selection(m_start_selection));

			k3d::split_edge* edge = dynamic_cast<k3d::split_edge*>(m_start_selection.selectable);
			k3d::mesh* mesh = m_start_selection.mesh;

			if(m_previous_edge_mesh && mesh && (m_previous_edge_mesh != mesh))
			{
				m_previous_edge = 0;
				m_previous_edge_mesh = 0;

				m_previous_click.reset();
			}

			if(!m_previous_click.edge1)
			{
				if(edge && m_previous_edge && (mesh == m_previous_edge_mesh))
				{
					k3d::split_edge* edge1 = 0;
					k3d::split_edge* edge2 = 0;

					// Check whether the last two edges belong to the same face
					if(detail::edge_in_loop(edge, m_previous_edge))
					{
						edge1 = m_previous_edge;
						edge2 = edge;
					}
					else if(edge->companion && detail::edge_in_loop(edge->companion, m_previous_edge))
					{
						edge1 = m_previous_edge;
						edge2 = edge->companion;
					}
					else if(m_previous_edge->companion && detail::edge_in_loop(edge, m_previous_edge->companion))
					{
						edge1 = m_previous_edge->companion;
						edge2 = edge;
					}
					else if(edge->companion && m_previous_edge->companion && detail::edge_in_loop(edge->companion, m_previous_edge->companion))
					{
						edge1 = m_previous_edge->companion;
						edge2 = edge->companion;
					}
					else
						assert_not_reached();

					if(edge1 && edge2)
					{
						k3d::face* face = 0;
						k3d::polyhedron* polyhedron = 0;
						if(detail::edge_face(edge1, *mesh, face, polyhedron))
						{
							m_previous_click.edge1 = edge2->companion;
							m_previous_click.mesh = mesh;

							k3d::face* new_face = detail::connect_edges(face, edge1, edge2, *mesh);
							polyhedron->faces.push_back(new_face);

							m_previous_click.edge1 = m_previous_click.edge1->face_clockwise;
						}
					}
				}
			}
			else
			{
				if(edge && (mesh == m_previous_click.mesh))
				{
					k3d::split_edge* edge1 = m_previous_click.edge1;
					k3d::split_edge* edge2 = 0;

					// Check whether the last two edges belong to the same face
					if(detail::edge_in_loop(edge1, edge))
					{
						edge2 = edge;
					}
					else if(edge->companion && detail::edge_in_loop(edge1, edge->companion))
					{
						edge2 = edge->companion;
					}
					else
						assert_not_reached();

					if(edge2)
					{
						k3d::face* face = 0;
						k3d::polyhedron* polyhedron = 0;
						if(detail::edge_face(edge1, *mesh, face, polyhedron))
						{
							m_previous_click.edge1 = edge2->companion;
							m_previous_click.mesh = mesh;

							k3d::face* new_face = detail::connect_point_to_edge(face, edge1, edge2, *mesh);
							polyhedron->faces.push_back(new_face);

							m_previous_click.edge1 = m_previous_click.edge1->face_clockwise;
						}
					}
				}
			}

			m_previous_edge = edge;
			m_previous_edge_mesh = mesh;
		}
*/
	}

	k3d::split_edge* m_previous_edge;
	k3d::mesh* m_previous_edge_mesh;

	split_edge_t m_previous_click;

	/// Stores a reference to the owning document
	document_state& m_document_state;
	/// Stores a reference to the owning base tool
	knife_tool& m_tool;
	/// Stores the selection under the pointer when selection started
	k3d::selection::record m_start_selection;

	basic_input_model m_input_model;
};

/////////////////////////////////////////////////////////////////////////////
// knife_tool

knife_tool::knife_tool(document_state& DocumentState, const std::string& Name) :
	base(DocumentState, Name),
	m_implementation(new implementation(DocumentState, *this))
{
}

knife_tool::~knife_tool()
{
	delete m_implementation;
}

bool knife_tool::execute_command(const std::string& Command, const std::string& Arguments)
{
	return m_implementation->execute_command(Command, Arguments);
}

iuser_input_model& knife_tool::get_input_model()
{
	return m_implementation->m_input_model;
}

} // namespace libk3dngui


