// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <gtkmm/liststore.h>
#include <gtkmm/menu.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/treeview.h>
#include <gtk/gtkmain.h>

#include "asynchronous_update.h"
#include "command_arguments.h"
#include "document_state.h"
#include "hotkey_cell_renderer_text.h"
#include "icons.h"
#include "interactive.h"
#include "keyboard.h"
#include "node_list.h"
#include "utility.h"

#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/idag.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_plugin_factory.h>
#include <k3dsdk/inode_collection.h>
#include <k3dsdk/iselectable.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/nodes.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/property.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/utility.h>

namespace libk3dngui
{

namespace node_list
{

namespace detail
{

struct sort_by_label
{
	bool operator()(const node* const LHS, const node* const RHS)
	{
		return LHS->label < RHS->label;
	}
};

struct sort_by_type
{
	bool operator()(const node* const LHS, const node* const RHS)
	{
		if(LHS->nodes.size() != RHS->nodes.size())
			return LHS->nodes.size() < RHS->nodes.size();

		for(unsigned int i = 0; i != LHS->nodes.size(); ++i)
			return typeid(*LHS->nodes[i]).before(typeid(*RHS->nodes[i]));

		return true;
	}
};

struct sort_by_name
{
	bool operator()(k3d::iplugin_factory* const LHS, k3d::iplugin_factory* const RHS)
	{
		return LHS->name() < RHS->name();
	}
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// graph

graph::~graph()
{
	std::for_each(nodes.begin(), nodes.end(), k3d::delete_object());
	std::for_each(edges.begin(), edges.end(), k3d::delete_object());
}

/////////////////////////////////////////////////////////////////////////////
// null_filter_policy

void null_filter_policy::populate_graph(graph& Graph)
{
}

/////////////////////////////////////////////////////////////////////////////
// all_nodes_filter_policy

all_nodes_filter_policy::all_nodes_filter_policy(k3d::idocument& Document) :
	m_document(Document)
{
}

void all_nodes_filter_policy::populate_graph(graph& Graph)
{
	for(k3d::inode_collection::nodes_t::const_iterator n = m_document.nodes().collection().begin(); n != m_document.nodes().collection().end(); ++n)
		{
			node* const new_node = new node;
			new_node->label = (*n)->name();
			new_node->nodes.push_back(*n);

			Graph.nodes.push_back(new_node);
		}
}

/////////////////////////////////////////////////////////////////////////////
// class_id_filter_policy

class_id_filter_policy::class_id_filter_policy(k3d::idocument& Document, const k3d::uuid& ClassID) :
	m_document(Document),
	m_class_id(ClassID)
{
}

void class_id_filter_policy::populate_graph(graph& Graph)
{
	for(k3d::inode_collection::nodes_t::const_iterator n = m_document.nodes().collection().begin(); n != m_document.nodes().collection().end(); ++n)
	{
		if((*n)->factory().class_id() != m_class_id)
			continue;

		node* const new_node = new node;
		new_node->label = (*n)->name();
		new_node->nodes.push_back(*n);

		Graph.nodes.push_back(new_node);
	}
}

/////////////////////////////////////////////////////////////////////////////
// null_layout_policy

void null_layout_policy::update_layout(graph& Graph)
{
}

/////////////////////////////////////////////////////////////////////////////
// sort_by_label_layout_policy

void sort_by_label_layout_policy::update_layout(graph& Graph)
{
	std::sort(Graph.nodes.begin(), Graph.nodes.end(), detail::sort_by_label());
}

/////////////////////////////////////////////////////////////////////////////
// sort_by_type_layout_policy

void sort_by_type_layout_policy::update_layout(graph& Graph)
{
	std::sort(Graph.nodes.begin(), Graph.nodes.end(), detail::sort_by_type());
}

/// This hack makes it easier to implement a context-menu
class tree_view :
	public Gtk::TreeView
{
	typedef Gtk::TreeView base;

	bool on_button_press_event(GdkEventButton* Event)
	{
		base::on_button_press_event(Event);
		return false;
	}

	bool on_button_release_event(GdkEventButton* Event)
	{
		base::on_button_release_event(Event);
		return false;
	}
};

/////////////////////////////////////////////////////////////////////////////
// control::implementation

struct control::implementation :
	public asynchronous_update
{
	implementation(document_state& DocumentState) :
		m_document_state(DocumentState),
		m_filter_policy(new all_nodes_filter_policy(DocumentState.document())),
		m_layout_policy(new sort_by_label_layout_policy()),
		m_new_selection(0)
	{
		m_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		m_scrolled_window.add(m_view);

		m_model = Gtk::ListStore::create(m_columns);

		m_view.set_model(m_model);
		m_view.set_headers_visible(false);
		m_view.set_reorderable(false);

		Gtk::CellRendererText* const cell_text = new hotkey_cell_renderer_text();
		cell_text->property_editable() = true;
		cell_text->signal_edited().connect(sigc::mem_fun(*this, &implementation::on_node_name_edited));

		Gtk::TreeViewColumn* const name_column = new Gtk::TreeViewColumn;
		name_column->pack_start(*manage(cell_text), true);
		name_column->add_attribute(cell_text->property_text(), m_columns.name);

		m_view.append_column("icon", m_columns.icon);
		m_view.append_column(*manage(name_column));

/** \todo There doesn't seem to be any compelling reason to allow this? */
//		m_view.get_selection()->set_mode(Gtk::SELECTION_MULTIPLE);

		// Connect signals
		m_view.get_selection()->signal_changed().connect(sigc::mem_fun(*this, &implementation::on_selection_changed));
		m_view.signal_button_press_event().connect_notify(sigc::mem_fun(*this, &implementation::on_button_press_event), true);
		m_view.signal_key_release_event().connect(sigc::mem_fun(*this, &implementation::on_key_release_event));

		// Reset the control contents anytime the document node collection changes
		m_document_state.document().nodes().add_nodes_signal().connect(sigc::mem_fun(*this, &implementation::on_nodes_added));
		m_document_state.document().nodes().remove_nodes_signal().connect(sigc::mem_fun(*this, &implementation::on_nodes_removed));
		m_document_state.document().nodes().rename_node_signal().connect(sigc::mem_fun(*this, &implementation::on_node_renamed));

		schedule_update();
	}

	void set_filter_policy(filter_policy* const Policy)
	{
		return_if_fail(Policy);

		m_filter_policy.reset(Policy);
		schedule_update();
	}

	void set_layout_policy(layout_policy* const Policy)
	{
		return_if_fail(Policy);

		m_layout_policy.reset(Policy);
		schedule_update();
	}

	/// Called by the signal system anytime new nodes are added to the document
	void on_nodes_added(const k3d::inode_collection::nodes_t& Nodes)
	{
		schedule_update();
	}

	/// Called by the signal system anytime nodes are removed from the document
	void on_nodes_removed(const k3d::inode_collection::nodes_t& Nodes)
	{
		schedule_update();
	}

	/// Called by the signal system anytime an node is renamed
	void on_node_renamed(k3d::inode* const Node)
	{
		schedule_update();
	}

	/// Looks-up a model row based on an node
	bool get_row(k3d::inode* const Node, Gtk::TreeIter& Row)
	{
		Gtk::TreeNodeChildren rows = m_model->children();
		for(Gtk::TreeIter row = rows.begin(); row != rows.end(); ++row)
		{
			if(row->get_value(m_columns.node) == Node)
			{
				Row = row;
				return true;
			}
		}

		return false;
	}

	/// Called by the signal system anytime the user edits a node name in-placea
	void on_node_name_edited(const Glib::ustring& Path, const Glib::ustring& NewText)
	{
		// Get the row that was activated ...
		Gtk::TreeModel::Row row = *m_model->get_iter(Path);

		// Look-up the actual node ...
		k3d::inode* const node = row[m_columns.node];
		return_if_fail(node);

		command_arguments arguments(m_document_state.document());
		arguments.append("oldname", node->name());
		arguments.append("newname", NewText);
		m_command_signal.emit("rename", arguments);

		k3d::record_state_change_set change_set(m_document_state.document(), k3d::string_cast(boost::format(_("Rename node %1%")) % NewText));
		node->set_name(NewText);
	}

	/// Updates the contents of the control
	void on_update()
	{
		m_model->clear();

		m_current_graph.reset(new graph());
		m_filter_policy->populate_graph(*m_current_graph);
		m_layout_policy->update_layout(*m_current_graph);

		// Keep track of roughly where we were in the list ...
//		const double position = ScrolledWindow("scrolledwindow").VerticalAdjustment().Value();

		for(graph::nodes_t::const_iterator n = m_current_graph->nodes.begin(); n != m_current_graph->nodes.end(); ++n)
		{
			Gtk::TreeModel::Row row = *m_model->append();
			row[m_columns.name] = (*n)->label;

			if(1 == (*n)->nodes.size())
			{
				k3d::inode* const node = dynamic_cast<k3d::inode*>((*n)->nodes.front());
				if(node)
					row[m_columns.icon] = quiet_load_icon(node->factory().name(), Gtk::ICON_SIZE_MENU);

				row[m_columns.node] = node;
			}
		}

		// Try to restore our original position (give or take) ...
//		ScrolledWindow("scrolledwindow").VerticalAdjustment().SetValue(position);
	}

	void on_selection_changed()
	{
		// Get the currently-selected rows
		std::list<Gtk::TreePath> selected_rows = m_view.get_selection()->get_selected_rows();
		if(selected_rows.size())
		{
			// Get the selected node
			Gtk::TreeModel::Row row = *m_model->get_iter(selected_rows.back());
			m_new_selection = row[m_columns.node];

			if(m_new_selection)
			{
				// Update the document selection
				m_document_state.set_selection_mode(SELECT_NODES);
				m_document_state.deselect_all();
				m_document_state.select(k3d::selection::make_record(m_new_selection));

				// Send view_node_properties signal
				m_document_state.view_node_properties_signal().emit(m_new_selection);
				m_document_state.view_node_history_signal().emit(m_new_selection);
			}
		}
	}

	void on_button_press_event(GdkEventButton* Event)
	{
		if(Event->button == 3)
		{
			m_document_state.popup_context_menu();

			command_arguments arguments(m_document_state.document());
			arguments.append("node", m_new_selection);
			m_command_signal.emit("context_menu", arguments);
			return;
		}

		if(!m_new_selection)
			return;

		command_arguments arguments(m_document_state.document());
		arguments.append("node", m_new_selection);
		m_command_signal.emit("select_node", arguments);

		m_new_selection = 0;
	}

	bool on_key_release_event(GdkEventKey* Event)
	{
		if(!m_new_selection)
			return false;

		switch(Event->keyval)
		{
			case GDK_Up:
			case GDK_Down:
			case GDK_Home:
			case GDK_End:
			case GDK_Page_Down:
			case GDK_Page_Up:
				break;

			default:
				return false;
		}

		command_arguments arguments(m_document_state.document());
		arguments.append("node", m_new_selection);
		m_command_signal.emit("select_node", arguments);

		m_new_selection = 0;

		return false;
	}

	bool execute_command(const std::string& Command, const std::string& Arguments)
	{
		try
		{
			if(Command == "rename")
			{
				command_arguments arguments(m_document_state.document(), Arguments);
				const std::string old_name = arguments.get_string("oldname");
				const std::string new_name = arguments.get_string("newname");

				k3d::inode* const node = k3d::find_node(m_document_state.document().nodes(), old_name);
				return_val_if_fail(node, false);

				Gtk::TreeIter row;
				return_val_if_fail(get_row(node, row), false);

				/** \todo This is an ugly hack, but interactive::set_text() has its own problems ... */
				interactive::show(m_view);
				interactive::move_pointer(m_view, *m_view.get_column(1), row);
				k3d::record_state_change_set change_set(m_document_state.document(), k3d::string_cast(boost::format(_("Rename node %1%")) % new_name));
				node->set_name(new_name);
				handle_pending_events();
				non_blocking_sleep(500);

//				interactive::set_text(m_view, *m_view.get_column(1), *m_view.get_column_cell_renderer(1), row, new_name);

				return true;
			}

			if(Command == "context_menu")
			{
				command_arguments arguments(m_document_state.document(), Arguments);
				k3d::inode* const node = arguments.get_node("node");
				return_val_if_fail(node, false);

				Gtk::TreeIter row;
				return_val_if_fail(get_row(node, row), false);

				interactive::select_row(m_view, *m_view.get_column(1), row);
				m_document_state.popup_context_menu();
				return true;
			}

			if(Command == "select_node")
			{
				command_arguments arguments(m_document_state.document(), Arguments);
				k3d::inode* const node = arguments.get_node("node");
				return_val_if_fail(node, false);

				Gtk::TreeIter row;
				return_val_if_fail(get_row(node, row), false);

				interactive::select_row(m_view, *m_view.get_column(1), row);
				return true;
			}
		}
		catch(std::exception& e)
		{
			k3d::log() << error << e.what() << std::endl;
		}

		return false;
	}

	/// Stores a reference to the owning document
	document_state& m_document_state;
	/// Stores the current graph being visualized
	std::auto_ptr<graph> m_current_graph;
	/// Stores a policy that controls what's visible in the current graph
	std::auto_ptr<filter_policy> m_filter_policy;
	/// Stores a policy that handles layout of the current graph
	std::auto_ptr<layout_policy> m_layout_policy;

	class columns :
		public Gtk::TreeModel::ColumnRecord
	{
	public:
		columns()
		{
			add(node);
			add(icon);
			add(name);
		}

		Gtk::TreeModelColumn<k3d::inode*> node;
		Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
		Gtk::TreeModelColumn<Glib::ustring> name;
	};

	columns m_columns;
	Glib::RefPtr<Gtk::ListStore> m_model;
	std::auto_ptr<Gtk::Menu> m_context_menu;
	Gtk::ScrolledWindow m_scrolled_window;
	tree_view m_view;

	sigc::signal2<void, const std::string&, const std::string&> m_command_signal;

	k3d::inode* m_new_selection;
};

/////////////////////////////////////////////////////////////////////////////
// control

control::control(document_state& DocumentState, k3d::icommand_node& Parent) :
	base(false, 0),
	ui_component("node_list", &Parent),
	m_implementation(new implementation(DocumentState))
{
	m_implementation->m_command_signal.connect(sigc::mem_fun(*this, &control::record_command));

	pack_start(m_implementation->m_scrolled_window, Gtk::PACK_EXPAND_WIDGET);
	show_all();
}

control::~control()
{
	delete m_implementation;
}

void control::set_filter_policy(filter_policy* const Policy)
{
	m_implementation->set_filter_policy(Policy);
}

void control::set_layout_policy(layout_policy* const Policy)
{
	m_implementation->set_layout_policy(Policy);
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(m_implementation->execute_command(Command, Arguments))
		return true;

	return ui_component::execute_command(Command, Arguments);
}

} // namespace node_list

} // namespace libk3dngui

