// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the property_widget::control, which provides a MVC UI for connecting properties in the DAG
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <gtkmm/image.h>
#include <gtkmm/menu.h>
#include <gtkmm/targetlist.h>
#include <gtk/gtkmain.h>

#include "document_state.h"
#include "icons.h"
#include "interactive.h"
#include "menu_item.h"
#include "messages.h"
#include "plug_tool.h"
#include "property_widget.h"
#include "utility.h"
#include "widget_manip.h"

#include <k3dsdk/color.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/iproperty.h>
#include <k3dsdk/imesh_source.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/nodes.h>
#include <k3dsdk/state_change_set.h>

#include <boost/filesystem/path.hpp>

namespace libk3dngui
{

namespace property_widget
{

/////////////////////////////////////////////////////////////////////////////
// control

property_control::property_control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	ui_component(Name, &Parent),
	m_command_node(Parent),
	m_data(Data)
{
	assert(m_data.get());
}

bool property_control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "connect_to" || Command == "start_connection" || Command == "complete_connection")
		{
			std::stringstream buffer(Arguments);
			k3d::xml::element arguments;
			buffer >> arguments;

			return_val_if_fail(arguments.name == "arguments", false);
			const std::string object_name = k3d::xml::element_text(arguments, "object");
			const std::string property_name = k3d::xml::element_text(arguments, "property");
			return_val_if_fail(!object_name.empty(), false);
			return_val_if_fail(!property_name.empty(), false);

			// Find named object pointer
			k3d::inode* const object = k3d::find_node(m_data->document().document().nodes(), object_name);
			return_val_if_fail(object, false);

			// Find named property pointer
			k3d::iproperty* to_property = 0;

			k3d::iproperty_collection* property_collection = dynamic_cast<k3d::iproperty_collection*>(object);
			return_val_if_fail(property_collection, 0);

			const k3d::iproperty_collection::properties_t properties = property_collection->properties();
			for(k3d::iproperty_collection::properties_t::const_iterator property = properties.begin(); property != properties.end(); ++property)
				{
					if(property_name == (*property)->property_name())
						{
							to_property = (*property);
							break;
						}
				}

			return_val_if_fail(to_property, false);

			if(Command == "connect_to")
				connect_to(to_property);
			else if(Command == "start_connection")
				start_connection(to_property);
			else if(Command == "complete_connection")
				complete_connection(to_property);
			else
				assert_not_reached();

			return true;
		}
	else if(Command == "disconnect")
		{
			k3d::iproperty* const dependency = m_data->document().document().dag().dependency(m_data->property());
			if(dependency)
				{
					k3d::inode* const object = k3d::find_node(m_data->document().document().nodes(), *dependency);
					disconnect(object);
				}

			return true;
		}
	else if(Command == "show_connected")
		{
			k3d::iproperty* const dependency = m_data->document().document().dag().dependency(m_data->property());
			if(dependency)
				{
					k3d::inode* const object = k3d::find_node(m_data->document().document().nodes(), *dependency);
					show_connected(object);
				}

			return true;
		}

	return ui_component::execute_command(Command, Arguments);
}

bool property_control::button_press_event(GdkEventButton* Event)
{
	// Open context menu with left or right mouse
	if(Event->button != 1 && Event->button != 3)
		return false;

	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	if(tool && tool->connection_started())
		{
			on_cancel_connection();
			return true;
		}

	// Reset menu
	m_menu.reset(new Gtk::Menu());

	// Build node path for tutorial recording
	const std::string node_name = m_data->property().property_name() + "_property";

	// If the property's connected ...
	k3d::iproperty* const dependency = m_data->document().document().dag().dependency(m_data->property());
	if(dependency)
		{
			k3d::inode* const object = k3d::find_node(m_data->document().document().nodes(), *dependency);

			m_menu->items().push_back(*Gtk::manage(
				new menu_item::control(m_command_node, node_name, _("Show Connected")) <<
				connect_menu_item(sigc::bind(sigc::mem_fun(*this, &property_control::on_show_connected), object))));

			m_menu->items().push_back(*Gtk::manage(
				new menu_item::control(m_command_node, node_name, _("Disconnect")) <<
				connect_menu_item(sigc::bind(sigc::mem_fun(*this, &property_control::on_disconnect), object))));
		}

	// Get the list of compatible properties
	Gtk::Menu* const submenu = new Gtk::Menu();
	m_menu->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Connect to ..."), *manage(submenu)));

	typedef std::map<std::string, k3d::iproperty*> properties_t;
	properties_t properties_map;
	for(k3d::inode_collection::nodes_t::const_iterator object = m_data->document().document().nodes().collection().begin(); object != m_data->document().document().nodes().collection().end(); ++object)
		{
			k3d::iproperty_collection* const property_collection = dynamic_cast<k3d::iproperty_collection*>(*object);
			if(!property_collection)
				continue;

			k3d::iproperty_collection::properties_t all_properties = property_collection->properties();
			for(k3d::iproperty_collection::properties_t::iterator p = all_properties.begin(); p != all_properties.end(); ++p)
				{
					if(m_data->property().property_type() != (*p)->property_type())
						continue;

					const std::string label = (*object)->name() + " : " + (*p)->property_name();
					properties_map.insert(std::make_pair(label, *p));
				}
		}

	// Add compatible properties to the 'connect' submenu
	for(properties_t::const_iterator property = properties_map.begin(); property != properties_map.end(); ++property)
		{
			submenu->items().push_back(*Gtk::manage(
				new menu_item::control(m_command_node, node_name, property->first) <<
				connect_menu_item(sigc::bind(sigc::mem_fun(*this, &property_control::on_connect_to), property->second))
				));
		}

	// The interactive plug tool
	m_menu->items().push_back(*Gtk::manage(
		new Gtk::MenuItem(_("Start Connection ...")) <<
			connect_menu_item(sigc::mem_fun(*this, &property_control::on_start_connection))));

	m_menu->show_all();
	m_menu->popup(3, gtk_get_current_event_time());

	return true;
}

bool property_control::button_release_event(GdkEventButton* Event)
{
	if(Event->button != 1)
		return false;

	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	if(tool && tool->connection_started())
		{
			on_complete_connection();
			return true;
		}

	return true;
}

bool property_control::on_enter_notify_event(GdkEventCrossing* event)
{
	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	if(tool && tool->connection_started() && !tool->allow_connection(m_data->property()))
		m_data->document().set_cursor_signal().emit(load_icon("cant_connect_property_cursor", Gtk::ICON_SIZE_BUTTON));

//	m_data->document().push_status_message_signal().emit(_("Do something!"));
	return true;
}

bool property_control::on_leave_notify_event(GdkEventCrossing* event)
{
	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	if(tool && tool->connection_started() && !tool->allow_connection(m_data->property()))
		m_data->document().set_cursor_signal().emit(load_icon("connect_property_cursor", Gtk::ICON_SIZE_BUTTON));

//	m_data->document().pop_status_message_signal().emit();
	return true;
}

void property_control::on_start_connection()
{
	k3d::xml::element arguments("arguments");
	arguments.append(k3d::xml::element("object", m_data->property().property_node()->name()));
	arguments.append(k3d::xml::element("property", m_data->property().property_name()));
	std::stringstream buffer;
	buffer << k3d::xml::single_line() << arguments;
	record_command("start_connection", buffer.str());

	start_connection(&m_data->property());
}

void property_control::on_complete_connection()
{
	k3d::xml::element arguments("arguments");
	arguments.append(k3d::xml::element("object", m_data->property().property_node()->name()));
	arguments.append(k3d::xml::element("property", m_data->property().property_name()));
	std::stringstream buffer;
	buffer << k3d::xml::single_line() << arguments;
	record_command("complete_connection", buffer.str());

	k3d::record_state_change_set changeset(m_data->document().document(), _("Connect Properties"));

	complete_connection(&m_data->property());
}

void property_control::on_cancel_connection()
{
	m_data->document().set_active_tool(m_data->document().selection_tool());
}

void property_control::on_connect_to(k3d::iproperty* Property)
{
	k3d::xml::element arguments("arguments");
	arguments.append(k3d::xml::element("object", Property->property_node()->name()));
	arguments.append(k3d::xml::element("property", Property->property_name()));
	std::stringstream buffer;
	buffer << k3d::xml::single_line() << arguments;
	record_command("connect_to", buffer.str());

	k3d::record_state_change_set changeset(m_data->document().document(), _("Connect Properties"));
	connect_to(Property);
}

void property_control::on_disconnect(k3d::inode* Object)
{
	record_command("disconnect");

	k3d::record_state_change_set changeset(m_data->document().document(), m_data->change_message + " Disconnect");
	disconnect(Object);
}

void property_control::on_show_connected(k3d::inode* Object)
{
	record_command("show_connected");

	show_connected(Object);
}

void property_control::start_connection(k3d::iproperty* Property)
{
	m_data->document().set_active_tool(m_data->document().plug_tool());
	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	return_if_fail(tool);

	tool->set_selected_properties(k3d::iproperty_collection::properties_t(1, Property));
}

void property_control::connect_to(k3d::iproperty* Property)
{
	return_if_fail(Property);

	// If the property's already connected, disconnect it first
	if(m_data->document().document().dag().dependency(m_data->property()))
		{
			k3d::idag::dependencies_t dependencies;
			dependencies[&m_data->property()] = static_cast<k3d::iproperty*>(0);
			m_data->document().document().dag().set_dependencies(dependencies);
		}

	// Make connection
	k3d::idag::dependencies_t dependencies;
	dependencies.insert(std::make_pair(&m_data->property(), Property));
	m_data->document().document().dag().set_dependencies(dependencies);
}

void property_control::disconnect(k3d::inode* Object)
{
	return_if_fail(Object);

	k3d::idag::dependencies_t dependencies;
	dependencies[&m_data->property()] = static_cast<k3d::iproperty*>(0);
	m_data->document().document().dag().set_dependencies(dependencies);
}

void property_control::show_connected(k3d::inode* Node)
{
	return_if_fail(Node);
	m_data->document().view_node_properties_signal().emit(Node);
}

void property_control::complete_connection(k3d::iproperty* Property)
{
	plug_tool* const tool = dynamic_cast<plug_tool*>(&m_data->document().active_tool());
	return_if_fail(tool);

	if(!tool->allow_connection(*Property))
		{
			error_message(_("Can't connect to this property"), _("Connect Properties:"));
			return;
		}

	k3d::idag::dependencies_t dependencies;
	const k3d::iproperty_collection::properties_t selected_properties = tool->get_selected_properties();
	for(k3d::iproperty_collection::properties_t::const_iterator property = selected_properties.begin(); property != selected_properties.end(); ++property)
		dependencies.insert(std::make_pair(*property, Property));

	m_data->document().document().dag().set_dependencies(dependencies);
	m_data->document().set_active_tool(m_data->document().selection_tool());
}

} // namespace property_widget

} // namespace libk3dngui


