// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <gdkmm/cursor.h>
#include <gtkmm/widget.h>

#include "basic_input_model.h"
#include "command_arguments.h"
#include "cursors.h"
#include "document_state.h"
#include "modifiers.h"
#include "icons.h"
#include "interactive.h"
#include "keyboard.h"
#include "rotate_tool.h"
#include "transform_tool.h"
#include "tutorial_message.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/property.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/xml.h>

#include <k3dsdk/fstream.h>

namespace libk3dngui
{

/////////////////////////////////////////////////////////////////////////////
// rotate_tool::implementation

class rotate_tool::implementation :
	public detail::abstract_tool,
	public transform_tool
{
	typedef transform_tool base;

	struct constraint :
		public k3d::selectable
	{
		constraint(const std::string& Label, Glib::RefPtr<Gdk::Pixbuf> Cursor, const k3d::selection::token& SelectionToken) :
			m_selection_token(SelectionToken),
			m_label(Label),
			m_cursor(Cursor),
			m_plane(k3d::normal3(0, 0, 0), 0),
			m_axis(0, 0, 0)
		{
			assert_warning(!m_label.empty());
		}

		const std::string label()
		{
			return m_label;
		}

		Glib::RefPtr<Gdk::Pixbuf> cursor()
		{
			return m_cursor;
		}

		void set_plane(const k3d::plane& Plane, const k3d::normal3& Axis)
		{
			m_plane = Plane;
			m_axis = Axis;
		}

		void begin_mouse_move(const k3d::vector2& Mouse, const k3d::vector3& Origin)
		{
			m_first_mouse = Mouse;
			m_rotation_origin = Origin;
		}

		k3d::angle_axis mouse_move(viewport::control& Viewport, const k3d::vector2& Coordinates)
		{
			const k3d::vector2 current_mouse(Coordinates);
			const k3d::vector2 origin = Viewport.project(m_rotation_origin);

			const k3d::vector2 v1 = m_first_mouse - origin;
			const k3d::vector2 v2 = current_mouse - origin;

			// First or current mouse is on the origin
			if(!v1.Length2() || !v2.Length2())
				return k3d::angle_axis(0, m_axis);

			// Angle between v1 and v2
			const double dot_product = v1 * v2;
			double angle = acos(dot_product / k3d::length(v1) / k3d::length(v2));

			// Check for rotation manipulator direction
			const k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
			// Forward camera vector is +Z
			const k3d::normal3 camera_forward = screen_matrix * k3d::normal3(0, 0, 1);
			if(camera_forward * m_axis < 0)
				angle = -angle;

			// Test whether current_mouse point is below line defined by (m_first_mouse, origin) :
			if(m_first_mouse[0] == origin[0])
			{
				// Vertical line, test left or right instead
				if(current_mouse[0] > origin[0])
					angle = -angle;
			}
			else
			{
				// Check whether m_first_mouse was in the left or right quadrants
				if(m_first_mouse[0] > origin[0])
					angle = -angle;

				// Project current_mouse vertically on line defined by (m_first_mouse, origin)
				const double a = (current_mouse[0] - m_first_mouse[0]) / (origin[0] - m_first_mouse[0]);
				const double y = m_first_mouse[1] + a * (origin[1] - m_first_mouse[1]);
				if(y > current_mouse[1])
					angle = -angle;
			}

			return k3d::angle_axis(angle, m_axis);
		}

		const k3d::normal3 axis()
		{
			return m_axis;
		}

		const k3d::selection::token m_selection_token;
		const std::string m_label;
		const Glib::RefPtr<Gdk::Pixbuf> m_cursor;

		/// Stores the constraint plane in world coordinates
		k3d::plane m_plane;
		/// Stores the rotation axis in world coordinates
		k3d::normal3 m_axis;

		k3d::vector2 m_first_mouse;
		k3d::vector3 m_rotation_origin;
	};

public:
	implementation(k3d::idocument& Document, document_state& DocumentState, rotate_tool& RotateTool) :
		base(Document, DocumentState, *this),
		m_rotate_tool(RotateTool),
		m_current_constraint(&m_screen_z_constraint),
		m_screen_z_constraint(_("Rotate Screen Z"), load_icon("move_cursor_screen_xy", Gtk::ICON_SIZE_BUTTON), k3d::selection::token(k3d::selection::USER1, 0)),
		m_x_constraint(_("Rotate X"), load_icon("move_cursor_x", Gtk::ICON_SIZE_BUTTON), k3d::selection::token(k3d::selection::USER1, 1)),
		m_y_constraint(_("Rotate Y"), load_icon("move_cursor_y", Gtk::ICON_SIZE_BUTTON), k3d::selection::token(k3d::selection::USER1, 2)),
		m_z_constraint(_("Rotate Z"), load_icon("move_cursor_z", Gtk::ICON_SIZE_BUTTON), k3d::selection::token(k3d::selection::USER1, 3)),
		m_angle(init_owner(*this) + init_name("angle") + init_label(_("Angle")) + init_description(_("Rotation angle")) + init_value(0.0) + init_precision(2) + init_step_increment(1.0) + init_units(typeid(k3d::measurement::angle))),
		m_center_x(init_owner(*this) + init_name("center_x") + init_label(_("Center X")) + init_description(_("Rotation center X position")) + init_value(0.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_center_y(init_owner(*this) + init_name("center_y") + init_label(_("Center Y")) + init_description(_("Rotation center Y position")) + init_value(0.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_center_z(init_owner(*this) + init_name("center_z") + init_label(_("Center Z")) + init_description(_("Rotation center Z position")) + init_value(0.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance)))
	{
		m_angle.changed_signal().connect(sigc::mem_fun(*this, &rotate_tool::implementation::angle_update));

		m_center_x.changed_signal().connect(sigc::mem_fun(*this, &implementation::center_spin_button_update));
		m_center_y.changed_signal().connect(sigc::mem_fun(*this, &implementation::center_spin_button_update));
		m_center_z.changed_signal().connect(sigc::mem_fun(*this, &implementation::center_spin_button_update));

		m_input_model.connect_lbutton_down(sigc::mem_fun(*this, &implementation::on_lbutton_down));
		m_input_model.connect_lbutton_click(sigc::mem_fun(*this, &implementation::on_lbutton_click));
		m_input_model.connect_lbutton_start_drag(sigc::mem_fun(*this, &implementation::on_lbutton_start_drag));
		m_input_model.connect_lbutton_drag(sigc::mem_fun(*this, &implementation::on_lbutton_drag));
		m_input_model.connect_lbutton_end_drag(sigc::mem_fun(*this, &implementation::on_lbutton_end_drag));
		m_input_model.connect_mbutton_click(sigc::mem_fun(*this, &implementation::on_mbutton_click));
		m_input_model.connect_rbutton_click(sigc::mem_fun(*this, &implementation::on_rbutton_click));
		m_input_model.connect_mouse_move(sigc::mem_fun(*this, &implementation::on_mouse_move));
	}

	~implementation()
	{
	}

	void on_activate()
	{
		try
		{
			k3d::filesystem::ifstream layout_stream(k3d::application().share_path() / "ngui/tool_layout.k3d");
			k3d::xml::element layout_xml;
			layout_stream >> layout_xml;
			k3d::xml::element& xml_rotate = layout_xml.safe_element("application").safe_element("user_interface").safe_element("tools").safe_element("rotate");

			m_manipulators_size = k3d::xml::attribute_value<double>(xml_rotate, "size", 10);
			m_current_color = k3d::xml::attribute_value<k3d::color>(xml_rotate, "current_color", k3d::color(1, 1, 0));
			m_x_color = k3d::xml::attribute_value<k3d::color>(xml_rotate, "x_color", k3d::color(1, 0, 0));
			m_y_color = k3d::xml::attribute_value<k3d::color>(xml_rotate, "y_color", k3d::color(0, 1, 0));
			m_z_color = k3d::xml::attribute_value<k3d::color>(xml_rotate, "z_color", k3d::color(0, 0, 1));
			m_screen_z_color = k3d::xml::attribute_value<k3d::color>(xml_rotate, "screen_z_color", k3d::color(0.9, 0.9, 0.9));
			m_handle_size = k3d::xml::attribute_value<double>(xml_rotate, "handle_size", 5);
			m_handle_major_radius = k3d::xml::attribute_value<double>(xml_rotate, "handle_major_radius", 1.0);
			m_handle_minor_radius = k3d::xml::attribute_value<double>(xml_rotate, "handle_minor_radius", 0.03);
			m_handle_u_segments = k3d::xml::attribute_value<unsigned long>(xml_rotate, "handle_u_segments", 16);
			m_handle_v_segments = k3d::xml::attribute_value<unsigned long>(xml_rotate, "handle_v_segments", 4);
			m_screen_z_handle_size = k3d::xml::attribute_value<double>(xml_rotate, "screen_z_size", 8);
		}
		catch(std::exception& e)
		{
			k3d::log() << error << e.what() << std::endl;
		}

		m_visible_manipulators.set_value(true);
		set_constraint(&m_screen_z_constraint);
		on_selection_changed();

		redraw_all();
	}

	void on_deactivate()
	{
		if(MOTION_NONE != m_current_motion)
			cancel_mouse_move();

		m_document_state.clear_cursor_signal().emit();
		redraw_all();
	}

	void on_selection_changed()
	{
		get_current_selection();
		set_center(world_position());
	}

	void record_command(viewport::control& Viewport, const GdkEventButton& Event, const bool Move)
	{
		command_arguments arguments(m_document);
		arguments.append_viewport_coordinates("mouse", Viewport, Event);

		if(Move)
			m_rotate_tool.record_command("mouse_move", arguments);

		m_rotate_tool.record_command(m_tutorial_action, arguments);
		m_tutorial_action = "";
	}

	void record_transform(viewport::control& Viewport, const GdkEventMotion& Event, const k3d::angle_axis& Rotation)
	{
		command_arguments arguments(m_document);
		arguments.append_viewport_coordinates("mouse", Viewport, Event);

		m_rotate_tool.record_command("mouse_warp", arguments);

		arguments.append("rotation", Rotation);
		m_rotate_tool.record_command(m_tutorial_action, arguments);
		m_tutorial_action = "";
	}

	void on_lbutton_down(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);
		const k3d::key_modifiers modifiers = convert(Event.state);

		lbutton_down(Viewport, coordinates, modifiers);

		// Record command for tutorials
		record_command(Viewport, Event, true);
	}

	void on_lbutton_click(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		lbutton_click(Viewport, coordinates);

		// Record command for tutorials
		record_command(Viewport, Event, false);
	}

	void on_lbutton_start_drag(viewport::control& Viewport, const GdkEventMotion& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		lbutton_start_drag(Viewport, coordinates);

		// Record command for tutorials
		record_transform(Viewport, Event, k3d::angle_axis(0, 1, 0, 0));
	}

	void on_lbutton_drag(viewport::control& Viewport, const GdkEventMotion& Event)
	{
		k3d::vector2 coordinates(Event.x, Event.y);

		// Off-screen wrap
		if(off_screen_warp(Viewport, coordinates))
			m_current_constraint->begin_mouse_move(coordinates, get_center());

		const k3d::angle_axis rotation = lbutton_drag(Viewport, coordinates);

		// Record command for tutorials
		record_transform(Viewport, Event, rotation);
	}

	void on_lbutton_end_drag(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		lbutton_end_drag(Viewport, coordinates);

		// Record command for tutorials
		record_command(Viewport, Event, false);
	}

	void on_mbutton_click(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);
		const k3d::key_modifiers modifiers = convert(Event.state);

		mbutton_click(Viewport, coordinates, modifiers);

		// Record command for tutorials
		record_command(Viewport, Event, true);
	}

	void on_rbutton_click(viewport::control& Viewport, const GdkEventButton& Event)
	{
		const k3d::vector2 coordinates(Event.x, Event.y);

		rbutton_click(Viewport, coordinates);

		// Record command for tutorials
		record_command(Viewport, Event, true);
	}

	void on_mouse_move(viewport::control& Viewport, const GdkEventMotion& Event)
	{
		if(MOTION_CLICK_DRAG != m_current_motion)
			return;

		k3d::vector2 coordinates(Event.x, Event.y);

		// Off-screen wrap
		if(off_screen_warp(Viewport, coordinates))
			m_current_constraint->begin_mouse_move(coordinates, get_center());

		const k3d::angle_axis rotation = mouse_move_action(Viewport, coordinates);

		// Record command for tutorials
		record_transform(Viewport, Event, rotation);
	}

	void on_redraw(viewport::control& Viewport)
	{
		// Sanity checks
		return_if_fail(Viewport.gl_engine());
		return_if_fail(Viewport.camera());

		// Update target list
		update_targets();

		// Update constraints
		const k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
		const k3d::normal3 screen_normal = screen_matrix * k3d::normal3(0, 0, 1);

		const k3d::vector3 origin = world_position();
		const k3d::matrix4 orientation = world_orientation();

		// Update the screen xy constraint so it always aligns with the camera direction vector in world coordinates
		m_screen_z_constraint.set_plane(k3d::plane(screen_normal, origin), screen_normal);

		// Update axis constraints so their planes are always as perpendicular to the screen as-possible
		m_x_constraint.set_plane(k3d::plane((orientation * k3d::normal3(1, 0, 0)) ^ ((orientation * k3d::normal3(1, 0, 0)) ^ screen_normal), origin), k3d::normal3(1, 0, 0));
		m_y_constraint.set_plane(k3d::plane((orientation * k3d::normal3(0, 1, 0)) ^ ((orientation * k3d::normal3(0, 1, 0)) ^ screen_normal), origin), k3d::normal3(0, 1, 0));
		m_z_constraint.set_plane(k3d::plane((orientation * k3d::normal3(0, 0, 1)) ^ ((orientation * k3d::normal3(0, 0, 1)) ^ screen_normal), origin), k3d::normal3(0, 0, 1));

		// Draw manipulators
		if(!m_visible_manipulators.value() || !target_number())
			return;

		k3d::gl::store_attributes attributes;
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		k3d::gl::push_matrix(k3d::translation3D(origin));

		glClear(GL_DEPTH_BUFFER_BIT);

		// Scale manipulators to show them at constant size (except on mouse drag)
		if(m_current_motion == MOTION_NONE)
			update_manipulators_scale(Viewport, origin);

		glScaled(m_manipulators_scale, m_manipulators_scale, m_manipulators_scale);

		// Draw manipulators
		const bool motion = (MOTION_DRAG == m_current_motion) || (MOTION_CLICK_DRAG == m_current_motion);

		glPushMatrix();
		k3d::gl::push_matrix(orientation);
		if(!motion || &m_x_constraint == m_current_constraint)
			draw_handle(m_current_constraint == &m_x_constraint ? m_current_color : m_x_color, k3d::rotation3D(k3d::radians(90.0), k3d::vector3(0, 1, 0)), m_handle_size);
		if(!motion || &m_y_constraint == m_current_constraint)
			draw_handle(m_current_constraint == &m_y_constraint ? m_current_color : m_y_color, k3d::rotation3D(k3d::radians(90.0), k3d::vector3(1, 0, 0)), m_handle_size);
		if(!motion || &m_z_constraint == m_current_constraint)
			draw_handle(m_current_constraint == &m_z_constraint ? m_current_color : m_z_color, k3d::rotation3D(k3d::radians(0.0), k3d::vector3(1, 0, 0)), m_handle_size);
		glPopMatrix();

		if(!motion || &m_screen_z_constraint == m_current_constraint)
		{
			k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
			// Zero translation
			screen_matrix[0][3] = screen_matrix[1][3] = screen_matrix[2][3] = 0;

			draw_handle(m_current_constraint == &m_screen_z_constraint ? m_current_color : m_screen_z_color, screen_matrix, m_screen_z_handle_size);
		}

		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}

	void on_select(viewport::control& Viewport)
	{
		if(!m_visible_manipulators.value() || !target_number())
			return;

		const k3d::vector3 origin = world_position();
		const k3d::matrix4 orientation = world_orientation();

		k3d::gl::store_attributes attributes;
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		k3d::gl::push_matrix(k3d::translation3D(origin));

		glDisable(GL_LIGHTING);
		glClear(GL_DEPTH_BUFFER_BIT);

		// Scale manipulators
		glScaled(m_manipulators_scale, m_manipulators_scale, m_manipulators_scale);

		// Draw manipulators
		glPushMatrix();
		k3d::gl::push_matrix(orientation);
		select_handle(m_x_constraint, k3d::rotation3D(k3d::radians(90.0), k3d::vector3(0, 1, 0)), m_handle_size);
		select_handle(m_y_constraint, k3d::rotation3D(k3d::radians(90.0), k3d::vector3(1, 0, 0)), m_handle_size);
		select_handle(m_z_constraint, k3d::rotation3D(k3d::radians(0.0), k3d::vector3(1, 0, 0)), m_handle_size);
		glPopMatrix();

		k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
		screen_matrix[0][3] = screen_matrix[1][3] = screen_matrix[2][3] = 0;
		select_handle(m_screen_z_constraint, screen_matrix, m_screen_z_handle_size);

		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}

	bool execute_command(const std::string& Command, const std::string& Arguments)
	{
		try
		{
			command_arguments arguments(m_document, Arguments);
			//tutorial_mouse_message("Move tool click", k3d::iuser_interface::LMB_CLICK, k3d::key_modifiers());

			if(Command == "mouse_move")
			{
				interactive::move_pointer(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "mouse_warp")
			{
				interactive::warp_pointer(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "lmb_down_add")
			{
				lmb_down_add();
			}
			else if(Command == "lmb_down_subtract")
			{
				lmb_down_subtract();
			}
			else if(Command.substr(0, std::min(static_cast<size_t>(21), Command.size())) == "lmb_down_manipulator_")
			{
				lmb_down_manipulator(Command.substr(21));
			}
			else if(Command == "lmb_down_selected")
			{
				lmb_down_selected();
			}
			else if(Command == "lmb_down_deselected")
			{
				lmb_down_deselected();
			}
			else if(Command == "lmb_down_nothing")
			{
				lmb_down_nothing();
			}
			else if(Command == "lmb_click_add")
			{
				lmb_click_add();
			}
			else if(Command == "lmb_click_subtract")
			{
				lmb_click_subtract();
			}
			else if(Command == "lmb_click_start_motion")
			{
				lmb_click_start_motion(arguments.get_viewport_vector2());
			}
			else if(Command == "lmb_click_stop_motion")
			{
				lmb_click_stop_motion();
			}
			else if(Command == "lmb_click_deselect_all")
			{
				lmb_click_deselect_all();
			}
			else if(Command == "lmb_start_drag_start_motion")
			{
				lmb_start_drag_start_motion(arguments.get_viewport_vector2());
			}
			else if(Command == "lmb_start_drag_box_select")
			{
				lmb_start_drag_box_select(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "lmb_drag_move")
			{
				rotate_selection(arguments.get_angle_axis("rotation"));
				k3d::gl::redraw_all(m_document, k3d::gl::irender_engine::SYNCHRONOUS);
			}
			else if(Command == "lmb_drag_box_select")
			{
				lmb_drag_box_select(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "lmb_end_drag_stop_motion")
			{
				lmb_end_drag_stop_motion();
			}
			else if(Command == "lmb_end_drag_box_select")
			{
				lmb_end_drag_box_select(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "mmb_click_toggle_manipulators_visibility")
			{
				mmb_click_toggle_manipulators_visibility();
			}
			else if(Command == "mmb_click_manipulators_next_selection")
			{
				// TODO : show SHIFT key for tutorials
				mmb_click_manipulators_next_selection();
			}
			else if(Command == "mmb_click_switch_coordinate_system")
			{
				// TODO : show CONTROL key for tutorials
				mmb_click_switch_coordinate_system();
			}
			else if(Command == "mmb_click_next_constraint")
			{
				mmb_click_next_constraint(arguments.get_viewport(), arguments.get_viewport_vector2());
			}
			else if(Command == "rmb_click_selection_tool")
			{
				rmb_click_selection_tool();
			}
			else if(Command == "rmb_click_cancel_move")
			{
				rmb_click_cancel_move();
			}
			else if(Command == "mouse_drag_move")
			{
				rotate_selection(arguments.get_angle_axis("rotation"));
				k3d::gl::redraw_all(m_document, k3d::gl::irender_engine::SYNCHRONOUS);
			}
			else
				return false;

			return true;
		}
		catch(std::exception& e)
		{
			k3d::log() << k3d_file_reference << ": caught exception: " << e.what() << std::endl;
		}

		return false;
	}

private:
	// abstract tool implementation
	virtual std::string manipulator_name(const k3d::selection::id ID)
	{
		if(ID == m_screen_z_constraint.m_selection_token.id)
			return "screen_z";
		else if(ID == m_x_constraint.m_selection_token.id)
			return "x_axis";
		else if(ID == m_y_constraint.m_selection_token.id)
			return "y_axis";
		else if(ID == m_z_constraint.m_selection_token.id)
			return "z_axis";

		return "";
	}

	virtual std::string get_manipulator(const manipulators_t& Manipulators)
	{
		if(!Manipulators.size())
			return std::string("");

		return *Manipulators.begin();
	}

	// Set manipulator constraint
	void set_manipulator(const std::string ManipulatorName)
	{
		if(ManipulatorName == "screen_z")
			set_constraint(&m_screen_z_constraint);
		else if(ManipulatorName == "x_axis")
			set_constraint(&m_x_constraint);
		else if(ManipulatorName == "y_axis")
			set_constraint(&m_y_constraint);
		else if(ManipulatorName == "z_axis")
			set_constraint(&m_z_constraint);
		else
			assert_not_reached();
	}

	std::string get_constraint_name()
	{
		return_val_if_fail(m_current_constraint, "");

		return m_current_constraint->label();
	}

	virtual void begin_mouse_move(const k3d::vector2& Coords)
	{
		return_if_fail(m_current_constraint);

		m_current_constraint->begin_mouse_move(Coords, get_center());

		init_rotation();
	}

	void set_constraint(constraint* const Constraint)
	{
		return_if_fail(Constraint);

		m_current_constraint = Constraint;
		m_document_state.set_cursor_signal().emit(m_current_constraint->cursor());

		redraw_all();
	}

	constraint* cycle_constraint(constraint* Constraint)
	{
		if(Constraint == &m_x_constraint)
			return &m_y_constraint;

		if(Constraint == &m_y_constraint)
			return &m_z_constraint;

		if(Constraint == &m_z_constraint)
			return &m_x_constraint;

		return 0;
	}

	void update_constraint(viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		const k3d::vector3 origin = world_position();

		// Cycle through X - Screen - Y - Screen - Z - Screen - X - etc
		if(m_current_constraint == &m_x_constraint || m_current_constraint == &m_y_constraint || m_current_constraint == &m_z_constraint)
		{
			// Save current constraint
			m_previous_constraint = m_current_constraint;

			// Go back to screeen mode
			set_constraint(&m_screen_z_constraint);
		}
		else if(m_previous_constraint)
		{
			// An axis was previously chosen, cycle to next one
			set_constraint(cycle_constraint(m_previous_constraint));
		}
		else
		{
			// Constrain movement to the "nearest" axis
			const k3d::vector2 mouse(Coordinates);
			const k3d::vector2 coords = Viewport.project(origin);
			const k3d::matrix4 orientation = world_orientation();

			std::map<double, constraint*> constraints;
			constraints.insert(std::make_pair(k3d::distance(mouse, k3d::line2(coords, Viewport.project(origin + (orientation * k3d::normal3(1, 0, 0))))), &m_x_constraint));
			constraints.insert(std::make_pair(k3d::distance(mouse, k3d::line2(coords, Viewport.project(origin + (orientation * k3d::normal3(0, 1, 0))))), &m_y_constraint));
			constraints.insert(std::make_pair(k3d::distance(mouse, k3d::line2(coords, Viewport.project(origin + (orientation * k3d::normal3(0, 0, 1))))), &m_z_constraint));

			set_constraint(constraints.begin()->second);
		}
	}

	// Manipulator drawing/selection functions
	void draw_handle(const k3d::color& Color, const k3d::matrix4& Matrix, const double Size)
	{
		k3d::gl::store_attributes attributes;
		k3d::gl::color3d(Color);

		glEnable(GL_LIGHTING);
		k3d::gl::material(GL_FRONT_AND_BACK, GL_AMBIENT, k3d::color(0, 0, 0));
		k3d::gl::material(GL_FRONT_AND_BACK, GL_DIFFUSE, Color);
		k3d::gl::material(GL_FRONT_AND_BACK, GL_SPECULAR, k3d::color(0, 0, 0));
		k3d::gl::material(GL_FRONT_AND_BACK, GL_EMISSION, k3d::color(0, 0, 0));
		draw_gl_handle(Matrix, Size);
	}

	void select_handle(constraint& Constraint, const k3d::matrix4& Matrix, const double Size)
	{
		k3d::gl::store_attributes attributes;
		k3d::gl::push_selection_token(Constraint.m_selection_token);

		glDisable(GL_LIGHTING);
		draw_gl_handle(Matrix, Size);

		k3d::gl::pop_selection_token();
	}

	void draw_gl_handle(const k3d::matrix4& Matrix, const double Size)
	{
		k3d::gl::store_attributes attributes;

		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		k3d::gl::push_matrix(Matrix);

		double major_step = k3d::pi_times_2() / static_cast<double>(m_handle_u_segments);
		double minor_step = k3d::pi_times_2() / static_cast<double>(m_handle_v_segments);

		for(unsigned long i = 0; i < m_handle_u_segments; ++i)
		{
			double phi = static_cast<double>(i) * major_step;
			double phi2 = phi + major_step;
			double x0 = cos(phi) * Size;
			double y0 = sin(phi) * Size;
			double x1 = cos(phi2) * Size;
			double y1 = sin(phi2) * Size;

			glBegin(GL_TRIANGLE_STRIP);
			for(unsigned long j = 0; j <= m_handle_v_segments; ++j)
			{
				double b = static_cast<double>(j) * minor_step;
				double c = cos(b);
				double r = m_handle_minor_radius * c + m_handle_major_radius;
				double z = sin(b) * Size;

				glNormal3d(x0*c, y0*c, z);
				glVertex3d(x0*r, y0*r, z * m_handle_minor_radius);

				glNormal3d(x1*c, y1*c, z);
				glVertex3d(x1*r, y1*r, z * m_handle_minor_radius);
			}
			glEnd();
		}

		glPopMatrix();
	}

	// LMB drag actions
	k3d::angle_axis lbutton_drag(viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		if(MOTION_DRAG == m_current_motion)
		{
			m_tutorial_action = "lmb_drag_move";

			const k3d::angle_axis rotation = mouse_move_to_3d(Viewport, Coordinates);
			rotate_selection(rotation);

			return rotation;
		}

		if(MOTION_BOX_SELECT == m_current_motion)
			lmb_drag_box_select(Viewport, Coordinates);

		return k3d::angle_axis(0, 1, 0, 0);
	}

	k3d::angle_axis mouse_move_action(viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		if(MOTION_CLICK_DRAG == m_current_motion)
		{
			m_tutorial_action = "mouse_drag_move";

			const k3d::angle_axis rotation = mouse_move_to_3d(Viewport, Coordinates);
			rotate_selection(rotation);

			return rotation;
		}

		return k3d::angle_axis(0, 1, 0, 0);
	}

	k3d::angle_axis mouse_move_to_3d(viewport::control& Viewport, const k3d::vector2& Coordinates)
	{
		return_val_if_fail(m_current_constraint, k3d::angle_axis(0, 1, 0, 0));

		// Transform mouse move to a world rotation
		const k3d::angle_axis rotation = m_current_constraint->mouse_move(Viewport, Coordinates);

		return rotation;
	}

	void rotate_selection(const k3d::angle_axis& Rotation)
	{
		if(!Rotation.angle)
			return;

		set_angle(Rotation.angle);

		rotate_targets(k3d::rotation3D(Rotation));
	}

	void set_angle(const double Angle)
	{
		// Prevent spinbuttons to modify values
		m_mutex = true;

		m_angle.set_value(k3d::degrees(Angle));

		m_mutex = false;
	}

	void angle_update()
	{
		if(m_mutex)
			return;

		const double angle = k3d::radians(m_angle.value());
		const k3d::normal3 axis = m_current_constraint->axis();

		rotate_targets(k3d::rotation3D(k3d::angle_axis(angle, axis)));
	}

	k3d::vector3 get_center()
	{
		return k3d::vector3(m_center_x.value(), m_center_y.value(), m_center_z.value());
	}

	void set_center(const k3d::vector3& Center)
	{
		// Prevent spinbuttons to modify values
		m_mutex = true;

		m_center_x.set_value(Center[0]);
		m_center_y.set_value(Center[1]);
		m_center_z.set_value(Center[2]);

		m_mutex = false;
	}

	void center_spin_button_update()
	{
		if(m_mutex)
			return;

		//m_center = k3d::normal3(m_center_x.value(), m_center_y.value(), m_center_z.value());
	}

	/// Stores a back-pointer to our parent
	rotate_tool& m_rotate_tool;

	/// Stores the constraint that was in effect when rotation began (if any)
	constraint* m_current_constraint;
	/// Stores the constraint that was in effect before m_current_constraint
	constraint* m_previous_constraint;

	// These are placeholders, used to identify hotspots in the manipulators
	constraint m_screen_z_constraint;
	constraint m_x_constraint;
	constraint m_y_constraint;
	constraint m_z_constraint;

	// Parameters that control the on-screen appearance of the tool
	k3d::color m_current_color;
	k3d::color m_x_color;
	k3d::color m_y_color;
	k3d::color m_z_color;
	k3d::color m_screen_z_color;
	double m_handle_size;
	double m_handle_major_radius;
	double m_handle_minor_radius;
	unsigned long m_handle_u_segments;
	unsigned long m_handle_v_segments;
	double m_screen_z_handle_size;

	// Rotation angle
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, no_serialization) m_angle;

	// Rotation center
	bool m_mutex;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, no_serialization) m_center_x;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, no_serialization) m_center_y;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, no_serialization) m_center_z;
};

/////////////////////////////////////////////////////////////////////////////
// rotate_tool

rotate_tool::rotate_tool(document_state& DocumentState, const std::string& Name) :
	base(DocumentState, Name),
	m_implementation(new implementation(DocumentState.document(), DocumentState, *this))
{
	m_implementation->navigation_model().connect_command_signal(sigc::mem_fun(*this, &rotate_tool::record_command));
}

rotate_tool::~rotate_tool()
{
	delete m_implementation;
}

void rotate_tool::on_activate()
{
	m_implementation->on_activate();
}

void rotate_tool::on_deactivate()
{
	m_implementation->on_deactivate();
}

void rotate_tool::on_document_selection_changed()
{
	m_implementation->on_selection_changed();
}

bool rotate_tool::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(m_implementation->navigation_model().execute_command(Command, Arguments))
		return true;

	if(m_implementation->execute_command(Command, Arguments))
		return true;

	return base::execute_command(Command, Arguments);
}

void rotate_tool::on_redraw(viewport::control& Viewport)
{
	m_implementation->on_redraw(Viewport);
}

void rotate_tool::on_select(viewport::control& Viewport)
{
	m_implementation->on_select(Viewport);
}

k3d::iproperty_collection* rotate_tool::get_property_collection()
{
	return dynamic_cast<k3d::iproperty_collection*>(m_implementation);
}

iuser_input_model& rotate_tool::get_input_model()
{
	return m_implementation->input_model();
}

} // namespace libk3dngui

