// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <gdkmm/cursor.h>
#include <gtkmm/widget.h>

#include "cursors.h"
#include "document_state.h"
#include "modifiers.h"
#include "icons.h"
#include "interactive.h"
#include "keyboard.h"
#include "transform_tool.h"
#include "tutorial_message.h"
#include "utility.h"
#include "viewport.h"

#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/imesh_sink.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/property.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/xml.h>

#include <k3dsdk/fstream.h>

namespace libk3dngui
{

namespace detail
{

k3d::vector3 get_selected_points(selection_mode_t SelectionMode, const k3d::mesh& Mesh, component_points_t& PointList)
{
	k3d::vector3 component_center(0, 0, 0);

	switch(SelectionMode)
	{
		case SELECT_POINTS:
		{
			// Get selected points
			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if((*point)->visible_selection)
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		case SELECT_LINES:
		{
			std::set<k3d::point*> points;

			// Get points belonging to selected lines
			for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				{
					k3d::split_edge* edge = (*face)->first_edge;
					do
					{
						if(edge->visible_selection)
						{
							points.insert(edge->vertex);
							points.insert(edge->face_clockwise->vertex);
						}

						edge = edge->face_clockwise;
					}
					while(edge != (*face)->first_edge);
				}
			}

			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if(points.find(*point) != points.end())
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		case SELECT_FACES:
		{
			std::set<k3d::point*> points;

			// Get points belonging to selected faces
			for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
				{
					if(!(*face)->visible_selection)
						continue;

					k3d::split_edge* edge = (*face)->first_edge;
					do
					{
						points.insert(edge->vertex);

						edge = edge->face_clockwise;
					}
					while(edge != (*face)->first_edge);
				}
			}

			unsigned long index = 0;
			for(k3d::mesh::points_t::const_iterator point = Mesh.points.begin(); point != Mesh.points.end(); ++point)
			{
				if(points.find(*point) != points.end())
				{
					component_point_t component_point;
					component_point.index = index;
					component_point.initial_position = (*point)->position;
					PointList.push_back(component_point);

					component_center += (*point)->position;
				}

				++index;
			}
		}
		break;
		default:
			assert_not_reached();
	}

	// Compute average position
	const double point_number = static_cast<double>(PointList.size());
	if(point_number)
		component_center /= point_number;

	return component_center;
}

} // namespace detail

	// transform_target implementation
	transform_tool::transform_target::transform_target(k3d::inode* Node)
	{
		node = Node;

		// Sanity check
		return_if_fail(node);
	}

	k3d::vector3 transform_tool::transform_target::world_position()
	{
		return k3d::world_position(*node);
	}

	unsigned long transform_tool::transform_target::target_number()
	{
		// There's always one object to move
		return 1;
	}

	void transform_tool::transform_target::reset()
	{
	}

	void transform_tool::transform_target::move(const k3d::normal3& Delta)
	{
		k3d::normal3 move = Delta;
		switch(current_system_type)
		{
			case GLOBAL:
				move = k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node))) * Delta;
				break;
			case LOCAL:
				move = Delta;
				break;
			case PARENT:
				move = k3d::inverse(k3d::extract_rotation(k3d::parent_to_world_matrix(*node))) * Delta;
				break;
			default:
				assert_not_reached();
		}

		if(create_transform_modifier(k3d::classes::Position(), "Move "))
			assert_warning(k3d::set_value(*modifier, "space", k3d::identity3D()));

		double x = boost::any_cast<double>(k3d::get_value(*modifier, "x"));
		double y = boost::any_cast<double>(k3d::get_value(*modifier, "y"));
		double z = boost::any_cast<double>(k3d::get_value(*modifier, "z"));

		assert_warning(k3d::set_value(*modifier, "x", x + move[0]));
		assert_warning(k3d::set_value(*modifier, "y", y + move[1]));
		assert_warning(k3d::set_value(*modifier, "z", z + move[2]));
	}

	void transform_tool::transform_target::init_rotation()
	{
		if(create_transform_modifier(k3d::classes::FrozenTransformation(), "Rotate "))
			assert_warning(k3d::set_value(*modifier, "matrix", k3d::identity3D()));

		m_original_rotation = boost::any_cast<k3d::matrix4>(k3d::get_value(*modifier, "matrix"));
	}

	void transform_tool::transform_target::rotate(const k3d::matrix4& RotationMatrix)
	{
		// GLOBAL orientation
		k3d::matrix4 coordinate_system_orientation = k3d::identity3D();

		if(LOCAL == current_system_type)
			coordinate_system_orientation = k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node)));
		else if(PARENT == current_system_type)
			coordinate_system_orientation = k3d::inverse(k3d::extract_rotation(k3d::parent_to_world_matrix(*node)));

		assert_warning(k3d::set_value(*modifier, "matrix", RotationMatrix * m_original_rotation * coordinate_system_orientation));
	}

	void transform_tool::transform_target::init_scaling()
	{
		if(create_transform_modifier(k3d::classes::Scale(), "Scale "))
			assert_warning(k3d::set_value(*modifier, "space", k3d::identity3D()));

		double x = boost::any_cast<double>(k3d::get_value(*modifier, "x"));
		double y = boost::any_cast<double>(k3d::get_value(*modifier, "y"));
		double z = boost::any_cast<double>(k3d::get_value(*modifier, "z"));

		m_original_scaling = k3d::vector3(x, y, z);
	}

	void transform_tool::transform_target::scale(const k3d::vector3& Delta)
	{
		k3d::vector3 scaling = Delta;
		switch(current_system_type)
		{
			case GLOBAL:
				scaling = k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node))) * Delta;
				break;
			case LOCAL:
				scaling = Delta;
				break;
			case PARENT:
				scaling = k3d::inverse(k3d::extract_rotation(k3d::parent_to_world_matrix(*node))) * Delta;
				break;
			default:
				assert_not_reached();
		}

		assert_warning(k3d::set_value(*modifier, "x", m_original_scaling[0] * scaling[0]));
		assert_warning(k3d::set_value(*modifier, "y", m_original_scaling[1] * scaling[1]));
		assert_warning(k3d::set_value(*modifier, "z", m_original_scaling[2] * scaling[2]));
	}

	bool transform_tool::transform_target::create_transform_modifier(const k3d::uuid& Class, const std::string& Name)
	{
		if(modifier)
			return false;

		return_val_if_fail(node, false);

		// Check for an existing transform modifier
		k3d::inode* upstream_node = upstream_transform_modifier(*node);
		/** \todo check for same name too */
		if(upstream_node && (Class == upstream_node->factory().class_id()))
		{
			set_transform_modifier(upstream_node);
			return false;
		}

		const std::string modifier_name = Name + node->name();
		set_transform_modifier(insert_transform_modifier(*node, Class, modifier_name));

		return true;
	}

	// mesh_target implementation
	transform_tool::mesh_target::mesh_target(selection_mode_t SelectionMode, k3d::inode* Node, k3d::iproperty& MeshSourceProperty) :
		mesh_source_property(MeshSourceProperty),
		component_center(k3d::vector3(0, 0, 0))
	{
		node = Node;

		// Sanity checks
		return_if_fail(node);

		// Save initial position
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);
		initial_component_center = component_center = detail::get_selected_points(SelectionMode, *mesh, selected_points);
	}

	k3d::vector3 transform_tool::mesh_target::world_position()
	{
		return component_center;
	}

	unsigned long transform_tool::mesh_target::target_number()
	{
		return selected_points.size();
	}

	void transform_tool::mesh_target::reset()
	{
		component_center = initial_component_center;
	}

	void transform_tool::mesh_target::move(const k3d::normal3& Delta)
	{
		k3d::normal3 move = Delta;
		switch(current_system_type)
		{
			case GLOBAL:
				move = k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node))) * Delta;
				component_center += Delta;
				break;
			case LOCAL:
				move = Delta;
				component_center += k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node))) * Delta;
				break;
			case PARENT:
				move = k3d::inverse(k3d::extract_rotation(k3d::parent_to_world_matrix(*node))) * Delta;
				//component_center += ;
				break;
			default:
				assert_not_reached();
		}

		create_mesh_modifier("Move ");
		move_components(move);
	}

	void transform_tool::mesh_target::init_rotation()
	{
		create_mesh_modifier("Rotate ");

		// Save initial positions
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);
		for(detail::component_points_t::iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const unsigned long index = point->index;
			point->initial_position = mesh->points[index]->position;
			point->tweak_value = tweaks[index];
		}
	}

	void transform_tool::mesh_target::rotate(const k3d::matrix4& RotationMatrix)
	{
		rotate_components(RotationMatrix);
	}

	void transform_tool::mesh_target::init_scaling()
	{
		create_mesh_modifier("Scale ");

		// Save initial positions
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);
		for(detail::component_points_t::iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const unsigned long index = point->index;
			point->initial_position = mesh->points[index]->position;
			point->tweak_value = tweaks[index];
		}
	}

	void transform_tool::mesh_target::scale(const k3d::vector3& Delta)
	{
		k3d::vector3 scaling = Delta;
		switch(current_system_type)
		{
			case GLOBAL:
				scaling = k3d::inverse(k3d::extract_rotation(k3d::node_to_world_matrix(*node))) * Delta;
				break;
			case LOCAL:
				scaling = Delta;
				break;
			case PARENT:
				scaling = k3d::inverse(k3d::extract_rotation(k3d::parent_to_world_matrix(*node))) * Delta;
				break;
			default:
				assert_not_reached();
		}

		scale_components(scaling);
	}

	void transform_tool::mesh_target::create_mesh_modifier(const std::string& Name)
	{
		if(modifier)
			return;

		return_if_fail(node);

		// Get mesh to tweak
		k3d::mesh* mesh = boost::any_cast<k3d::mesh*>(mesh_source_property.property_value());
		return_if_fail(mesh);

		// Modify with TweakPoints
		const k3d::uuid tweak_points(0xed302b87, 0x49bf4fe6, 0x99064963, 0x17ec12d9);

		// Check for an existing mesh modifier
		k3d::inode* upstream_node = upstream_mesh_modifier(*node);
		if(upstream_node && (tweak_points == upstream_node->factory().class_id()))
		{
			set_transform_modifier(upstream_node);

			// Init tweaks
			tweaks = boost::any_cast<tweaks_t>(k3d::get_value(*modifier, "tweaks"));
			tweaks.resize(mesh->points.size(), k3d::vector3(0, 0, 0));

			return;
		}

		// Create a new TweakPoints modifier
		const std::string modifier_name = Name + node->name() + " components";
		set_transform_modifier(insert_mesh_modifier(*node, tweak_points, modifier_name));

		// Initialize new TweakPoints modifier
		tweaks.resize(mesh->points.size(), k3d::vector3(0, 0, 0));
		assert_warning(k3d::set_value(*modifier, "tweaks", tweaks));
	}

	void transform_tool::mesh_target::move_components(const k3d::normal3& Move)
	{
		return_if_fail(modifier);

		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			tweaks[point->index] += Move;
		}

		assert_warning(k3d::set_value(*modifier, "tweaks", tweaks));
	}

	void transform_tool::mesh_target::rotate_components(const k3d::matrix4& RotationMatrix)
	{
		return_if_fail(modifier);

		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const k3d::vector3 position = point->initial_position;
			const k3d::vector3 new_position = RotationMatrix * (position - component_center) + component_center;

			tweaks[point->index] = (new_position - position) + point->tweak_value;
		}

		assert_warning(k3d::set_value(*modifier, "tweaks", tweaks));
	}

	void transform_tool::mesh_target::scale_components(const k3d::vector3& Scaling)
	{
		return_if_fail(modifier);

		const k3d::matrix4 scaling_3d = k3d::scaling3D(Scaling);

		for(detail::component_points_t::const_iterator point = selected_points.begin(); point != selected_points.end(); ++point)
		{
			const k3d::vector3 position = point->initial_position;
			const k3d::vector3 new_position = scaling_3d * (position - component_center) + component_center;

			tweaks[point->index] += (new_position - position) + point->tweak_value;
		}

		assert_warning(k3d::set_value(*modifier, "tweaks", tweaks));
	}

// transform_tool implementation
void transform_tool::lbutton_down(viewport::control& Viewport, const k3d::vector2& Coordinates, const k3d::key_modifiers& Modifiers)
{
	// Return if an action is in progress
	if(MOTION_CLICK_DRAG == m_current_motion)
		return;

	assert_warning(MOTION_NONE == m_current_motion);

	// Init action
	m_mouse_down_content = NOTHING;

	// Find what's under the mouse pointer
	k3d::selection::records picked_selectables;
	pick_selectables(picked_selectables, Viewport, Coordinates);

	// Shift modifier starts ADD action
	if(Modifiers.shift())
	{
		lmb_down_add();
		return;
	}

	// Control modifier starts SUBTRACT action
	if(Modifiers.control())
	{
		lmb_down_subtract();
		return;
	}

	// Manipulator selection starts move
	detail::abstract_tool::manipulators_t manipulators;
	for(k3d::selection::records::iterator record = picked_selectables.begin(); record != picked_selectables.end(); ++record)
	{
		for(k3d::selection::record::tokens_t::const_iterator token = record->tokens.begin(); token != record->tokens.end(); ++token)
		{
			if(token->type == k3d::selection::USER1)
				manipulators.push_back(m_abstract_tool.manipulator_name(token->id));
		}
	}

	const std::string manipulator = m_abstract_tool.get_manipulator(manipulators);
	if(manipulator != "")
	{
		lmb_down_manipulator(manipulator);
		return;
	}

	// If a node was hit ...
	if(k3d::selection::get_node(m_mouse_down_selection))
	{
		if(m_document_state.is_selected(m_mouse_down_selection))
			lmb_down_selected();
		else
			lmb_down_deselected();

		return;
	}

	lmb_down_nothing();
}

// LMB down actions
void transform_tool::lmb_down_add()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_add";

	m_mouse_down_content = SELECTION_ADD;
}

void transform_tool::lmb_down_subtract()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_subtract";

	m_mouse_down_content = SELECTION_SUBTRACT;
}

void transform_tool::lmb_down_manipulator(const std::string& ManipulatorName)
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_manipulator_" + ManipulatorName;

	m_abstract_tool.set_manipulator(ManipulatorName);
	set_motion(MOTION_DRAG);

	m_mouse_down_content = SELECTED_OBJECT;
}

void transform_tool::lmb_down_selected()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_selected";

	m_mouse_down_content = SELECTED_OBJECT;
}

void transform_tool::lmb_down_deselected()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_deselected";

	m_mouse_down_content = DESELECTED_OBJECT;

	// Deselect all
	m_document_state.deselect_all();
	// Select clicked object
	m_document_state.select(m_mouse_down_selection);
}

void transform_tool::lmb_down_nothing()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "lmb_down_nothing";

	m_mouse_down_content = NOTHING;
}

void transform_tool::lbutton_click(const viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	if(MOTION_CLICK_DRAG == m_current_motion)
	{
		// Stop click-drag
		lmb_click_stop_motion();
		return;
	}

	switch(m_mouse_down_content)
	{
		case SELECTION_ADD:
			lmb_click_add();
		break;
		case SELECTION_SUBTRACT:
			lmb_click_subtract();
		break;
		case SELECTED_OBJECT:
			lmb_click_start_motion(Coordinates);
		break;
		case DESELECTED_OBJECT:
			lmb_click_replace();
		break;
		case NOTHING:
			lmb_click_deselect_all();
		break;
		default:
			assert_not_reached();
	}
}

// LMB click actions
void transform_tool::lmb_click_add()
{
	m_tutorial_action = "lmb_click_add";

	// Shift key modifier always adds to the selection
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.select(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection add");

	redraw_all();
}

void transform_tool::lmb_click_subtract()
{
	m_tutorial_action = "lmb_click_subtract";

	// Control key modifier always adds to the selection
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.deselect(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection subtract");

	redraw_all();
}

void transform_tool::lmb_click_replace()
{
	m_tutorial_action = "lmb_click_replace";

	// Replace selection
	m_document_state.deselect_all();
	if(k3d::selection::get_node(m_mouse_down_selection))
		m_document_state.select(m_mouse_down_selection);

	k3d::finish_state_change_set(m_document, "Selection replace");

	redraw_all();
}

void transform_tool::lmb_click_start_motion(const k3d::vector2& Coordinates)
{
	m_tutorial_action = "lmb_click_start_motion";

	set_motion(MOTION_CLICK_DRAG);
	m_abstract_tool.begin_mouse_move(Coordinates);

	redraw_all();
}

void transform_tool::lmb_click_stop_motion()
{
	m_tutorial_action = "lmb_click_stop_motion";

	const std::string label = complete_mouse_move();
	k3d::finish_state_change_set(m_document, label);

	redraw_all();
}

void transform_tool::lmb_click_deselect_all()
{
	m_tutorial_action = "lmb_click_deselect_all";

	// Deselect all
	m_document_state.deselect_all();

	k3d::finish_state_change_set(m_document, "Deselect all");

	redraw_all();
}

void transform_tool::lbutton_start_drag(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	switch(m_mouse_down_content)
	{
		case SELECTED_OBJECT:
		case DESELECTED_OBJECT:
			lmb_start_drag_start_motion(Coordinates);
		break;
		case SELECTION_ADD:
		case SELECTION_SUBTRACT:
		case NOTHING:
			lmb_start_drag_box_select(Viewport, Coordinates);
		break;
		default:
			assert_not_reached();
	}
}

// LMB start drag actions
void transform_tool::lmb_start_drag_start_motion(const k3d::vector2& Coordinates)
{
	m_tutorial_action = "lmb_start_drag_start_motion";

	set_motion(MOTION_DRAG);
	m_abstract_tool.begin_mouse_move(Coordinates);
}

void transform_tool::lmb_start_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	m_tutorial_action = "lmb_start_drag_box_select";

	set_motion(MOTION_BOX_SELECT);
	m_box_selection = k3d::rectangle(Coordinates[0], Coordinates[0], Coordinates[1], Coordinates[1]);
	draw_rubber_band(Viewport);
}

void transform_tool::lmb_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	m_tutorial_action = "lmb_drag_box_select";

	on_box_select_motion(Viewport, Coordinates);
}

void transform_tool::lbutton_end_drag(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	if(MOTION_DRAG == m_current_motion)
		lmb_end_drag_stop_motion();
	else if(MOTION_BOX_SELECT == m_current_motion)
		lmb_end_drag_box_select(Viewport, Coordinates);
}

// LMB end drag actions
void transform_tool::lmb_end_drag_stop_motion()
{
	m_tutorial_action = "lmb_end_drag_stop_motion";

	const std::string label = complete_mouse_move();
	k3d::finish_state_change_set(m_document, label);

	redraw_all();
}

void transform_tool::lmb_end_drag_box_select(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	m_tutorial_action = "lmb_end_drag_box_select";

	draw_rubber_band(Viewport);

	on_box_select_objects(Viewport, Coordinates, m_box_selection);

	// Stop motion
	set_motion(MOTION_NONE);

	k3d::finish_state_change_set(m_document, "Box selection");

	redraw_all();
}

void transform_tool::mbutton_click(viewport::control& Viewport, const k3d::vector2& Coordinates, const k3d::key_modifiers& Modifiers)
{
	// Motion mode
	if(MOTION_NONE != m_current_motion)
	{
		mmb_click_next_constraint(Viewport, Coordinates);
		return;
	}

	// We aren't moving
	if(Modifiers.control())
	{
		mmb_click_switch_coordinate_system();
	}
	else if(Modifiers.shift())
	{
		mmb_click_manipulators_next_selection();
	}
	else
	{
		mmb_click_toggle_manipulators_visibility();
	}
}

void transform_tool::mmb_click_toggle_manipulators_visibility()
{
	m_tutorial_action = "mmb_click_toggle_manipulators_visibility";

	// Toggle the visible state of the manipulators
	m_visible_manipulators.set_value(!m_visible_manipulators.value());

	redraw_all();
}

void transform_tool::mmb_click_manipulators_next_selection()
{
	m_tutorial_action = "mmb_click_manipulators_next_selection";

	// Show manipulators on the next selection
	m_current_target = m_targets.size() ? (m_current_target + 1) % m_targets.size() : 0;

	redraw_all();
}

void transform_tool::mmb_click_switch_coordinate_system()
{
	m_tutorial_action = "mmb_click_switch_coordinate_system";

	// Switch coordinate system between global and local modes
	switch(m_coordinate_system.value())
	{
		case COORDINATE_SYSTEM_GLOBAL:
			set_coordinate_system(COORDINATE_SYSTEM_LOCAL);
			break;
		case COORDINATE_SYSTEM_LOCAL:
			set_coordinate_system(COORDINATE_SYSTEM_GLOBAL);
			break;
		default:
			break;
	}

	redraw_all();
}

void transform_tool::mmb_click_next_constraint(viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	m_tutorial_action = "mmb_click_next_constraint";

	// Set next constraint
	m_abstract_tool.update_constraint(Viewport, Coordinates);

/*
	// Mouse warp
	const k3d::vector2 coords = Viewport.project(world_position());

	int root_x = 0;
	int root_y = 0;
	Viewport.get_window()->get_origin(root_x, root_y);

	const k3d::vector2 screen_coords = k3d::vector2(coords[0] + root_x, coords[1] + root_y);

	// We temporarily disable motion, so warping the pointer doesn't cause unintended side effects
	const motion_t current_motion = m_current_motion;
	m_current_motion = MOTION_NONE;
	interactive::warp_pointer(screen_coords);
	handle_pending_events();

	// Acknowledge new mouse position
	m_current_constraint->begin_mouse_move(coords);

	m_current_motion = current_motion;
*/

	redraw_all();
}

void transform_tool::rbutton_click(const viewport::control& Viewport, const k3d::vector2& Coordinates)
{
	if(MOTION_NONE == m_current_motion)
		rmb_click_selection_tool();
	else
		rmb_click_cancel_move();
}

// RMB click actions
void transform_tool::rmb_click_selection_tool()
{
	k3d::start_state_change_set(m_document);
	m_tutorial_action = "rmb_click_selection_tool";

	m_document_state.set_active_tool(m_document_state.selection_tool());

	k3d::finish_state_change_set(m_document, "Selection tool");

	redraw_all();
}

void transform_tool::rmb_click_cancel_move()
{
	m_tutorial_action = "rmb_click_cancel_move";

	cancel_mouse_move();

	redraw_all();
}

void transform_tool::cancel_mouse_move()
{
	// Stop motion
	set_motion(MOTION_NONE);

	// Undo changes
	k3d::cancel_state_change_set(m_document);

	// Reset targets
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->reset();

	redraw_all();
}

std::string transform_tool::complete_mouse_move()
{
	set_motion(MOTION_NONE);

	return m_abstract_tool.get_constraint_name();
}

void transform_tool::set_motion(const motion_t Motion)
{
	m_current_motion = Motion;
}

/// Returns current target's world position
k3d::vector3 transform_tool::world_position()
{
	if(target_number())
	{
		if(SELECT_NODES == m_document_state.selection_mode().value())
		{
			m_current_target = m_current_target % m_targets.size();
			itarget* t = m_targets[m_current_target];
			return t->world_position();
		}
		else
		{
			k3d::vector3 position(0, 0, 0);
			unsigned long count = 0;
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
			{
				if((*target)->target_number())
				{
					position += (*target)->world_position();
					count++;
				}
			}

			position /= static_cast<double>(count);

			return position;
		}
	}

	return k3d::vector3(0, 0, 0);
}

/// Returns current target's world orientation
k3d::matrix4 transform_tool::world_orientation()
{
	if(target_number())
	{
		m_current_target = m_current_target % m_targets.size();
		itarget* t = m_targets[m_current_target];

		return t->world_orientation();
	}

	return k3d::identity3D();
}

void transform_tool::update_manipulators_scale(viewport::control& Viewport, const k3d::vector3& Origin)
{
	k3d::icamera* camera = Viewport.camera();
	return_if_fail(camera);

	// Project unit axis on screen space
	const k3d::matrix4 screen_matrix = k3d::node_to_world_matrix(*Viewport.camera());
	const k3d::normal3 screen_parallel = screen_matrix * k3d::normal3(1, 0, 0);
	const k3d::vector2 position = Viewport.project(Origin);
	const k3d::vector2 x_axis = Viewport.project(Origin + screen_parallel);
	const double length = (position - x_axis).Length();

	return_if_fail(length);
	m_manipulators_scale = m_manipulators_size / length;
}

bool transform_tool::front_facing(viewport::control& Viewport, const k3d::normal3& Normal, const k3d::vector3& Origin)
{
	return_val_if_fail(Viewport.gl_engine(), false);
	return_val_if_fail(Viewport.camera(), false);

	const k3d::matrix4 matrix = k3d::inverse(k3d::node_to_world_matrix(*Viewport.camera()));
	const k3d::matrix4 orientation = world_orientation();
	const k3d::vector3 a = Origin + (orientation * Normal);
	const k3d::vector3 b = Origin + (orientation * -Normal);
	return (matrix * a).Length2() < (matrix * b).Length2();
}

bool transform_tool::off_screen_warp(viewport::control& Viewport, k3d::vector2& NewCoordinates)
{
	// Get mouse coordinates
	int x, y;
	Gdk::ModifierType modifiers;
	Gdk::Display::get_default()->get_pointer(x, y, modifiers);
	k3d::vector2 mouse(x, y);

	// Check for screen warp
	bool screen_warp = false;

	// Wrap the mouse if it goes off the top-or-bottom of the screen ...
	const int screen_height = Gdk::Display::get_default()->get_default_screen()->get_height();
	const int border = 5;
	if(y < border)
	{
		mouse = k3d::vector2(mouse[0], screen_height - (border + 1));
		screen_warp = true;
	}
	else if(screen_height - y < border)
	{
		mouse = k3d::vector2(mouse[0], (border + 1));
		screen_warp = true;
	}

	// Wrap the mouse if it goes off the left-or-right of the screen ...
	const int screen_width = Gdk::Display::get_default()->get_default_screen()->get_width();
	if(x < border)
	{
		mouse = k3d::vector2(screen_width - (border + 1), mouse[1]);
		screen_warp = true;
	}
	else if(screen_width - x < border)
	{
		mouse = k3d::vector2((border + 1), mouse[1]);
		screen_warp = true;
	}

	// No warp
	if(!screen_warp)
		return false;

	// Warp mouse pointer
	interactive::warp_pointer(mouse);

	// Give new position
	int root_x = 0;
	int root_y = 0;
	Viewport.get_window()->get_origin(root_x, root_y);
	NewCoordinates = mouse - k3d::vector2(root_x, root_y);

	return true;
}

void transform_tool::clear_targets()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
	{
		delete *target;
	}

	m_targets.clear();
}

/// Retrieves the current document selection, refreshing the target list
void transform_tool::get_current_selection()
{
	// Convert the current document selection into the set of targets to be moved interactively
	clear_targets();

	const k3d::nodes_t nodes = m_document_state.selected_nodes();

	if(SELECT_NODES == m_document_state.selection_mode().value())
	{
		// Save transformable nodes as targets
		for(k3d::nodes_t::const_iterator node = nodes.begin(); node != nodes.end(); ++node)
		{
			if(!dynamic_cast<k3d::gl::idrawable*>(*node))
				continue;
			if(!dynamic_cast<k3d::itransform_sink*>(*node))
				continue;

			m_targets.push_back(new transform_target(*node));
			(*node)->deleted_signal().connect(sigc::mem_fun(*this, &transform_tool::target_deleted));
		}
	}
	else
	{
		// Component mode : save mesh nodes as targets
		for(k3d::nodes_t::const_iterator node = nodes.begin(); node != nodes.end(); ++node)
		{
			if(!dynamic_cast<k3d::gl::idrawable*>(*node))
				continue;

			// Get node's mesh
			k3d::imesh_source* const mesh_source = dynamic_cast<k3d::imesh_source*>(*node);
			if(!mesh_source)
				continue;

			k3d::iproperty& property = mesh_source->mesh_source_output();
			m_targets.push_back(new mesh_target(m_document_state.selection_mode().value(), *node, property));
			(*node)->deleted_signal().connect(sigc::mem_fun(*this, &transform_tool::target_deleted));
		}
	}

	update_coordinate_system();
}

unsigned long transform_tool::target_number()
{
	unsigned long target_count = 0;
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		target_count += (*target)->target_number();

	return target_count;
}

void transform_tool::update_targets()
{
	// Update target list when one of them was deleted
	if(m_deleted_target)
	{
		get_current_selection();
		m_deleted_target = false;
	}
}

void transform_tool::move_targets(const k3d::normal3& Delta)
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->move(Delta);

	redraw_all();
}

void transform_tool::init_rotation()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->init_rotation();
}

void transform_tool::rotate_targets(const k3d::matrix4& Rotation)
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->rotate(Rotation);

	redraw_all();
}

void transform_tool::init_scaling()
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->init_scaling();
}

void transform_tool::scale_targets(const k3d::vector3& Scaling)
{
	for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
		(*target)->scale(Scaling);

	redraw_all();
}

void transform_tool::set_coordinate_system(const coordinate_system_t CoordinateSystem)
{
	m_coordinate_system.set_value(CoordinateSystem);
}

/// Updates all targets : tells them what's the new coordinate system
void transform_tool::update_coordinate_system()
{
	switch(m_coordinate_system.value())
	{
		case COORDINATE_SYSTEM_GLOBAL:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_global();
			break;
		case COORDINATE_SYSTEM_LOCAL:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_local();
			break;
		case COORDINATE_SYSTEM_PARENT:
			for(targets_t::iterator target = m_targets.begin(); target != m_targets.end(); ++target)
				(*target)->set_parent();
			break;
		default:
			assert_not_reached();
	}

	redraw_all();
}

} // namespace libk3dngui

