// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\brief Implements a "Tutorial Message" dialog for prompting the user and providing feedback during interactive tutorial playback
	\author Tim Shead (tshead@k-3d.com)
*/

#include "application_state.h"
#include "button.h"
#include "check_button.h"
#include "interactive.h"
#include "messages.h"
#include "open_uri.h"
#include "options.h"
#include "tutorial_message.h"
#include "utility.h"
#include "window.h"

#include <k3dsdk/application.h>
#include <k3dsdk/data.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iuser_interface.h>
#include <k3dsdk/string_cast.h>

#include <gdkmm/pixbuf.h>
#include <gtkmm/adjustment.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/frame.h>
#include <gtkmm/image.h>
#include <gtkmm/label.h>
#include <gtkmm/scrollbar.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/textview.h>

#ifndef K3D_PLATFORM_WIN32
#include <boost/regex.hpp>
#endif // !K3D_PLATFORM_WIN32

#include <iomanip>

namespace libk3dngui
{

namespace detail
{

/*
//////////////////////////////////////////////////////////////////////////////////////////////////////////
// modifiers_diagram

std::string modifiers_diagram(const k3d::key_modifiers Modifiers)
{
	return "modifiers_" + k3d::string_cast(Modifiers & k3d::key_modifiers().set_shift().set_lock().set_control());
}
*/

/////////////////////////////////////////////////////////////////////////////////////////////////////////
// mouse_description

std::string mouse_description(const k3d::iuser_interface::mouse_action_t Action)
{
	std::string result;

	switch(Action)
	{
		case k3d::iuser_interface::MOUSE_DONE:
			break;
		case k3d::iuser_interface::MB_NONE:
			result = _("Drag the mouse.\n");
			break;
		case k3d::iuser_interface::LMB_DRAG:
			result = _("Hold down the Left Mouse Button (LMB) and drag.\n");
			break;
		case k3d::iuser_interface::LMB_CLICK:
			result = _("Click the Left Mouse Button (LMB).\n");
			break;
		case k3d::iuser_interface::LMB_DOUBLE_CLICK:
			result = _("Double-click the Left Mouse Button (LMB).\n");
			break;
		case k3d::iuser_interface::MMB_DRAG:
			result = _("Hold down the Middle Mouse Button (MMB) and drag.\n");
			break;
		case k3d::iuser_interface::MMB_CLICK:
			result = _("Click the Middle Mouse Button (MMB).\n");
			break;
		case k3d::iuser_interface::MMB_DOUBLE_CLICK:
			result = _("Double-click the Middle Mouse Button (MMB).\n");
			break;
		case k3d::iuser_interface::RMB_DRAG:
			result = _("Hold down the Right Mouse Button (RMB) and drag.\n");
			break;
		case k3d::iuser_interface::RMB_CLICK:
			result = _("Click the Right Mouse Button (RMB).\n");
			break;
		case k3d::iuser_interface::RMB_DOUBLE_CLICK:
			result = _("Double-click the Right Mouse Button (RMB).\n");
			break;
		case k3d::iuser_interface::LMBRMB_DRAG:
			result = _("Hold the the Left (LMB) and Right (RMB) mouse buttons and drag.\n");
			break;
		default:
			return_val_if_fail(0, result);	// Did you add a mouse action without telling me?
	}

	return result;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// modifiers_description

std::string modifiers_description(const k3d::key_modifiers Modifiers)
{
	std::string result;

	if(Modifiers.lock())
		result += _("Turn on CAPS LOCK.\n");

	if(Modifiers.shift())
		result += _("Hold down SHIFT.\n");

	if(Modifiers.control())
		result += _("Hold down CTRL.\n");

	if(Modifiers.mod1())
		result += _("Hold down ALT.\n");

	return result;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// action_description

const std::string action_description(const std::string& Message, const k3d::iuser_interface::mouse_action_t Action, const k3d::key_modifiers Modifiers)
{
	std::string result;

	if(Message.size())
		result += Message + "\n\n";

	const std::string modifiersdescription = modifiers_description(Modifiers);
	if(modifiersdescription.size())
		result += modifiersdescription;

	const std::string mousedescription = mouse_description(Action);
	if(mousedescription.size())
		result += mousedescription;

	return result;
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// tutorial_message_window

class tutorial_message_window :
	public window
{
	typedef window base;

public:
	tutorial_message_window() :
		base("tutorial_message", 0),
		m_quit(*this, "quit", _("Quit"), Gtk::Stock::CANCEL),
		m_continue(*this, "continue", _("Continue"), Gtk::Stock::OK),
		m_run_state(IDLE),
		m_cruise_control(init_name("cruise_control") + init_label(_("Cruise Control")) + init_description(_("When on, continues tutorial without waiting for user action (e.g. mouse click)")) + init_value(application_state().batch_mode()))
	{
		m_cruise_control.changed_signal().connect(sigc::mem_fun(*this, &tutorial_message_window::on_cruise_control));

		set_title(_("K-3D Tutorial"));
		set_role("tutorial_message_window");
		resize(500, 170);
		set_position(Gtk::WIN_POS_CENTER);
		set_border_width(5);

		m_mouse_diagrams[k3d::iuser_interface::MOUSE_DONE] = load_pixbuf("mouse_dim.xpm");
		m_mouse_diagrams[k3d::iuser_interface::MB_NONE] = load_pixbuf("mouse.xpm");
		m_mouse_diagrams[k3d::iuser_interface::LMB_DRAG] = load_pixbuf("mouse_lmb.xpm");
		m_mouse_diagrams[k3d::iuser_interface::LMB_CLICK] = load_pixbuf("mouse_lmb_1.xpm");
		m_mouse_diagrams[k3d::iuser_interface::LMB_DOUBLE_CLICK] = load_pixbuf("mouse_lmb_2.xpm");
		m_mouse_diagrams[k3d::iuser_interface::MMB_DRAG] = load_pixbuf("mouse_mmb.xpm");
		m_mouse_diagrams[k3d::iuser_interface::MMB_CLICK] = load_pixbuf("mouse_mmb_1.xpm");
		m_mouse_diagrams[k3d::iuser_interface::MMB_DOUBLE_CLICK] = load_pixbuf("mouse_mmb_2.xpm");
		m_mouse_diagrams[k3d::iuser_interface::RMB_DRAG] = load_pixbuf("mouse_rmb.xpm");
		m_mouse_diagrams[k3d::iuser_interface::RMB_CLICK] = load_pixbuf("mouse_rmb_1.xpm");
		m_mouse_diagrams[k3d::iuser_interface::RMB_DOUBLE_CLICK] = load_pixbuf("mouse_rmb_2.xpm");
		m_mouse_diagrams[k3d::iuser_interface::LMBRMB_DRAG] = load_pixbuf("mouse_lmb_rmb.xpm");

		m_keyboard_diagrams[k3d::key_modifiers()] = load_pixbuf("modifiers_none.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_shift()] = load_pixbuf("modifiers_shift.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_shift().set_lock()] = load_pixbuf("modifiers_shift_lock.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_shift().set_control()] = load_pixbuf("modifiers_shift_control.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_shift().set_lock().set_control()] = load_pixbuf("modifiers_shift_lock_control.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_lock()] = load_pixbuf("modifiers_lock.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_lock().set_control()] = load_pixbuf("modifiers_lock_control.xpm");
		m_keyboard_diagrams[k3d::key_modifiers().set_control()] = load_pixbuf("modifiers_control.xpm");

		m_mouse_diagram.set(m_mouse_diagrams[k3d::iuser_interface::MOUSE_DONE]);
		m_keyboard_diagram.set(m_keyboard_diagrams[k3d::key_modifiers()]);

		m_message.set_wrap_mode(Gtk::WRAP_WORD);
		m_message.set_editable(false);
		m_message.set_justification(Gtk::JUSTIFY_LEFT);
		m_message.add_events(Gdk::ENTER_NOTIFY_MASK);
		m_message.add_events(Gdk::LEAVE_NOTIFY_MASK);

		m_message_tag_url = Gtk::TextBuffer::Tag::create("url");
		m_message_tag_url->property_foreground() = "blue";
		m_message_tag_url->property_underline() = Pango::UNDERLINE_SINGLE;
		m_message_tag_url->signal_event().connect(sigc::mem_fun(*this, &tutorial_message_window::on_url_event));
		m_message.get_buffer()->get_tag_table()->add(m_message_tag_url);

		Gtk::ScrolledWindow* const scrolled_window = new Gtk::ScrolledWindow();
		scrolled_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		scrolled_window->add(m_message);

		Gtk::Frame* const frame1 = new Gtk::Frame();
		frame1->add(m_keyboard_diagram);

		Gtk::Frame* const frame2 = new Gtk::Frame();
		frame2->add(m_mouse_diagram);

		Gtk::HBox* const hbox3 = new Gtk::HBox(false);
		hbox3->pack_start(*manage(frame1), Gtk::PACK_SHRINK);
		hbox3->pack_start(*manage(frame2), Gtk::PACK_SHRINK);
		hbox3->pack_start(*manage(scrolled_window), Gtk::PACK_EXPAND_WIDGET);

		m_speed.get_adjustment()->set_lower(-0.5);
		m_speed.get_adjustment()->set_upper(1.0);
		m_speed.get_adjustment()->set_value(0.0);
		m_speed.get_adjustment()->set_step_increment(0.1);
		m_speed.get_adjustment()->set_page_size(0.5);
		m_speed.get_adjustment()->signal_value_changed().connect(sigc::mem_fun(*this, &tutorial_message_window::on_speed_changed));
		m_speed.get_adjustment()->set_value(log10(options::tutorial_speed()));
		on_speed_changed();

		Gtk::HBox* const hbox2 = new Gtk::HBox(true);
		hbox2->pack_start(m_speed, Gtk::PACK_EXPAND_WIDGET);
		hbox2->pack_start(m_speed_label, Gtk::PACK_SHRINK);

		m_quit.enable_recording(false);
		m_continue.enable_recording(false);

		m_quit.signal_clicked().connect(sigc::mem_fun(*this, &tutorial_message_window::on_quit));
		m_continue.signal_clicked().connect(sigc::mem_fun(*this, &tutorial_message_window::on_continue));

		m_continue.property_can_default() = true;

		Gtk::HButtonBox* const buttonbox = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
		buttonbox->pack_start(m_quit);
		buttonbox->pack_start(m_continue);

		Gtk::HBox* const hbox1 = new Gtk::HBox(false, 6);
		hbox1->pack_start(*Gtk::manage(new Gtk::Label(_("Speed:"))), Gtk::PACK_SHRINK);
		hbox1->pack_start(*Gtk::manage(hbox2), Gtk::PACK_EXPAND_WIDGET);
		hbox1->pack_start(*Gtk::manage(new check_button::control(*this, "cruise_control", check_button::proxy(m_cruise_control), _("Cruise Control"), Gtk::PACK_SHRINK)));
		hbox1->pack_start(*Gtk::manage(buttonbox), Gtk::PACK_SHRINK);

		Gtk::VBox* const vbox1 = new Gtk::VBox(false, 5);
		vbox1->pack_start(*Gtk::manage(hbox3), Gtk::PACK_EXPAND_WIDGET);
		vbox1->pack_start(*Gtk::manage(hbox1), Gtk::PACK_SHRINK);

		add(*Gtk::manage(vbox1));

		realize();
	}

	bool on_url_event(const Glib::RefPtr<Glib::Object>& EventObject, GdkEvent* Event, const Gtk::TextIter& Begin)
	{
		switch(Event->type)
		{
			case GDK_BUTTON_PRESS:
			{
				Gtk::TextIter begin = Begin;
				if(!begin.begins_tag(m_message_tag_url))
					begin.backward_to_tag_toggle(m_message_tag_url);

				Gtk::TextIter end = Begin;
				end.forward_to_tag_toggle(m_message_tag_url);

				open_uri(begin.get_text(end));
				break;
			}
			default:
				break;
		}

		return false;
	}

#ifdef K3D_PLATFORM_WIN32
	void apply_tag(const std::string& Message, const std::string& RegularExpression, const Glib::RefPtr<Gtk::TextBuffer::Tag>& Tag)
	{
		/** \todo Get boost::regex to build in Code::Blocks so we can enable this */
		assert_not_implemented();
	}
#else // K3D_PLATFORM_WIN32
	void apply_tag(const std::string& Message, const std::string& RegularExpression, const Glib::RefPtr<Gtk::TextBuffer::Tag>& Tag)
	{
		std::string::const_iterator begin = Message.begin();
		std::string::const_iterator end = Message.end();
		boost::regex regex(RegularExpression);
		boost::match_results<std::string::const_iterator> match;
		while(boost::regex_search(begin, end, match, regex))
		{
			m_message.get_buffer()->apply_tag(
				Tag,
				m_message.get_buffer()->get_iter_at_offset(std::distance(begin, match[0].first)),
				m_message.get_buffer()->get_iter_at_offset(std::distance(begin, match[0].second)));

			begin = match[0].second;
		}
	}
#endif // !K3D_PLATFORM_WIN32

	void set_message(const std::string& Message)
	{
		m_message.get_buffer()->set_text(Message);
		apply_tag(Message, "http://[^[:space:]]*", m_message_tag_url);

		Gtk::TextBuffer::iterator i = m_message.get_buffer()->begin();
		m_message.scroll_to_iter(i, 0.0);
	}

	bool tutorial_message(const std::string& Message)
	{
		show_all();
		raise();

		set_message(Message);
		m_mouse_diagram.set(m_mouse_diagrams[k3d::iuser_interface::MOUSE_DONE]);
		m_keyboard_diagram.set(m_keyboard_diagrams[k3d::key_modifiers()]);

		m_quit.set_sensitive(true);
		m_continue.set_sensitive(true);
		m_continue.grab_default();
		m_continue.grab_focus();

		interactive::show(m_continue);
		interactive::move_pointer(m_continue);

		if(m_cruise_control.value())
		{
			interactive::activate(m_continue);
			return true;
		}

		set_modal(true);
		m_run_state = RUN;
		while(m_run_state == RUN)
			handle_pending_events();
		const bool result = CONTINUE == m_run_state ? true : false;
		m_run_state = IDLE;

		return result;
	}

	void tutorial_mouse_message(const std::string& Message, const k3d::iuser_interface::mouse_action_t Action, const k3d::key_modifiers Modifiers)
	{
		show_all();
		raise();

		set_message(detail::action_description(Message, Action, Modifiers));
		m_mouse_diagram.set(m_mouse_diagrams[Action]);
		m_keyboard_diagram.set(m_keyboard_diagrams[Modifiers]);
	}

private:
	bool on_safe_to_close()
	{
		// If we're in batch mode, go for it!
		if(application_state().batch_mode())
			return true;

		if(m_run_state != RUN)
			return true;

		// Give the user a chance to quit ...
		std::vector<std::string> buttons;
		buttons.push_back(_("Yes"));
		buttons.push_back(_("No"));
		if(1 == query_message(_("Quit the tutorial?"), _("Tutorial:"), 2, buttons))
		{
			on_quit();
			return true;
		}

		return false;
	}

	void on_speed_changed()
	{
		const double speed = pow(10.0, m_speed.get_adjustment()->get_value());

		std::stringstream buffer;
		buffer << "x " << std::setprecision(2) << speed;
		m_speed_label.set_text(buffer.str());

		options::set_tutorial_speed(speed);
	}

	void on_cruise_control()
	{
		// If the user turned-off cruise control, we're done ...
		if(!m_cruise_control.value())
			return;

		// Give the user a chance to cancel ...
		std::vector<std::string> buttons;
		buttons.push_back(_("Yes"));
		buttons.push_back(_("No"));
		unsigned int result = query_message(_("Enable Cruise Control?  Once you hit \"Continue\" the tutorial will run automatically without stopping.  You may hit the ESC key at any time during the tutorial to cancel Cruise Control."), _("Tutorial:"), 1, buttons);
		if(1 != result)
			m_cruise_control.set_value(false);
	}

	void on_quit()
	{
		set_modal(false);
		hide();
		m_run_state = QUIT;
	}

	void on_continue()
	{
		set_modal(false);
		m_run_state = CONTINUE;
	}

	Gtk::Image m_keyboard_diagram;
	Gtk::Image m_mouse_diagram;

	Gtk::TextView m_message;
	Glib::RefPtr<Gtk::TextBuffer::Tag> m_message_tag_url;

	Gtk::HScrollbar m_speed;
	Gtk::Label m_speed_label;
	button::control m_quit;
	button::control m_continue;

	std::map<k3d::iuser_interface::mouse_action_t, Glib::RefPtr<Gdk::Pixbuf> > m_mouse_diagrams;
	std::map<k3d::key_modifiers, Glib::RefPtr<Gdk::Pixbuf> > m_keyboard_diagrams;

	typedef enum
	{
		IDLE,
		RUN,
		QUIT,
		CONTINUE
	} run_state_t;
	run_state_t m_run_state;

	k3d_data(bool, immutable_name, change_signal, no_undo, local_storage, no_constraint, no_property, no_serialization) m_cruise_control;
};

tutorial_message_window& tutorial_message_instance()
{
	static tutorial_message_window* instance = 0;
	if(!instance)
		instance = new tutorial_message_window();

	return *instance;
}

void initialize_tutorial_messages()
{
//	tutorial_message_instance();
}

bool tutorial_message(const std::string& Message)
{
	return tutorial_message_instance().tutorial_message(Message);
}

void tutorial_mouse_message(const std::string& Message, const k3d::iuser_interface::mouse_action_t Action, const k3d::key_modifiers Modifiers)
{
	tutorial_message_instance().tutorial_mouse_message(Message, Action, Modifiers);
}

} // namespace libk3dui

