// K-3D SDS preview
// Copyright (c) 2005, Bart Janssens
//
// Contact: bart.janssens@lid.kviv.be
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**	\file Implements K-3D interface for Catmull-Clark algorithm
		\author Bart Janssens <bart.janssens@lid.kviv.be>
 */

#include "k3d_sds_binding.h"

namespace k3d
{

// Comparison operator to sort k3d::vector3 in a map
inline bool operator < (const k3d::vector3& a, const k3d::vector3& b)
{
	if(a[0] < b[0])
		return true;
	if(a[0] > b[0])
		return false;
	if(a[1] < b[1])
		return true;
	if(a[1] > b[1])
		return false;
	if(a[2] < b[2])
		return true;

	return false;
}

namespace sds
{

///////////
// k3d_cache_input
///////////

bool k3d_cache_input::set_input(const k3d::mesh* Input)
{
	return_val_if_fail(m_input, false);
	if (m_npoints != Input->points.size() || m_npolyhedra != Input->polyhedra.size())
		return false;
	for (unsigned long i = 0; i < m_npolyhedra; ++i)
	{
		if (Input->polyhedra[i]->faces.size() != m_nfaces[i])
			return false;
	}
	m_input = Input;
	init_counters();
	return true;
}

k3d_cache_input::k3d_cache_input(const k3d::mesh* Input) :
		m_input(Input)
{
	init_counters();
	for(unsigned long p = 0; p < m_input->polyhedra.size(); ++p)
	{
		facevertices_t* faces = new facevertices_t();
		m_faces_per_polyhedron.push_back(faces);
		k3d::polyhedron& polyhedron = *(m_input->polyhedra[p]);
		for(unsigned long i = 0; i < polyhedron.faces.size(); ++i)
		{
			k3d::face* face = polyhedron.faces[i];
			face_vertex* fv = new face_vertex();
			m_face_vertices.push_back(fv);
			faces->push_back(fv);
			k3d::split_edge* first_edge = face->first_edge;
			k3d::split_edge* edge = first_edge;
			do {
				fv->edge_vertices.push_back(get_ev(edge, fv));
				edge = edge->face_clockwise;
			} while (edge != first_edge);
		}
	}
	m_n_segments = get_segment(m_face_vertices.size()-1) + 1;
	k3d::log() << debug << "generated " << m_face_vertices.size() << " face vertices, " << m_edge_vertices.size() << " edge vertices and " << m_sds_vertices.size() << " sds vertices." << std::endl;
}

void k3d_cache_input::update_input(bool all, facevertices_t& updated_maps)
{
	m_modified_segments.clear();
	unsigned long global_index = 0;
	for(unsigned long p = 0; p < m_input->polyhedra.size(); ++p)
	{
		facevertices_t& faces = *(m_faces_per_polyhedron[p]);
		k3d::polyhedron& polyhedron = *(m_input->polyhedra[p]);
		for(unsigned long i = 0; i < polyhedron.faces.size(); ++i)
		{
			k3d::face* face = polyhedron.faces[i];
			if(all || selected(face->first_edge))
			{
				positions_t points;
				k3d::split_edge* edge = face->first_edge;
				do
				{
					points.push_back(&edge->vertex->position);
					edge = edge->face_clockwise;
				} while (edge != face->first_edge);
				faces[i]->update(points);
				updated_maps.push_back(faces[i]);
				m_modified_segments.insert(get_segment(global_index));
			}
			++global_index;
		}
	}
	for(edgemap_t::iterator point = m_edge_vertices.begin(); point != m_edge_vertices.end(); ++point)
	{
		point->second->update();
	}
	for(pointmap_t::iterator point = m_sds_vertices.begin(); point != m_sds_vertices.end(); ++point)
	{
		point->second->update();
	}
}

k3d_cache_input::~k3d_cache_input()
{
	for(edgemap_t::iterator it = m_edge_vertices.begin(); it != m_edge_vertices.end(); ++it)
	{
		delete it->second;
	}
	for(pointmap_t::iterator it = m_sds_vertices.begin(); it != m_sds_vertices.end(); ++it)
	{
		delete it->second;
	}
	for(facevertices_t::iterator it = m_face_vertices.begin(); it != m_face_vertices.end(); ++it)
	{
		delete *it;
	}
	for(unsigned long i = 0; i < m_faces_per_polyhedron.size(); ++i)
		delete m_faces_per_polyhedron[i];
}

edge_vertex* k3d_cache_input::get_ev(k3d::split_edge* Edge, face_vertex* Fv)
{
	if(Edge->companion)
	{
		edgemap_t::iterator evit = m_edge_vertices.find(Edge->companion);
		if(evit != m_edge_vertices.end())
		{
			edge_vertex* ev = evit->second;
			ev->face_vertices[1] = Fv;
			sds_point* corner = get_corner(Edge->vertex);
			corner->face_vertices.push_back(Fv);
			ev->corners[1] = corner;
			ev->corners[0]->corners.push_back(corner);
			corner->corners.push_back(ev->corners[0]);
			Fv->corners.push_back(corner);
			return ev;
		}
	}

	edge_vertex* ev = new edge_vertex();
	if(!Edge->companion)
		ev->corners[1] = get_corner(Edge->face_clockwise->vertex);
	m_edge_vertices[Edge] = ev;
	sds_point* corner = get_corner(Edge->vertex);
	ev->corners[0] = corner;
	corner->face_vertices.push_back(Fv);
	ev->face_vertices[0] = Fv;
	Fv->corners.push_back(corner);
	return ev;
}

sds_point* k3d_cache_input::get_corner(k3d::point* Point)
{
	pointmap_t::iterator corner_it = m_sds_vertices.find(Point);
	if(corner_it != m_sds_vertices.end())
		return corner_it->second;
	sds_point* corner = new sds_point();
	m_sds_vertices[Point] = corner;
	return corner;
}

bool k3d_cache_input::selected(k3d::split_edge* first_edge, bool recurse)
{
	k3d::split_edge* edge = first_edge;
	do {
		//k3d::log() << debug << "selection_weights: " << edge->selection_weight << ", " << edge->vertex->selection_weight << std::endl;
		if(edge->selection_weight != 0.0 || edge->vertex->selection_weight != 0.0 || (recurse && edge->companion && selected(edge->companion, false)))
		//if(edge->visible_selection || edge->vertex->visible_selection || (recurse && edge->companion && selected(edge->companion, false)))
		{
			return true;
		}
		edge = edge->face_clockwise;
	} while (edge != first_edge);
	return false;
}

////////
// k3d_opengl_sds_cache
/////////

// helper function to draw a quadrilateral in opengl.
void draw_quad(position_t** points, position_t** normals)
{
	glBegin(GL_POLYGON);
	for(int k = 0; k < 4; ++k)
	{
		glNormal3dv(*normals[k]);
		glVertex3dv(*points[k]);
	}
	glEnd();
}

void k3d_opengl_sds_cache::client_output( k3d::mesh * Output )
{
	init_lists();
	segments_t& ModifiedSegments = dynamic_cast<k3d_cache_input*>(m_input)->get_modified_segments();
	//k3d::log() << debug << "outputting " << ModifiedSegments.size() << " segments" << std::endl;
	unsigned long Level = m_levels-1;
	for(segments_t::const_iterator segment_it = ModifiedSegments.begin(); segment_it != ModifiedSegments.end(); ++segment_it)
	{
		const unsigned long segment = *segment_it;
		const unsigned long start = segment * segment_size;
		const unsigned long end = (start + segment_size) > m_input->face_vertices().size() ? m_input->face_vertices().size() : (start + segment_size);
		glDeleteLists(m_display_lists[segment], 1);
		m_display_lists[segment] = glGenLists(1);
		glNewList(m_display_lists[segment],GL_COMPILE);

		gl_setup();

		if(Level == 0)
		{
			draw_faces_first_level(start, end);
			if(m_draw_borders)
			{
				draw_borders_first_level(start, end);
			}
		}
		else // higher levels
		{
			draw_faces_higher_level(start, end, Level);
			if(m_draw_borders)
			{
				draw_borders_higher_level(start, end, Level);
			}
		}
		glEndList();
	}
	for(unsigned long i = 0; i < dynamic_cast<k3d_cache_input*>(m_input)->get_n_segments(); ++i)
		glCallList(m_display_lists[i]);
	ModifiedSegments.clear();
}

void k3d_opengl_sds_cache::client_output_nurbs( k3d::mesh * Output )
{
	init_lists();
	segments_t& ModifiedSegments = dynamic_cast<k3d_cache_input*>(m_input)->get_modified_segments();
	unsigned long Level = m_levels-1;
	if(Level == 0)
		return;
	for(segments_t::const_iterator segment_it = ModifiedSegments.begin(); segment_it != ModifiedSegments.end(); ++segment_it)
	{
		const unsigned long segment = *segment_it;
		const unsigned long start = segment * segment_size;
		const unsigned long end = (start + segment_size) > m_input->face_vertices().size() ? m_input->face_vertices().size() : (start + segment_size);

		if(!m_nurbs_renderer)
		{
			m_nurbs_renderer = gluNewNurbsRenderer();
			gluNurbsProperty(m_nurbs_renderer, GLU_AUTO_LOAD_MATRIX, GL_FALSE);
			gluNurbsProperty(m_nurbs_renderer, GLU_CULLING, GL_TRUE);
		}
		
		glDeleteLists(m_display_lists[segment], 1);
		m_display_lists[segment] = glGenLists(1);
		glNewList(m_display_lists[segment],GL_COMPILE);

		gl_setup();
		
		GLfloat gl_modelview_matrix[16];
		glGetFloatv(GL_MODELVIEW_MATRIX, gl_modelview_matrix);
		return_if_fail(m_render_state);
		gluLoadSamplingMatrices(m_nurbs_renderer, gl_modelview_matrix, (*m_render_state).gl_projection_matrix, (*m_render_state).gl_viewport);
		glDisable(GL_CULL_FACE);
		gluNurbsProperty(m_nurbs_renderer, GLU_DISPLAY_MODE, GLU_FILL);

		draw_nurbs_patches(start, end, Level);
		
		if(m_draw_borders)
		{
			draw_nurbs_borders(start, end, Level);
		}
		glEndList();
	} // for all segments
	for(unsigned long i = 0; i < dynamic_cast<k3d_cache_input*>(m_input)->get_n_segments(); ++i)
		glCallList(m_display_lists[i]);
	ModifiedSegments.clear();
}

void k3d_opengl_sds_cache::gl_setup()
{
	glEnable(GL_LIGHTING);
	glColor3d(0.8, 0.8, 1);

	glFrontFace(GL_CW);
	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	glEnable(GL_CULL_FACE);

	glEnable(GL_POLYGON_OFFSET_FILL);
	glPolygonOffset(1.0, 1.0);

	GLfloat diffuse[4];
	diffuse[0] = 0.8f;
	diffuse[1] = 0.8f;
	diffuse[2] = 0.8f;
	diffuse[3] = 1.0f;
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
}

void k3d_opengl_sds_cache::draw_faces_first_level( unsigned long start, unsigned long end )
{
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		unsigned long size = f.edge_vertices.size();
		for(unsigned long i = 0; i < size; ++i)
		{
			position_t n = normal(f.vertex, f.edge_vertices[i]->vertex, f.corners[(i+1)%size]->vertex, f.edge_vertices[(i+1)%size]->vertex);
			position_t* points[4] = {&f.vertex, &f.edge_vertices[i]->vertex, &f.corners[(i+1)%size]->vertex, &f.edge_vertices[(i+1)%size]->vertex};
			position_t* normals[4] = {&n, &n, &n, &n};
			draw_quad(points, normals);
		}
	}
}

void k3d_opengl_sds_cache::draw_borders_first_level( unsigned long start, unsigned long end )
{
	glDisable(GL_LIGHTING);
	glColor3d(0.0, 0.0, 1.0);
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		unsigned long size = f.edge_vertices.size();
		for(unsigned long i = 0; i < size; ++i)
		{
			glBegin(GL_LINES);
			glVertex3dv(f.edge_vertices[i]->vertex);
			glVertex3dv(f.corners[(i+1)%size]->vertex);
			glVertex3dv(f.corners[(i+1)%size]->vertex);
			glVertex3dv(f.edge_vertices[(i+1)%size]->vertex);
			glEnd();
		}
	}
}

void k3d_opengl_sds_cache::draw_faces_higher_level( unsigned long start, unsigned long end, unsigned long Level )
{
	int size = static_cast<int>(pow(2.0, static_cast<double>(Level)))+1;
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		for(face_vertex::mipmaps_t::iterator mipmap = f.mipmaps.begin(); mipmap != f.mipmaps.end(); ++mipmap)
		{
			const point_array& points = (*mipmap)->points(Level);
			const point_array& normals = (*mipmap)->normals(Level);
			for(int i = 0; i < size-1; ++i)
			{
				for(int j = 0; j < size-1; ++j)
				{
					position_t* quad[4] = {points[i][j], points[i][j+1], points[i+1][j+1], points[i+1][j]};
					position_t* qnormals[4] = {normals[i][j], normals[i][j+1], normals[i+1][j+1], normals[i+1][j]};
					draw_quad(quad, qnormals);
				}
			}
		}
	}
}

void k3d_opengl_sds_cache::draw_borders_higher_level( unsigned long start, unsigned long end, unsigned long Level )
{
	int size = static_cast<int>(pow(2.0, static_cast<double>(Level)))+1;
	glDisable(GL_LIGHTING);
	glColor3d(0.0, 0.0, 1.0);
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		for(face_vertex::mipmaps_t::iterator mipmap = f.mipmaps.begin(); mipmap != f.mipmaps.end(); ++mipmap)
		{
			const point_array& points = (*mipmap)->points(Level);
			glBegin(GL_LINES);
			for(int i = 0; i < size-1; ++i)
			{
				glVertex3dv(*points[0][i]);
				glVertex3dv(*points[0][i+1]);
			}
			for(int i = 0; i < size-1; ++i)
			{
				glVertex3dv(*points[i][size-1]);
				glVertex3dv(*points[i+1][size-1]);
			}
			glEnd();
		}
	}
}

void k3d_opengl_sds_cache::draw_nurbs_patches( unsigned long start, unsigned long end, unsigned long Level )
{
	int size = static_cast<int>(pow(2.0, static_cast<double>(Level)))+1;
	int n_size = size + 4;
	int k_size = n_size + 4;
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		for(face_vertex::mipmaps_t::iterator mipmap = f.mipmaps.begin(); mipmap != f.mipmaps.end(); ++mipmap)
		{
			const point_array& nurbs = (*mipmap)->nurbs(Level);
			float* const knots = (*mipmap)->knots(Level);
			float points[n_size][n_size][3];
			for(int i = 0; i < n_size; ++i)
			{
				for(int j = 0; j < n_size; ++j)
				{
					points[i][j][0] = (*nurbs[i][j])[0];
					points[i][j][1] = (*nurbs[i][j])[1];
					points[i][j][2] = (*nurbs[i][j])[2];
				}
			}
			gluBeginSurface(m_nurbs_renderer);
			gluNurbsSurface(m_nurbs_renderer, k_size, knots, k_size, knots, 3*n_size, 3, &points[0][0][0], 4, 4, GL_MAP2_VERTEX_3);
			gluEndSurface(m_nurbs_renderer);
		}
	}
}

void k3d_opengl_sds_cache::draw_nurbs_borders( unsigned long start, unsigned long end, unsigned long Level )
{
	int size = static_cast<int>(pow(2.0, static_cast<double>(Level)))+1;
	int n_size = size + 4;
	int k_size = n_size + 4;
	glDisable(GL_LIGHTING);
	glColor3d(0.0, 0.0, 1.0);
	for(unsigned long face = start; face != end; ++face)
	{
		face_vertex& f = *m_input->face_vertices()[face];
		for(face_vertex::mipmaps_t::iterator mipmap = f.mipmaps.begin(); mipmap != f.mipmaps.end(); ++mipmap)
		{
			const point_array& nurbs = (*mipmap)->nurbs(Level);
			float* const knots = (*mipmap)->knots(Level);
			float points1[n_size][3];
			float points2[n_size][3];
			for(int i = 0; i < n_size; ++i)
			{
				points1[i][0] = (*nurbs[0][i])[0];
				points1[i][1] = (*nurbs[0][i])[1];
				points1[i][2] = (*nurbs[0][i])[2];
				points2[i][0] = (*nurbs[i][n_size-1])[0];
				points2[i][1] = (*nurbs[i][n_size-1])[1];
				points2[i][2] = (*nurbs[i][n_size-1])[2];
			}
			gluBeginCurve(m_nurbs_renderer);
			gluNurbsCurve(m_nurbs_renderer, k_size, knots, 3, &points1[0][0], 4, GL_MAP1_VERTEX_3);
			gluEndCurve(m_nurbs_renderer);
			gluBeginCurve(m_nurbs_renderer);
			gluNurbsCurve(m_nurbs_renderer, k_size, knots, 3, &points2[0][0], 4, GL_MAP1_VERTEX_3);
			gluEndCurve(m_nurbs_renderer);
		}
	}
}

typedef std::map<k3d::vector3, k3d::point*> vertex_map_t;

k3d::point* create_unique_point(vertex_map_t& VertexMap, const k3d::vector3& Position, k3d::mesh& Mesh)
{
	vertex_map_t::iterator i = VertexMap.find(Position);
	if(i != VertexMap.end())
		return i->second;

	// Create a new point
	k3d::point* new_point = new k3d::point(Position);
	return_val_if_fail(new_point, 0);
	Mesh.points.push_back(new_point);

	VertexMap.insert(std::make_pair(Position, new_point));

	return new_point;
}

void k3d_mesh_sds_cache::client_output(k3d::mesh* Output)
{
	k3d::mesh& Mesh = *Output;
	const unsigned long Level = m_levels;
	/** \todo link materials to original face materials. */
	k3d_cache_input* input = dynamic_cast<k3d_cache_input*>(m_input);
	return_if_fail(input);
	return_if_fail(!input->mesh().polyhedra.empty());
	return_if_fail(!input->mesh().polyhedra.front()->faces.empty());
	k3d::imaterial* Material = input->mesh().polyhedra.front()->faces.front()->material;

	const segments_t& ModifiedSegments = input->get_modified_segments();
	const facevertices_t& faces = input->face_vertices();

	k3d::polyhedron* const polyhedron = new k3d::polyhedron();
	Mesh.polyhedra.push_back(polyhedron);

	for(segments_t::const_iterator segment_it = ModifiedSegments.begin(); segment_it != ModifiedSegments.end(); ++segment_it)
	{
		const unsigned long segment = *segment_it;
		const unsigned long start = segment * segment_size;
		const unsigned long end = (start + segment_size) > faces.size() ? faces.size() : (start + segment_size);
		int size = static_cast<int>(pow(2.0, static_cast<double>(Level-1)))+1;

		vertex_map_t vertex_map;
		if(Level-1 == 0)
		{
			for(unsigned long face = start; face != end; ++face)
			{
				face_vertex& f = *faces[face];
				unsigned long size = f.edge_vertices.size();
				for(unsigned long i = 0; i < size; ++i)
				{
					const k3d::vector3 vertex1 = f.vertex;
					const k3d::vector3 vertex2 = f.edge_vertices[i]->vertex;
					const k3d::vector3 vertex3 = f.corners[(i+1)%size]->vertex;
					const k3d::vector3 vertex4 = f.edge_vertices[(i+1)%size]->vertex;
					k3d::point* point1 = create_unique_point(vertex_map, vertex1, Mesh);
					k3d::point* point2 = create_unique_point(vertex_map, vertex2, Mesh);
					k3d::point* point3 = create_unique_point(vertex_map, vertex3, Mesh);
					k3d::point* point4 = create_unique_point(vertex_map, vertex4, Mesh);
						//const k3d::vector3 normal = normal(f.vertex, f.edge_vertices[i]->vertex, f.corners[(i+1)%size]->vertex, f.edge_vertices[(i+1)%size]->vertex);

					k3d::split_edge* edge1 = new k3d::split_edge(point1);
					k3d::split_edge* edge2 = new k3d::split_edge(point2);
					k3d::split_edge* edge3 = new k3d::split_edge(point3);
					k3d::split_edge* edge4 = new k3d::split_edge(point4);

					edge1->face_clockwise = edge2;
					edge2->face_clockwise = edge3;
					edge3->face_clockwise = edge4;
					edge4->face_clockwise = edge1;

					k3d::face* new_face = new k3d::face(edge1, Material);
					assert_warning(new_face);

					polyhedron->faces.push_back(new_face);
				}
			}
		}

		for(unsigned long face = start; face != end; ++face)
		{
			face_vertex& f = *faces[face];
			for(face_vertex::mipmaps_t::iterator mipmap = f.mipmaps.begin(); mipmap != f.mipmaps.end(); ++mipmap)
			{
				const point_array& points = (*mipmap)->points(Level-1);
						//const point_array& normals = (*mipmap)->normals(Level-1);
				for(int i = 0; i < size-1; ++i)
				{
					for(int j = 0; j < size-1; ++j)
					{
						const k3d::vector3 vertex1 = *points[i][j];
						const k3d::vector3 vertex2 = *points[i][j+1];
						const k3d::vector3 vertex3 = *points[i+1][j+1];
						const k3d::vector3 vertex4 = *points[i+1][j];
						k3d::point* point1 = create_unique_point(vertex_map, vertex1, Mesh);
						k3d::point* point2 = create_unique_point(vertex_map, vertex2, Mesh);
						k3d::point* point3 = create_unique_point(vertex_map, vertex3, Mesh);
						k3d::point* point4 = create_unique_point(vertex_map, vertex4, Mesh);
								//const k3d::vector3 normal1 = *normals[i][j];
								//const k3d::vector3 normal2 = *normals[i][j+1];
								//const k3d::vector3 normal3 = *normals[i+1][j+1];
								//const k3d::vector3 normal4 = *normals[i+1][j];

						k3d::split_edge* edge1 = new k3d::split_edge(point1);
						k3d::split_edge* edge2 = new k3d::split_edge(point2);
						k3d::split_edge* edge3 = new k3d::split_edge(point3);
						k3d::split_edge* edge4 = new k3d::split_edge(point4);

						edge1->face_clockwise = edge2;
						edge2->face_clockwise = edge3;
						edge3->face_clockwise = edge4;
						edge4->face_clockwise = edge1;

						k3d::face* new_face = new k3d::face(edge1, Material);
						assert_warning(new_face);

						polyhedron->faces.push_back(new_face);
					}
				}
			}
		}
	}
	k3d::set_companions(*polyhedron);
}

} // namespace sds

} // namespace k3d

