// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/algebra.h>
#include <k3dsdk/bitmap_modifier.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/transformable.h>

#include "bitmap_element.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_instance

class bitmap_instance :
	public k3d::bitmap_modifier<k3d::persistent<k3d::node> >
{
	typedef k3d::bitmap_modifier<k3d::persistent<k3d::node> > base_t;
public:
	bitmap_instance(k3d::idocument& Document) :
		base_t(Document),
		m_position(init_owner(*this) + init_name("position") + init_label(_("Position")) + init_description(_("Position")) + init_value(k3d::point3(0.0, 0.0, 0.0))),
		m_scale(init_owner(*this) + init_name("scale") + init_label(_("Scale")) + init_description(_("Scale")) + init_value(k3d::point3(1.0, 1.0, 1.0))),
		m_angle(init_owner(*this) + init_name("angle") + init_label(_("Angle")) + init_description(_("Angle")) + init_value(k3d::radians(0.0)) + init_precision(2) + init_step_increment(k3d::radians(1.0)) + init_units(typeid(k3d::measurement::angle))),
		m_premultiplied(init_owner(*this) + init_name("premultiplied") + init_label(_("Premultiplied")) + init_description(_("Image is pre-multiplied")) + init_value(true)),
		m_color_matrix(init_owner(*this) + init_name("color_matrix") + init_label(_("Color Matrix")) + init_description(_("Color matrix")) + init_value(k3d::identity3D()))
	{
		m_position.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
		m_scale.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
		m_angle.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
		m_premultiplied.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
		m_color_matrix.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
		m_input.changed_signal().connect(sigc::mem_fun(*this, &bitmap_instance::on_reset_bitmap));
	}

	void on_reset_bitmap(k3d::iunknown* const Hint)
	{
		m_output.reset(0, Hint);
	}

	void on_create_bitmap(k3d::bitmap& Bitmap)
	{
		// Get the input bitmap ...
		k3d::bitmap* const input = m_input.value();
		if(!input)
			return;

		// Cache properties ...
		k3d::matrix4 color_matrix = m_color_matrix.value();
		bool premultiplied = m_premultiplied.value();
		double angle = m_angle.value();
		k3d::point3 scale = m_scale.value();
		k3d::point3 position = m_position.value();

		// Transform ...
		bitmap_element* element = new bitmap_element(input);
		element->color(color_matrix, premultiplied);
		element->scale(k3d::point2(scale[0], scale[1]));
		element->rotate(angle);
		element->position(k3d::point2(position[0], position[1]));
		element->true_translation();

		Bitmap = *element->get_bitmap();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_instance>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x9a247cdf, 0xebcf4861, 0xaa814c6f, 0xed947b51),
				"BitmapInstance",
				_("Gives bitmap a position"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::point3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_position;
	k3d_data(k3d::point3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_scale;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_angle;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_premultiplied;
	k3d_data(k3d::matrix4, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_color_matrix;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_instance_factory

k3d::iplugin_factory& bitmap_instance_factory()
{
	return bitmap_instance::get_factory();
}

} // namespace libk3dbitmap


