// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Material K-3D object, which controls the surface appearance of rendered geometry
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/vectors.h>

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// material

const std::string shadowtype_none = "None";
const std::string shadowtype_opaque = "Opaque";
const std::string shadowtype_opacity = "Opacity";
const std::string shadowtype_shaded = "Shaded";

class material :
	public k3d::persistent<k3d::node> ,
	public k3d::imaterial,
	public k3d::ri::imaterial
{
	typedef k3d::persistent<k3d::node>  base;

public:
	material(k3d::idocument& Document) :
		base(Document),
		m_surface_shader(init_owner(*this) + init_name("surface_shader") + init_label(_("Surface Shader")) + init_description(_("Surface shader")) + init_value<k3d::ri::isurface_shader*>(0)),
		m_displacement_shader(init_owner(*this) + init_name("displacement_shader") + init_label(_("Displacement Shader")) + init_description(_("Displacement shader")) + init_value<k3d::ri::idisplacement_shader*>(0)),
		m_atmosphere_shader(init_owner(*this) + init_name("atmosphere_shader") + init_label(_("Atmosphere Shader")) + init_description(_("Atmosphere shader")) + init_value<k3d::ri::ivolume_shader*>(0)),
		m_interior_shader(init_owner(*this) + init_name("interior_shader") + init_label(_("Interior Shader")) + init_description(_("Interior shader")) + init_value<k3d::ri::ivolume_shader*>(0)),
		m_exterior_shader(init_owner(*this) + init_name("exterior_shader") + init_label(_("Exterior Shader")) + init_description(_("Exterior shader")) + init_value<k3d::ri::ivolume_shader*>(0)),
		m_Matte(init_owner(*this) + init_name("matte") + init_label(_("Matte")) + init_description(_("Render geometry as a matte")) + init_value(false)),
		m_DisplacementBounds(init_owner(*this) + init_name("displacement_bounds") + init_label(_("Displacement Bounds")) + init_description(_("Displacement bounds")) + init_value(1.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_Color(init_owner(*this) + init_name("color") + init_label(_("Color")) + init_description(_("Color")) + init_value(k3d::color(1, 1, 1))),
		m_Opacity(init_owner(*this) + init_name("opacity") + init_label(_("Opacity")) + init_description(_("Opacity")) + init_value(k3d::color(1, 1, 1)))
	{
		deleted_signal().connect(sigc::mem_fun(*this, &material::on_deleted));

		m_RiTexture = 0;
	}

	void on_deleted()
	{
	}

	k3d::gl::imaterial* gl_material()
	{
		return 0;
	}

	k3d::ri::imaterial* ri_material()
	{
		return this;
	}

	k3d::yafray::imaterial* yafray_material()
	{
		return 0;
	}

	void setup_renderman_material(const k3d::ri::render_state& State)
	{
		// We only generate RIB on the final sample ...
		if(!k3d::ri::last_sample(State))
			return;

		// Setup displacement bounds ...
		k3d::ri::parameter_list displacement_attributes;
		displacement_attributes.push_back(k3d::ri::parameter("sphere", k3d::ri::UNIFORM, static_cast<k3d::ri::real>(m_DisplacementBounds.value())));
		displacement_attributes.push_back(k3d::ri::parameter("coordinatesystem", k3d::ri::UNIFORM, k3d::ri::string("world")));
		State.engine.RiAttributeV("displacementbound", displacement_attributes);

		// Set base color
		State.engine.RiColor(m_Color.value());

		// Set opacity
		State.engine.RiOpacity(m_Opacity.value());

		// Set the matte attribute
		State.engine.RiMatte(m_Matte.value() ? 1 : 0);

		// Setup shaders ...
		if(m_surface_shader.value())
			m_surface_shader.value()->setup_renderman_surface_shader(State);
		if(m_displacement_shader.value())
			m_displacement_shader.value()->setup_renderman_displacement_shader(State);
		if(m_atmosphere_shader.value())
			m_atmosphere_shader.value()->setup_renderman_atmosphere_shader(State);
		if(m_interior_shader.value())
			m_interior_shader.value()->setup_renderman_interior_shader(State);
		if(m_exterior_shader.value())
			m_exterior_shader.value()->setup_renderman_exterior_shader(State);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<material>,
				k3d::interface_list<k3d::imaterial,
				k3d::interface_list<k3d::ri::imaterial> > > factory(
			k3d::classes::RenderManMaterial(),
			"RenderManMaterial",
			_("A RenderMan surface / displacement material"),
			"RenderMan Materials",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::ri::isurface_shader*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) m_surface_shader;
	k3d_data(k3d::ri::idisplacement_shader*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) m_displacement_shader;
	k3d_data(k3d::ri::ivolume_shader*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) m_atmosphere_shader;
	k3d_data(k3d::ri::ivolume_shader*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) m_interior_shader;
	k3d_data(k3d::ri::ivolume_shader*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) m_exterior_shader;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_Matte;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_DisplacementBounds;

	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_Color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_Opacity;

	k3d::ri::itexture* m_RiTexture;
};

/////////////////////////////////////////////////////////////////////////////
// material_factory

k3d::iplugin_factory& material_factory()
{
	return material::get_factory();
}

} // namespace libk3drenderman


