// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Material K-3D object, which controls the surface appearance of rendered geometry
		\author Tim Shead (tshead@k-3d.com)
*/

#include "utility.h"

#include <k3dsdk/classes.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/igl.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/node_change_signal.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/module.h>
#include <k3dsdk/vectors.h>

#include <sh/sh.hpp>
#include <shutil/shutil.hpp>
using namespace SH;
using namespace ShUtil;

namespace libk3dsh
{

/////////////////////////////////////////////////////////////////////////////
// gooch

class gooch :
	public k3d::node_change_signal<k3d::persistent<k3d::node> >,
	public k3d::imaterial,
	public k3d::gl::imaterial
{
	typedef k3d::node_change_signal<k3d::persistent<k3d::node> > base;

public:
	gooch(k3d::idocument& Document) :
		base(Document),
		m_diffuse_color(init_owner(*this) + init_name("diffuse_color") + init_label(_("Diffuse Color")) + init_description(_("Diffuse Color")) + init_value(k3d::color(1, 1, 1))),
		m_warm_color(init_owner(*this) + init_name("warm_color") + init_label(_("Warm Color")) + init_description(_("Warm Color")) + init_value(k3d::color(1, 1, 0.8))),
		m_cool_color(init_owner(*this) + init_name("cool_color") + init_label(_("Cool Color")) + init_description(_("Cool Color")) + init_value(k3d::color(0.8, 0.8, 1)))
//		m_light_vector(init_owner(*this) + init_name("light_vector") + init_label(_("Light Vector")) + init_description(_("Light Vector")) + init_value(k3d::vector3(0, 0, 1)))
	{
		m_diffuse_color_cache = convert(m_diffuse_color.value());
		m_warm_color_cache = convert(m_warm_color.value());
		m_cool_color_cache = convert(m_cool_color.value());
//		m_light_vector_cache = convert(m_light_vector.value());
		m_light_vector_cache = ShVector3f(1, 1, 1);

		m_diffuse_color.changed_signal().connect(sigc::mem_fun(*this, &gooch::on_diffuse_color_changed));
		m_warm_color.changed_signal().connect(sigc::mem_fun(*this, &gooch::on_warm_color_changed));
		m_cool_color.changed_signal().connect(sigc::mem_fun(*this, &gooch::on_cool_color_changed));
//		m_light_vector.changed_signal().connect(sigc::mem_fun(*this, &gooch::on_light_vector_changed));

		try
		{
			// Sh data
			ShMatrix4x4f mv;
			ShMatrix4x4f mvp;
			ShPoint3f lightp;
			ShColor3f diffuse;
			ShColor3f ambient;

			// setup OpenGL bindings
			mv.meta("opengl:state", "state.matrix.modelview");
			mv.meta("opengl:readonly", "true");
			mvp.meta("opengl:state", "state.matrix.mvp");
			mvp.meta("opengl:readonly", "true");
			lightp.meta("opengl:state", "state.light[0].position");
			lightp.meta("opengl:readonly", "true");
			diffuse.meta("opengl:state", "state.gooch.diffuse");
			diffuse.meta("opengl:readonly", "true");
			ambient.meta("opengl:state", "state.gooch.ambient");
			ambient.meta("opengl:readonly", "true");

			// construct vertex program
			vsh = SH_BEGIN_VERTEX_PROGRAM
			{
				ShInputPosition4f ipos;
				ShInputNormal3f inrm;

				ShOutputPosition4f opos;
				ShOutputPosition4f objectpos;
				ShOutputNormal3f onrm;
				ShOutputVector3f olightv;

				// transform position and normal and
				// generate the light vector
				opos = mvp|ipos;
				objectpos = ipos;
				onrm = mv|inrm;
				olightv = lightp - (mv|ipos)(0, 1, 2);
			} SH_END;

			fsh = ShKernelSurface::gooch<ShColor3f>()
				<< m_diffuse_color_cache
				<< m_cool_color_cache
				<< m_warm_color_cache
				<< m_light_vector_cache;
			
			k3d::log() << debug << "Vertex Unit:" << std::endl;
			vsh.node()->code()->print(k3d::log());
			k3d::log() << debug << "Fragment Unit:" << std::endl;
			fsh.node()->code()->print(k3d::log());
		}
		catch(std::exception& e)
		{
			k3d::log() << error << "GoochMaterial exception: " << e.what() << std::endl;
		}
		catch(...)
		{
			k3d::log() << error << "Unknown GoochMaterial exception" << std::endl;
		}
	}

	void on_diffuse_color_changed(k3d::iunknown* Hint)
	{
		m_diffuse_color_cache = convert(m_diffuse_color.value());
		make_node_change_slot()(Hint);
	}

	void on_warm_color_changed(k3d::iunknown* Hint)
	{
		m_warm_color_cache = convert(m_warm_color.value());
		make_node_change_slot()(Hint);
	}

	void on_cool_color_changed(k3d::iunknown* Hint)
	{
		m_cool_color_cache = convert(m_cool_color.value());
		make_node_change_slot()(Hint);
	}

/*
	void on_light_vector_changed(k3d::iunknown* Hint)
	{
		m_light_vector_cache = convert(m_light_vector.value());
		make_node_change_slot()(Hint);
	}
*/

	k3d::gl::imaterial* gl_material()
	{
		return this;
	}

	k3d::ri::imaterial* ri_material()
	{
		return 0;
	}

	k3d::yafray::imaterial* yafray_material()
	{
		return 0;
	}

	void setup_gl_material()
	{
		try
		{
			shBind(vsh);
			shBind(fsh);
		}
		catch(std::exception& e)
		{
			k3d::log() << error << "GoochMaterial exception: " << e.what() << std::endl;
		}
		catch(...)
		{
			k3d::log() << error << "Unknown GoochMaterial exception" << std::endl;
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<gooch>,
				k3d::interface_list<k3d::imaterial,
				k3d::interface_list<k3d::gl::imaterial> > > factory(
			k3d::uuid(0x1ca37321, 0x55684731, 0x9b170005, 0x2682e232),
			"GoochMaterial",
			_("Technical illustration shader"),
			"OpenGL Materials",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_diffuse_color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_warm_color;
	k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_cool_color;
//	k3d_data(k3d::vector3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_light_vector;

	ShColor3f m_diffuse_color_cache;
	ShColor3f m_warm_color_cache;
	ShColor3f m_cool_color_cache;
	ShVector3f m_light_vector_cache;
	
	ShProgram vsh;
	ShProgram fsh;
};

/////////////////////////////////////////////////////////////////////////////
// gooch_factory

k3d::iplugin_factory& gooch_factory()
{
	return gooch::get_factory();
}

} // namespace libk3dsh

