/***************************************************************************
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <aspell.h>

#include <qcolor.h>
#include <qlistbox.h>
#include <qtimer.h>

#include "chat_manager.h"
#include "chat_widget.h"
#include "custom_input.h"
#include "kadu.h"
#include "message_box.h"
#include "misc.h"
#include "modules.h"

#include "spellchecker.h"

#define MODULE_SPELLCHECKER_VERSION 0.21

// HTML mark for mispelled words
const char *endMark = "</span>";
SpellChecker* spellcheck;

extern "C" int spellchecker_init()
{
	spellcheck = new SpellChecker();

	// use configuration settings to create spellcheckers for languages
	if (!spellcheck->buildCheckers())
	{
		delete spellcheck;
		return 1;
	}
	else
	{
		MainConfigurationWindow::registerUiFile(dataPath("kadu/modules/configuration/spellchecker.ui"), spellcheck);
		return 0;
	}
}

extern "C" void spellchecker_close()
{
	if (spellcheck)
	{
		MainConfigurationWindow::unregisterUiFile(dataPath("kadu/modules/configuration/spellchecker.ui"), spellcheck);
		delete spellcheck;
	}
}

SpellChecker::SpellChecker()
{
	connect(chat_manager, SIGNAL(chatWidgetCreated(ChatWidget *)), this, SLOT(chatCreated(ChatWidget *)));

	// this timer will wake up spell checker to analyze chat windows'
	// input and mark spelling mistakes
	myWakeupTimer = new QTimer(this);
	connect(myWakeupTimer, SIGNAL(timeout()), this, SLOT(executeChecking()));

	// prepare configuration of spellchecker
	spellConfig = new_aspell_config();
	aspell_config_replace(spellConfig, "encoding", "utf-8");

	import_0_5_0_Configuration();

	createDefaultConfiguration();
	// load mark settings
	buildMarkTag();
}

SpellChecker::~SpellChecker()
{
	disconnect(chat_manager, SIGNAL(chatWidgetCreated(ChatWidget *)), this, SLOT(chatCreated(ChatWidget *)));

	// Disabling spellcheker
	myWakeupTimer->stop();
	disconnect(myWakeupTimer, SIGNAL(timeout()), this, SLOT(executeChecking()));

	FOREACH(it, chat_manager->chats())
		cleanMessage(*it);

	delete_aspell_config(spellConfig);

	delete myWakeupTimer;

	FOREACH(it, checkers)
		delete_aspell_speller(it.data());
}

QStringList SpellChecker::notCheckedLanguages()
{
	QStringList result;
	AspellDictInfoList* dlist;
	AspellDictInfoEnumeration* dels;
	const AspellDictInfo* entry;

	/* the returned pointer should _not_ need to be deleted */
	dlist = get_aspell_dict_info_list(spellConfig);

	dels = aspell_dict_info_list_elements(dlist);
	while ((entry = aspell_dict_info_enumeration_next(dels)) != 0)
	{
		if (checkers.find(entry->name) == checkers.end())
			result.push_back(entry->name);
	}
	delete_aspell_dict_info_enumeration(dels);

	return result;
}

QStringList SpellChecker::checkedLanguages()
{
	QStringList result;
	for (Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++)
		result.append(it.key());

	return result;
}

bool SpellChecker::addCheckedLang(QString &name)
{
	if (checkers.find(name) != checkers.end())
		return true;

	aspell_config_replace(spellConfig, "lang", name.ascii());

	// create spell checker using prepared configuration
	AspellCanHaveError* possibleErr = new_aspell_speller(spellConfig);
	if (aspell_error_number(possibleErr) != 0)
	{
		MessageBox::msg(aspell_error_message(possibleErr));
		return false;
	}
	else
		checkers[name] = to_aspell_speller(possibleErr);

	if (checkers.size() == 1)
	{
		FOREACH(it, chat_manager->chats())
			chatCreated(*it);
	}

	return true;
}

void SpellChecker::removeCheckedLang(QString& name)
{
	Checkers::Iterator checker = checkers.find(name);
	if (checker != checkers.end())
	{
		delete_aspell_speller(checker.data());
		checkers.erase(name);
	}
}

bool SpellChecker::buildCheckers()
{
	FOREACH(it, checkers)
		delete_aspell_speller(it.data());

	checkers.clear();

	// load languages to check from configuration
	QString checkedStr = config_file.readEntry("ASpell", "Checked", "pl");
	QStringList checkedList = QStringList::split(',', checkedStr);

	if (config_file.readBoolEntry("ASpell", "Accents", false))
		aspell_config_replace(spellConfig, "ignore-accents", "true");
	else
		aspell_config_replace(spellConfig, "ignore-accents", "false");

	if (config_file.readBoolEntry( "ASpell", "Case", false))
		aspell_config_replace(spellConfig, "ignore-case", "true");
	else
		aspell_config_replace(spellConfig, "ignore-case", "false");

	// create aspell checkers for each language
	for (unsigned int i = 0; i < checkedList.count(); i++)
	{
		addCheckedLang(checkedList[i]);
		/*
		if ( !addCheckedLang( checkedList[i] ) )
		{
			delete_aspell_config( spellConfig );
			delete config;
			return false;
		}
		*/
	}
	return true;
}

void SpellChecker::buildMarkTag()
{
	FOREACH(it, chat_manager->chats())
		cleanMessage(*it);

	beginMark = "<span style=\"";

	if (config_file.readBoolEntry( "ASpell", "Bold", false))
		beginMark += "font-weight:600;";
	if (config_file.readBoolEntry( "ASpell", "Italic", false))
		beginMark += "font-style:italic;";
	if (config_file.readBoolEntry( "ASpell", "Underline", false))
		beginMark += "text-decoration:underline;";
	QColor colorMark("#FF0101");
	colorMark = config_file.readColorEntry("ASpell", "Color", &colorMark);
	beginMark += "color:" + colorMark.name() + "\">";
}

void SpellChecker::chatCreated(ChatWidget *chat)
{
	if (checkers.size() > 0)
	{
		if (!myWakeupTimer->isActive())
			myWakeupTimer->start(200);

		connect(chat, SIGNAL(messageSendRequested(ChatWidget*)), this, SLOT(cleanMessage(ChatWidget*)));
	}
}

void SpellChecker::cleanMessage(ChatWidget *chat)
{
	HtmlDocument parsedHtml;
	parsedHtml.parseHtml(chat->edit()->text());
	bool change = false;

	for (int i = 0; i < parsedHtml.countElements(); i++)
	{
		if (isTagMyOwn(parsedHtml, i))
		{
			parsedHtml.setElementValue(i, "");
			i++;
			parsedHtml.setElementValue(i + 1, "");
			i++;
			change = true;
		}
	}

	// if we have changed contents of chat window, than update it
	if (change)
		updateChat(chat->edit(), parsedHtml.generateHtml());
}

void SpellChecker::executeChecking()
{
	if (chat_manager->chats().size() == 0 || checkers.size() == 0)
		myWakeupTimer->stop();

	// iterate through open chats and check their inputs
	FOREACH(it, chat_manager->chats())
	{
		HtmlDocument parsedHtml;
		parsedHtml.parseHtml((*it)->edit()->text());
		bool change = false;

		for (int i = 0; i < parsedHtml.countElements(); i++)
		{
			if (parsedHtml.isTagElement(i))
				continue;

			QString text = parsedHtml.elementText(i);
			bool inWhite = true;
			int lastBegin = -1, lastEnd = -1;

			for (unsigned int j = 0; j < text.length(); j++)
			{
				if (inWhite)
				{
					if (text[j].isLetter())
					{
						inWhite = false;
						lastBegin = j;
					}
				}
				else
				{
					// if we are at the end of current word
					if (!text[j].isLetter() || j == text.length() - 1)
					{
						inWhite = true;
						if (text[j].isLetter() && j == text.length() - 1)
						{
							lastEnd = j + 1;

							if (i + 1 < parsedHtml.countElements() && isTagMyOwn(parsedHtml, i + 1))
							{
								parsedHtml.splitElement(i, lastBegin, lastEnd - lastBegin);
								parsedHtml.setElementValue(i + 2, parsedHtml.elementText(i) +
								parsedHtml.elementText(i+2), false);
								parsedHtml.setElementValue(i, "");

								continue;
							}
						}
						else
							lastEnd = j;

						QString word = text.mid(lastBegin, lastEnd - lastBegin);
						QCString wordUtf8 = word.utf8();

						// run checkers for all languages to check if this word is
						// valud in some of them
						bool isWordValid = checkers.size() == 0;
						for (Checkers::Iterator it = checkers.begin(); it != checkers.end(); it++)
						{
							if (aspell_speller_check(it.data(), wordUtf8, -1))
							{
								isWordValid = true;
								break;
							}
						}

						if (!isWordValid)
						{
							// current word isn't in dictionary, so take a look at it
							parsedHtml.splitElement(i, lastBegin, lastEnd - lastBegin);

							// check if this word isn't already marked as misspelled
							if ((i == 0 || !isTagMyOwn(parsedHtml, i - 1)) &&
									i < parsedHtml.countElements() - 1 &&
									!parsedHtml.isTagElement(i + 1))
							{
								parsedHtml.insertTag(i, beginMark);
								parsedHtml.insertTag(i + 2, endMark);
								change = true;
							}
							else if (i > 0 && i < parsedHtml.countElements())
							{
								// word is currently marked, but we must check if we don't
								// have some extra charactes inserted between this word
								// and the endmark
								if (!parsedHtml.isTagElement(i + 1))
								{
									parsedHtml.setElementValue(i + 2, parsedHtml.elementText(i + 1),
																							false);
									parsedHtml.setElementValue(i + 1, endMark, true);
									change = true;
								}
							}

							break;
						}
						else
						{
							// current word is correct, so remove possible tags
							if (i > 0 && isTagMyOwn(parsedHtml, i - 1) &&
									 i < parsedHtml.countElements() - 1 &&
									 parsedHtml.isTagElement(i + 1))
							{
								// we bring word back to not marked
								parsedHtml.setElementValue(i - 1, "");
								parsedHtml.setElementValue(i + 1, "");

								// trigger chat update
								change = true;
							}
						}
					}
				}
			}
		}
		// if we have changed contents of chat window, than update it
		if (change)
			updateChat((*it)->edit(), parsedHtml.generateHtml());
	}
}

void SpellChecker::updateChat(CustomInput *edit, QString text)
{
	int currentY, currentX;
	edit->getCursorPosition(&currentY, &currentX);
	edit->setUpdatesEnabled(false);
	edit->setText(text);
	// set cursor to initial position
	edit->setCursorPosition(currentY, currentX);
	edit->setUpdatesEnabled(true);
}

// check some part of element content to distiguish it from
// tags generated by external code
bool SpellChecker::isTagMyOwn(HtmlDocument &doc, int idx)
{
	unsigned int len = beginMark.length();
	if (doc.isTagElement(idx))
	{
		QString text = doc.elementText(idx);
		return text.length() == len && text[len - 3] == beginMark[len - 3] &&
			text[len - 5] == beginMark[len - 5] &&
			text[len - 7] == beginMark[len - 7];
	}
	else
		return false;
}

void SpellChecker::configForward()
{
	QListBoxItem* it = availList->selectedItem();
	if (it)
		configForward2(it);
}

void SpellChecker::configBackward()
{
	QListBoxItem* it = checkList->selectedItem();
	if (it)
		configBackward2(it);
}

void SpellChecker::configForward2(QListBoxItem *it)
{
	QString langName = it->text();
	if (addCheckedLang(langName))
	{
		checkList->insertItem(langName);
		availList->removeItem(availList->currentItem());
	}
}

void SpellChecker::configBackward2(QListBoxItem *it)
{
	QString langName = it->text();
	availList->insertItem(langName);
	checkList->removeItem(checkList->currentItem());
	removeCheckedLang(langName);
}

void SpellChecker::mainConfigurationWindowCreated(MainConfigurationWindow *mainConfigurationWindow)
{
	connect(mainConfigurationWindow, SIGNAL(configurationWindowApplied()), this, SLOT(configurationWindowApplied()));

	ConfigGroupBox *optionsGroupBox = mainConfigurationWindow->configGroupBox("Chat", "SpellChecker", tr("ASpell options"));

	QWidget *options = new QWidget(optionsGroupBox->widget());
	QGridLayout *optionsLayout = new QGridLayout(options);
	optionsLayout->setSpacing(5);
	optionsLayout->setMargin(5);

	availList = new QListBox(options);
	QPushButton *moveToCheckList = new QPushButton(tr("Move to 'Checked'"), options);

	optionsLayout->addWidget(new QLabel(tr("Available languages"), options), 0, 0);
	optionsLayout->addWidget(availList, 1, 0);
	optionsLayout->addWidget(moveToCheckList, 2, 0);

	checkList = new QListBox(options);
	QPushButton *moveToAvailList = new QPushButton(tr("Move to 'Available languages'"), options);

	optionsLayout->addWidget(new QLabel(tr("Checked"), options), 0, 1);
	optionsLayout->addWidget(checkList, 1, 1);
	optionsLayout->addWidget(moveToAvailList, 2, 1);

	connect(moveToCheckList, SIGNAL(clicked()), this, SLOT(configForward()));
	connect(moveToAvailList, SIGNAL(clicked()), this, SLOT(configBackward()));
	connect(checkList, SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configBackward2(QListBoxItem *)));
	connect(availList, SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(configForward2(QListBoxItem*)));

	optionsGroupBox->addWidgets(0, options);

	availList->setSelectionMode(QListBox::Single);
	checkList->setSelectionMode(QListBox::Single);
	availList->insertStringList(notCheckedLanguages());
	checkList->insertStringList(checkedLanguages());
}

void SpellChecker::configurationUpdated()
{
	buildMarkTag();
}

void SpellChecker::configurationWindowApplied()
{
	config_file.writeEntry("ASpell", "Checked", checkedLanguages().join(","));
}

void SpellChecker::createDefaultConfiguration()
{
	config_file.addVariable("ASpell", "Bold", "false");
	config_file.addVariable("ASpell", "Italic", "false");
	config_file.addVariable("ASpell", "Underline", "false");
	config_file.addVariable("ASpell", "Color", "#FF0101");
	config_file.addVariable("ASpell", "Checked", "pl");
	config_file.addVariable("ASpell", "Accents", "false");
	config_file.addVariable("ASpell", "Case", "false");
}

void SpellChecker::import_0_5_0_Configuration()
{
	ConfigFile *oldConfig = new ConfigFile(ggPath("spellchecker.conf"));
	config_file.addVariable("ASpell", "Checked", oldConfig->readEntry("ASpell", "Checked"));
	config_file.addVariable("ASpell", "Color", oldConfig->readEntry("ASpell", "Color"));
	config_file.addVariable("ASpell", "Bold", oldConfig->readEntry("ASpell", "Bold"));
	config_file.addVariable("ASpell", "Italic", oldConfig->readEntry("ASpell", "Italic"));
	config_file.addVariable("ASpell", "Underline", oldConfig->readEntry("ASpell", "Underline"));
	config_file.addVariable("ASpell", "Accents", oldConfig->readEntry("ASpell", "Accents"));
	config_file.addVariable("ASpell", "Case", oldConfig->readEntry("ASpell", "Case"));
	delete oldConfig;
}
