/***************************************************************************
                           kaffeine_part.cpp - kpart for embedded konqueror preview
                             -------------------
    begin                : Don Apr 17 11:30:44 CEST 2003
    revision             : $Revision: 1.36 $
    last modified        : $Date: 2004/05/01 08:05:45 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by J. Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <kiconloader.h>
#include <klocale.h>
#include <kaction.h>
#include <kglobalsettings.h>
#include <kdebug.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kparts/genericfactory.h>

#include <qtimer.h>
#include <qgrid.h>
#include <qtooltip.h>

#include "kaffeine_part.h"
#include "kaffeine_part.moc"


typedef KParts::GenericFactory<KaffeinePart> KaffeinePartFactory;
K_EXPORT_COMPONENT_FACTORY (libkaffeinepart, KaffeinePartFactory);



KaffeinePart::KaffeinePart(QWidget* parentWidget, const char* widgetName, QObject* parent, const char* name, const QStringList& args)
: KParts::ReadOnlyPart(parent, name ? name : "KaffeinePart"), videoWin(NULL), infoRow(NULL), playList(NULL), stopped(true), autoStart(true), imageWindow(true)
{
  kdDebug(555) << "KaffeinePart: Creating new..." << endl;
  
/*   Parsing parameters of <embed>
 *   Fix for Quicktime movies on www.apple.com:
 *   use the given href parameter instead of the URL passed by konqueror
 */

  hrefParam = QString::null;
  for (uint i=0; i<args.count(); i++)
  {
    kdDebug(555) << "Argument: " << args[i] << endl;
    if (args[i].left(9).lower()  == "autostart")
    {
      if (args[i].section( '"', 1, 1 ).lower() == "false")
      {
        kdDebug(555) << "Found parameter autoStart=false, disable autostart" << endl;
        autoStart = false;
      }
    }
    if (args[i].left(4).lower() == "href")
    {
      hrefParam = args[i].section( '"',1, 1 );
      kdDebug(555) << "Found href parameter, value: " << hrefParam << endl;
    }
    if (args[i].left(8).lower() == "controls")
    {
      if (args[i].section( '"', 1, 1 ).lower() != "imagewindow")
      {
        kdDebug(555) << "Not an ImageWindow object" << endl;
        imageWindow = false;
      }  
    }  
  }

  KInstance* instance = KaffeinePartFactory::instance();
  setInstance(instance);

  QWidget* mainFrame = new QWidget(parentWidget, widgetName);
  mainFrame->setFocusPolicy(QWidget::ClickFocus);

  if (imageWindow)
  {   
    QVBoxLayout* boxLayout = new QVBoxLayout(mainFrame);
    QWidget* panel = new QWidget(mainFrame, "panel");
    panel->setMaximumHeight(32);
    QHBoxLayout* panelLayout = new QHBoxLayout(panel);
    panelLayout->setSpacing(5);
    panelLayout->setMargin(2);
  
    videoWin = new VideoWindow(mainFrame, "videowin", QString::null, QString::null, false, true);
    connect(videoWin, SIGNAL(signalPlaybackFinished()), this, SLOT(slotPlaybackFinished()));
    connect(videoWin, SIGNAL(signalNewTitle(const QString&)), this, SLOT(slotNewTitle(const QString&)));
    connect(videoWin, SIGNAL(signalMetaInfo(const QString&)), this, SLOT(slotNewTitle(const QString&)));
    connect(videoWin, SIGNAL(signalNewXineMessage()), this, SLOT(slotNewXineMessage()));
  
    infoRow = new TextRow(mainFrame, "inforow");
    connect(videoWin, SIGNAL(signalNewInfo(const QString&)), infoRow, SLOT(slotChangeText(const QString&)));

    KPushButton* previousButton = new KPushButton(panel);
    previousButton->setPixmap(KGlobal::iconLoader()->loadIcon("player_start", KIcon::Small));
    connect(previousButton, SIGNAL(clicked()), this, SLOT(slotPrevious()));

    KPushButton* playButton = new KPushButton(panel);
    playButton->setPixmap(KGlobal::iconLoader()->loadIcon("player_play", KIcon::Small));
    connect(playButton, SIGNAL(clicked()), this, SLOT(slotPlay()));
  
    KPushButton* stopButton = new KPushButton(panel);
    stopButton->setPixmap(KGlobal::iconLoader()->loadIcon("player_stop", KIcon::Small));
    connect(stopButton, SIGNAL(clicked()), this, SLOT(slotStop()));
  
    KPushButton* nextButton = new KPushButton(panel);
    nextButton->setPixmap(KGlobal::iconLoader()->loadIcon("player_end", KIcon::Small));
    connect(nextButton, SIGNAL(clicked()), this, SLOT(slotNext()));

    posSlider = new MySlider(Qt::Horizontal, this, panel);
    posSlider->setRange(0,65535);
    posSlider->setSteps(100, 1000);
    connect(posSlider, SIGNAL(sliderMoved(int)), this, SIGNAL(signalChangePosition(int)));
    connect(this, SIGNAL(signalChangePosition(int)), videoWin, SLOT(slotChangePosition(int)));
  
    timeButton = new KPushButton(panel);
    QFontMetrics met( KGlobalSettings::generalFont() );
    timeButton->setFixedWidth( met.width("5:55:55")+18 );
    timeButton->setSizePolicy ( QSizePolicy (QSizePolicy::Fixed, QSizePolicy::Fixed) );
    connect(timeButton, SIGNAL(clicked()), videoWin, SLOT(slotTogglePlayMode()));
    connect(videoWin, SIGNAL(signalNewPosition(int, const QString&)), this, SLOT(slotSetPosition(int, const QString&)));

    KPushButton* logoButton = new KPushButton(panel);
    logoButton->setPixmap(KGlobal::iconLoader()->loadIcon("kaffeine", KIcon::Small));
    connect(logoButton, SIGNAL(clicked()), this, SLOT(slotStartExternal()));

    QToolTip::add(timeButton,i18n("Change Counter-Mode"));
    QToolTip::add(nextButton, i18n("Next Entry/Chapter"));
    QToolTip::add(previousButton, i18n("Previous Entry/Chapter"));
    QToolTip::add(stopButton, i18n("Stop"));
    QToolTip::add(logoButton, i18n("Play in Kaffeine externally"));
  
    panelLayout->addWidget(playButton);
    panelLayout->addWidget(stopButton);
    panelLayout->addWidget(previousButton);
    panelLayout->addWidget(nextButton);
    panelLayout->addWidget(posSlider);
    panelLayout->addWidget(timeButton);
    panelLayout->addWidget(logoButton);
  
    boxLayout->addWidget(videoWin);
    boxLayout->addWidget(infoRow);
    boxLayout->addWidget(panel);


    /*** load configuration ***/

    KConfig* config = instance->config();
    config->setGroup("General Options");

    metaString = config->readEntry("Meta String", QString("artist - title (album)"));
    videoWin->slotSetScreensaverTimeout( 0 );

    QFont f = KGlobalSettings::generalFont();
    QFont infoFont = config->readFontEntry("Textrow Font", &f);
    QColor c = KGlobalSettings::baseColor();
    QColor infoColor = config->readColorEntry("Textrow Color", &c);
    infoRow->slotSetConfig( &infoFont, &infoColor );

    config->setGroup("Visualization");
    visualPlugin = config->readEntry("Visual Plugin", "goom");

    /***/

    oldPosition = videoWin->mapToGlobal( QPoint(0,0) );
    connect( &posCheckTimer, SIGNAL( timeout() ), this, SLOT( slotCheckMoved() ) );
    posCheckTimer.start( 333 );
  }
  else
  {
    mainFrame->setPaletteBackgroundColor(QColor(0,0,0));
  }  

  mainFrame->show();
  setWidget(mainFrame);
}


KaffeinePart::~KaffeinePart()
{
 kdDebug(555) << "KaffeinePart destructor..." << endl;
 posCheckTimer.stop();
 delete playList;
 playList = NULL;
}


KAboutData *KaffeinePart::createAboutData()
{
    KAboutData* aboutData = new KAboutData( "kaffeine", "Kaffeine",
    VERSION, I18N_NOOP("A xine based media player. Take a look at Kaffeine Handbook for more information."),
    KAboutData::License_GPL,
    "(c) 2003-2004, Jürgen Kofler.", 0, "http://kaffeine.sourceforge.net", "kaffeine@gmx.net");
    aboutData->addAuthor("Jürgen Kofler.",0, "kaffeine@gmx.net");

   return aboutData;
}


bool KaffeinePart::openURL(const KURL& url)
{
  kdDebug(555) << "KaffeinePart::openURL()" << endl;

  if ((!imageWindow) || (!url.isValid())) return false;

  if (hrefParam.isNull())
    m_url = url;
   else
    m_url = hrefParam;

  hrefParam = QString::null;  
  m_bTemp = false;

  if (!videoWin->xineRunning)
  {
   /* start xine manual, because polish() sometimes called too early */
    if (!videoWin->InitXine())
    {
      emit canceled(i18n("Can't init Xine Engine!"));
      return false;
    }
    videoWin->SetVisualPlugin( visualPlugin );
    videoWin->slotSetMetaString( metaString );
    playList = new PlayList( videoWin->GetXineEngine(), videoWin, 0, "playlist", false /* without GUI */ );
    connect(videoWin, SIGNAL(signalNewMrlReference(const QString&)), playList, SLOT(slotAddMrl(const QString&)));
    kdDebug(555) << "KaffeinePart: Got URL: " << m_url.url() << endl;
    playList->Add(m_url, NULL);

    if (!playList->GetCurrent())
    {
      kdWarning(555) << "KaffeinePart: No valid url..." << endl;    
      return false;
    }  
  } 

  if (autoStart)
    QTimer::singleShot(100, this, SLOT(slotPlay()));
   else
    QTimer::singleShot(100, this, SLOT(slotStop()));

  return true;
}


bool KaffeinePart::closeURL()
{
  kdDebug(555) << "KaffeinePart::closeURL()" << endl;
  if (playList)
   playList->slotClearList();
  return true;
}   
  

void KaffeinePart::slotPlay()
{
  kdDebug(555) << "KaffeinePart::slotPlay()" << endl;

  if (!videoWin->xineRunning) return;  /** failed to start xine-engine **/

  stopped = false;
  
  QListViewItem* item = playList->GetCurrent();

  if (!item) return;

  if (!videoWin->PlayMRL(item->text(4), item->text(1), true))
    Reset();
   else
     emit setWindowCaption( item->text(1) );
}


void KaffeinePart::slotStop()
{
  kdDebug(555) << "KaffeinePart::slotStop()" << endl;

  if (!videoWin->xineRunning) return;  /** failed to start xine-engine **/
  
  Reset();
  infoRow->slotChangeText(i18n("Ready."));  
}


void KaffeinePart::slotNext()
{
  if (stopped) return;

  QListViewItem* item = playList->GetNext();

  if (!item) return;

  if (!videoWin->PlayMRL(item->text(4), item->text(1), true))
    Reset();
   else
    emit setWindowCaption( item->text(1) ); 
}


void KaffeinePart::slotPrevious()
{
  if (stopped) return;

  QListViewItem* item = playList->GetPrevious();

  if (!item) return;

  if (!videoWin->PlayMRL(item->text(4), item->text(1), true))
    Reset();
   else
    emit setWindowCaption( item->text(1) );  
}


void KaffeinePart::slotPlaybackFinished()
{
  if (stopped) return;  /* end of logo */

  QListViewItem* item = playList->GetNext();

  if (!item)
  {
    slotStop();
    return;
  }  
   
  if (!videoWin->PlayMRL(item->text(4), item->text(1), true))
     Reset();
    else
      emit setWindowCaption( item->text(1) );
}


void KaffeinePart::slotNewTitle(const QString& title)
{
  emit setWindowCaption( title );
}  


void KaffeinePart::Reset()
{
  stopped = true;

  videoWin->slotStopPlayback();
  emit setWindowCaption(i18n("Kaffeine Media Player %1").arg(VERSION));
  slotSetPosition(0, "0:00:00");
}


void KaffeinePart::slotNewXineMessage()
{
  KMessageBox::information( 0, videoWin->GetXineMessage() );
} 
  

void KaffeinePart::slotStartExternal()
{
   KProcess process;
   process << "kaffeine" << m_url.prettyURL();
   kdDebug(555) << "Launch Kaffeine externaly..." << endl;
   process.start(KProcess::DontCare, KProcess::Stderr);
   process.clearArguments();
   
   slotStop();
}


void KaffeinePart::slotSetPosition(int pos, const QString& time)
{
  posSlider->SetValue(pos, false);
  timeButton->setText(time);
}  


/* check if konqueror was moved, send new global position
   of the video window to xine */
void KaffeinePart::slotCheckMoved()
{
  if (!stopped) return;

  QPoint newPos = videoWin->mapToGlobal( QPoint(0,0) );
  if ( newPos != oldPosition )
  {
     videoWin->GlobalPosChanged();
     oldPosition = newPos;
  }   
}  




