/***************************************************************************
                           postfilter.h - wrapper for xine's postprocessing filters
                             -------------------
    begin                : Son Dez 7 2003
    revision             : $Revision: 1.5 $
    last modified        : $Date: 2004/04/26 11:04:40 $ by $Author: juergenk $
    copyright            : (C) 2003-2004 by Juergen  Kofler
    email                : kaffeine@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef POSTFILTER_H
#define POSTFILTER_H

#include <qwidget.h>
#include <qobject.h>
#include <qstring.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qtextedit.h>

#include <kcombobox.h>
#include <knuminput.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kdialogbase.h>


#include <xine.h>

/**
  *@author Juergen  Kofler
  */

class PostFilterParameter : public QObject
{
  Q_OBJECT
public:
  PostFilterParameter( const QString& _name, int _offset, QWidget* parent ) : QObject(parent)
                  { offset = _offset; setName(_name); }
  ~PostFilterParameter() {};
  
  virtual void setValue( const QString& ) = 0;
  virtual QString getValue() = 0;
  virtual QWidget *getWidget() = 0;

protected:
  int offset;
};


class PostFilterParameterInt : public PostFilterParameter
{
  Q_OBJECT
public:
  PostFilterParameterInt( const QString& _name, int _offset, int value, int min, int max, QWidget* parent );
  ~PostFilterParameterInt() {};

  void setValue( const QString &value )
    { int i = value.toInt(); numInput->setValue(i); slotIntValue(i); }
  QString getValue()
    { QString s; s.sprintf("%d",numInput->value()); return s; }
  QWidget *getWidget() { return numInput; }
  
signals:
  void signalIntValue( int, int );

public slots:
  void slotIntValue( int val ) { emit signalIntValue( offset, val ); }

private:
  KIntNumInput* numInput;
};


class PostFilterParameterDouble : public PostFilterParameter
{
  Q_OBJECT
public:
  PostFilterParameterDouble( const QString& _name, int _offset, double value, double min, double max, QWidget* parent );
  ~PostFilterParameterDouble() {};

  void setValue( const QString &value )
    { double d = value.toDouble(); numInput->setValue(d); slotDoubleValue(d); }
  QString getValue()
    { QString s; s.sprintf("%lf",numInput->value()); return s; }
  QWidget *getWidget() { return numInput; }

signals:
  void signalDoubleValue( int, double );

public slots:
  void slotDoubleValue( double val ) { emit signalDoubleValue( offset, val ); }

private:
  KDoubleNumInput* numInput;
};


class PostFilterParameterChar : public PostFilterParameter
{
  Q_OBJECT
public:
  PostFilterParameterChar( const QString& _name, int _offset, char *value, int size, QWidget* parent );
  ~PostFilterParameterChar() {};

  void setValue( const QString &value )
    { charInput->setText(value); slotCharValue(value); }
  QString getValue() { return charInput->text(); }
  QWidget *getWidget() { return charInput; }

signals:
  void signalCharValue( int, const QString&);

public slots:
  void slotCharValue( const QString& val ) { emit signalCharValue( offset, val ); }

private:
  KLineEdit* charInput;
};


class PostFilterParameterCombo : public PostFilterParameter
{
  Q_OBJECT
public:
  PostFilterParameterCombo( const QString& _name, int _offset, int value, char **enums, QWidget* parent );
  ~PostFilterParameterCombo() {};

  void setValue( const QString &value ) { comboBox->setCurrentItem(value); }
  QString getValue() { return comboBox->currentText(); }
  QWidget *getWidget() { return comboBox; }

signals:
  void signalIntValue( int, int );

public slots:
  void slotIntValue( int val ) { emit signalIntValue( offset, val ); }

private:
  KComboBox* comboBox;
};


class PostFilterParameterBool : public PostFilterParameter
{
  Q_OBJECT
public:
  PostFilterParameterBool( const QString& _name, int _offset, bool value, QWidget* parent );
  ~PostFilterParameterBool() {};

  void setValue( const QString &value )
    { bool b = (bool)value.toInt(); checkBox->setChecked(b); slotBoolValue(b); }
  QString getValue()
    { QString s; s.sprintf("%d",(int)checkBox->isOn()); return s; }
  QWidget *getWidget() { return checkBox; }

signals:
  void signalIntValue( int, int );

public slots:
  void slotBoolValue( bool val ) { emit signalIntValue( offset, (int)val ); }

private:
  QCheckBox* checkBox;
};


class PostFilterHelp : public KDialogBase
{
  Q_OBJECT
public:
  PostFilterHelp(QWidget *parent=0, const char *name=0, const char *text=0);
  ~PostFilterHelp();

private:
  QTextEdit *textEdit;
};

 
class PostFilter : public QObject
{
   Q_OBJECT
public: 
   PostFilter(const QString& name, xine_t* engine, xine_audio_port_t* audioDriver,
              xine_video_port_t* videoDriver, QWidget *parent);
  ~PostFilter();

  xine_post_in_t* GetInput() const;
  xine_post_out_t* GetOutput() const;
  void SetConfig( const QString & );
  QString GetConfig();
  
  
signals:
  void signalDeleteMe( PostFilter* me );


private slots:
  void slotDeletePressed() { emit signalDeleteMe( this ); }
  void slotApplyIntValue(int offset, int val); 
  void slotApplyDoubleValue(int offset, double val);
  void slotApplyCharValue(int offset, const QString& val);
  void slotHelpPressed();

private:
  xine_t* xineEngine;
  xine_post_t* xinePost;
  xine_post_api_t* xinePostAPI;
  xine_post_api_descr_t* xinePostDescr;
  xine_post_api_parameter_t* xinePostParameter;
  char* data;

  QGroupBox* groupBox;
  QString filterName;

  QPtrList<PostFilterParameter> parameterList;

};  



#endif
