"""
File: blogdb.py
Author: AN
Date: Jan 2006
Description:

Contains the code related to saving, retrieving, editing and deleting blog
entries.

Todo:
- User management?
"""

import os
import datetime

from kirbybase import KirbyBase

class BlogDB:
    def __init__(self):
        self.dbname = 'blog.tbl'
        self.cdbname = 'comments.tbl'
        self.db = self.getconnection()
        # The field names as the .ks expects them
        self.fieldnames = ['recno', 'date', 'subject', 'body', 'username',
            'comments']
        # The field names as the db expects them
       
    def getconnection(self):
        """Get a kirbybase connection"""
        db = KirbyBase()
        if not os.path.exists(self.dbname):
            db.create(self.dbname,
                ['username:str', 'date:datetime.datetime', 'subject:str', 'body:str', 'comments:int'])
        if not os.path.exists(self.cdbname):
            db.create(self.cdbname,
                ['entryid:int', 'name:str', 'date:datetime.datetime', 'comment:str'])
        return db

    def _getcommentcount(self, recno):
        """Return int, number of comments associated with <recno>"""
        count = self.db.select(self.dbname, ['recno'], [recno], ['comments'])
        if count:
            return count[0][0]
        else:
            return 0
        
    def _setcommentcount(self, recno, count):
        """Set the comment count for <recno>"""
        self.db.update(self.dbname, ['recno'], [recno], {'comments': count})

    def addcomment(self, recno, author, comment):
        """Add a comment"""
        date = datetime.datetime.now()
        recno = int(recno)
        # Store the comment
        self.db.insert(self.cdbname, [recno, author, date, comment])
        count = self._getcommentcount(recno)
        count += 1
        self._setcommentcount(recno, count)

    def getcomments(self, entryid):
        """Fetch comments associated with recno"""
        return self.db.select(self.cdbname, ['entryid'], ['==%s' % entryid], returnType='dict')

    def deletecomment(self, recno):
        """Remove a comment"""
        # Get comment parent id first so we can set the count.
        comment = self.db.select(self.cdbname, ['recno'], [recno], ['entryid'])
        entry = comment[0][0]
        self.db.delete(self.cdbname, ['recno'], [recno])
        count = self._getcommentcount(entry)
        count -= 1
        self._setcommentcount(entry, count)
        

    def getentry(self, recno):
        """Fetch an entry for a given entry id"""
        if not recno.isdigit():
            return None
        
        entry = self.db.select(self.dbname, ['recno'], [int(recno)],
            self.fieldnames, returnType='dict')

        return entry

    def getlastposts(self, number):
        """Get the <number> most recent posts"""
        posts = self.db.select(self.dbname, ['recno'], ['*'],
            self.fieldnames, sortFields=['date'], sortDesc=['date'],
            returnType='dict')
        # No 'limit' operator in KB. Have to do this manually.
        posts = [post for post in posts[:number]]
        return posts

    def getheaders(self):
        """Return date, title, recno dict"""
        posts =  self.db.select(self.dbname, ['recno'], ['*'],
            ['recno', 'date', 'subject'], sortFields=['date'], sortDesc=['date'],
            returnType='dict')
        return posts

    def saveentry(self, username, recno, subject, body):
        """Save or create a new entry (if recno is 'new').
        Returns the entry id or None"""
        # New entry, insert
        if recno == 'new':
            date = datetime.datetime.now()
            return self.db.insert(self.dbname, [username, date, subject, body, 0])
        # Existing entry, update
        elif recno.isdigit():
            self.db.update(
                self.dbname,
                ['recno'], [int(recno)],
                {'username': username, 'subject': subject, 'body': body}    
                )
            return int(recno)
        # None of the above. Fails.
        else:
            return None
        
    def deleteentry(self, recno):
        """Delete the entry <recno>"""
        # Remove the associated comments
        self.db.delete(self.cdbname, ['entryid'], ['==%s' % recno])
        # Remove the blog entry
        self.db.delete(self.dbname, ['recno'], [int(recno)])
        # Compact the db.
        self.db.pack(self.dbname)
        self.db.pack(self.cdbname)