/*
 * menubar.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <gtkmm/menu.h>
#include <gtkmm/stock.h>
#include <gdkmm/types.h>
#include <gdk/gdkkeysyms.h>
#include "menubar.hh"
#include "macros.h"
#include "utils.hh"
#include <iostream>
#include <cassert>

#ifdef ENABLE_HIGHLIGHT
#include <gtksourceviewmm/sourcelanguagesmanager.h>
#include <gtksourceviewmm/init.h>
std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > MenuBar::languages = MenuBar::init_languages();
#endif /* ENABLE_HIGHLIGHT */

// TODO: recent should display the file name only.
// TODO: Integrate with the gtk recent files thing.
MenuBar::MenuBar(Conf& config, Encodings& encodings, std::vector<std::string>& em) :
  _conf(config)
{
  file(config);
  create_recent();
  edit(config);
  search(config);
  view(config, encodings);
  tools(config);
  documents(config);
  help(config);
  emulator(em);
  show_all();
  recent_menu_item->hide();
}

MenuBar::~MenuBar()
{
}

Gtk::Menu *MenuBar::menu(char *str, Gtk::Menu *menu)
{
  Gtk::Menu *m = Gtk::manage(new Gtk::Menu());
  if (menu)
    menu->items().push_back(Gtk::Menu_Helpers::MenuElem(str, *m));
  else
    items().push_back(Gtk::Menu_Helpers::MenuElem(str, *m));

  return m;
}

Gtk::MenuItem *MenuBar::radio_item(Gtk::Menu *menu, Gtk::RadioButtonGroup& group, const std::string& str)
{
  // I'm doing this by hand to disable the mnemonic

  Gtk::RadioMenuItem *item = manage(new Gtk::RadioMenuItem (group, str));

  menu->items().push_back(*item);
  //Gtk::Menu_Helpers::RadioMenuElem(group, str));

  //  return &menu->items().back();
  //  delete item;

  return &menu->items().back();
  //  return item;
}

Gtk::MenuItem *MenuBar::check_item(Gtk::Menu *menu, const std::string& str)
{
  menu->items().push_back(Gtk::Menu_Helpers::CheckMenuElem(str));
  return &menu->items().back();
}

Gtk::MenuItem *MenuBar::item(Gtk::Menu *menu, const Gtk::StockID& stock_id)
{
  menu->items().push_back(Gtk::Menu_Helpers::StockMenuElem(stock_id));
  return &menu->items().back();
}

Gtk::MenuItem *MenuBar::item(Gtk::Menu *menu, const std::string& str)
{
  menu->items().push_back(Gtk::Menu_Helpers::MenuElem(str));
  return &menu->items().back();
}

Gtk::MenuItem *MenuBar::item(Gtk::Menu *menu, const Gtk::StockID& stock_id, guint key, Gdk::ModifierType mod)
{
  Gtk::AccelKey _key(key, mod);
  menu->items().push_back(Gtk::Menu_Helpers::StockMenuElem(stock_id, _key));
  return &menu->items().back();
}

Gtk::MenuItem *MenuBar::item(Gtk::Menu *menu, const std::string& str, guint key, Gdk::ModifierType mod)
{
  Gtk::AccelKey _key(key, mod);
  menu->items().push_back(Gtk::Menu_Helpers::MenuElem(str, _key));
  return &menu->items().back();
}

void MenuBar::separator(Gtk::Menu *menu)
{
  menu->items().push_back(Gtk::Menu_Helpers::SeparatorElem());
}

void MenuBar::file(Conf& config)
{
  file_menu = menu(_("_File"));

  _create = item(file_menu, Gtk::Stock::NEW);
  _open = item(file_menu, Gtk::Stock::OPEN);
  _open_location = item(file_menu, _("Open _Location..."));

  /* Recent */
  recent_menu = menu(_("Recent"), file_menu);
  recent_menu_item = &file_menu->items().back();

  separator(file_menu);
  _save = item(file_menu, Gtk::Stock::SAVE);
  _save_as = item(file_menu, Gtk::Stock::SAVE_AS, GDK_s, Gdk::ModifierType(GDK_CONTROL_MASK|GDK_SHIFT_MASK));
  _save_copy = item(file_menu, _("Sa_ve Copy..."));
  _revert = item(file_menu, Gtk::Stock::REVERT_TO_SAVED);
  separator(file_menu);
  _import_menu = menu(_("_Import"), file_menu);

  std::vector<Import> imports;
  import_init(imports);
  for (unsigned x = 0; x < imports.size(); x++)
    item(_import_menu, imports[x].name)->signal_activate().connect(sigc::bind<Import>(sigc::mem_fun(this, &MenuBar::import_clicked), imports[x]));

  _export_menu = menu(_("_Export"), file_menu);
  std::vector<Export> exports;
  export_init(exports);
  for (unsigned x = 0; x < exports.size(); x++)
    item(_export_menu, exports[x].name)->signal_activate().connect(sigc::bind<Export>(sigc::mem_fun(this, &MenuBar::export_clicked), exports[x]));

  /*
  _export_html = item(_export_menu, _("_HTML Character Reference..."));
  _export_iran = item(_export_menu, _("_Iran System..."));
  _export_plain =  item(_export_menu, _("_Plain text (No unicode control characters..."));
  */
  separator(file_menu);
#ifdef ENABLE_PRINT
  _print = item(file_menu, Gtk::Stock::PRINT, GDK_p, Gdk::ModifierType(GDK_CONTROL_MASK));
  separator(file_menu);
#endif
  _close = item(file_menu, Gtk::Stock::CLOSE);
  _quit = item(file_menu, Gtk::Stock::QUIT);
}

void MenuBar::create_recent()
{
  if (!_conf.get("recent", true))
    return;
  recent_menu->items().clear();
  unsigned recentno = _conf.get("recentno", 10);
  recentno = recentno > _conf.get_recent().size() ? _conf.get_recent().size() : recentno;
  for (unsigned x = 0; x < recentno; x++)
    {
      Gtk::MenuItem *_item = item(recent_menu, Glib::filename_to_utf8(_conf.get_recent()[x]));
      _item->signal_activate().connect(sigc::bind<std::string>(sigc::mem_fun(*this, &MenuBar::recent_clicked), _conf.get_recent()[x]));
    }
}

void MenuBar::edit(Conf& config)
{
  edit_menu = menu(_("_Edit"));
  _undo = item(edit_menu, Gtk::Stock::UNDO, GDK_z, Gdk::ModifierType(GDK_CONTROL_MASK));
  _redo = item(edit_menu, Gtk::Stock::REDO, GDK_z, Gdk::ModifierType(GDK_CONTROL_MASK|GDK_SHIFT_MASK));
  separator(edit_menu);
  _cut = item(edit_menu, Gtk::Stock::CUT);
  _copy = item(edit_menu, Gtk::Stock::COPY);
  _paste = item(edit_menu, Gtk::Stock::PASTE);
  _erase = item(edit_menu, Gtk::Stock::DELETE, GDK_d, Gdk::ModifierType(GDK_CONTROL_MASK));
  separator(edit_menu);
  _select_all = item(edit_menu, _("_Select All"), GDK_a, Gdk::ModifierType(GDK_CONTROL_MASK));
  separator(edit_menu);
  _insert_file = item(edit_menu, _("_Insert File"));
  separator(edit_menu);
  _preferences = item(edit_menu, Gtk::Stock::PREFERENCES);
}

void MenuBar::search(Conf& config)
{
  search_menu = menu(_("_Search"));
  _find = item(search_menu, Gtk::Stock::FIND);
  _find_next = item(search_menu, _("Find Ne_xt"), GDK_g, Gdk::ModifierType(GDK_CONTROL_MASK));
  _replace = item(search_menu, Gtk::Stock::FIND_AND_REPLACE);
  separator(search_menu);
  _goto_line = item(search_menu, Gtk::Stock::JUMP_TO);
}

void MenuBar::view(Conf& config, Encodings& encodings)
{
  Gtk::RadioButtonGroup toolbars, encoding;
  view_menu = menu(_("_View"));

  _statusbar = check_item(view_menu, _("_Statusbar"));
  _line_numbers = check_item(view_menu, _("_Line Numbers"));
  _wrap_text = check_item(view_menu, _("_Wrap Text"));

  separator(view_menu);

  toolbars_menu = menu(_("Toolbars"), view_menu);

  _toolbar = check_item(toolbars_menu, _("_Main Toolbar"));
  _extended_toolbar = check_item(toolbars_menu, _("_Extended Toolbar"));
  separator(toolbars_menu);
  // TODO: Disable these when the main toolbar is not shown.
  _icons = radio_item(toolbars_menu, toolbars, _("Icons only"));
  _text = radio_item(toolbars_menu, toolbars, _("Text only"));
  _both = radio_item(toolbars_menu, toolbars, _("Both"));
  _beside = radio_item(toolbars_menu, toolbars, _("Text beside icons"));
  _encoding_menu = menu(_("_Encoding"), view_menu);
  _encoding_menu->items().push_back(Gtk::Menu_Helpers::TearoffMenuElem());
  Gtk::Menu *item;
  int e = 0;
  for (int x = 0; x < encodings.languages(); x++)
    {
      item = menu(const_cast<char *>(encodings.language(x).c_str()), _encoding_menu);
      for (int y = 0; y < encodings.languages(x); y++)
	{
	Gtk::MenuItem *_item = radio_item(item, encoding, encodings.name(x, y));
	_item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::encoding_clicked), e++));
	encoding_menu_items.push_back(_item);
	}
    }
#ifdef ENABLE_HIGHLIGHT
  Gtk::MenuItem *_item;
  Gtk::RadioButtonGroup highlighters_radio;
  Gtk::Menu *highlight = menu(_("_Highlight"), view_menu);

  Glib::RefPtr<gtksourceview::SourceLanguagesManager> manager = gtksourceview::SourceLanguagesManager::create();
  std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > languages = manager->get_available_language();
  std::map<std::string, std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > > cats;
  for (unsigned x = 0; x < languages.size(); x++) {
    cats[languages[x]->get_section()].push_back(languages[x]);
  }

  _item = radio_item(highlight, highlighters_radio, _("None"));
  highlighters.push_back(_item);
  _item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::highlighter_clicked_cb), 0));

  std::map<std::string, std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > >::iterator iter;

  for (iter = cats.begin(); iter != cats.end(); iter++) {
    item = menu(const_cast<char *>(iter->first.c_str()), highlight);
    std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> >& langs = iter->second;
    for (unsigned x = 0; x < langs.size(); x++) {
      _item = radio_item(item, highlighters_radio, langs[x]->get_name());
      highlighters.push_back(_item);
      _item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::highlighter_clicked_cb), highlighters.size()-1));
    }
  }
#endif /* ENABLE_HIGHLIGHT */
}

void MenuBar::tools(Conf& config)
{
  tools_menu = menu(_("_Tools"));
  _execute = item(tools_menu, _("_Execute Command On Buffer..."), GDK_e, Gdk::ModifierType(GDK_CONTROL_MASK));
  separator(tools_menu);
#ifdef HAVE_SPELL
  _spell = item(tools_menu, Gtk::Stock::SPELL_CHECK, GDK_F7, Gdk::ModifierType(GDK_CONTROL_MASK));
  _auto_spell = check_item(tools_menu, _("_Autocheck Spelling"));
#endif
}

void MenuBar::emulator(std::vector<std::string>& items)
{
  Gtk::RadioButtonGroup em;
  _emulator_menu = menu(_("Keyboard emulator"), tools_menu);

  Gtk::MenuItem *__item = radio_item(_emulator_menu, em, _("Disable"));
  dynamic_cast<Gtk::RadioMenuItem *>(__item)->set_active(true);
  __item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::emulator_clicked), -1));
  separator(_emulator_menu);

  for (unsigned x = 0; x < items.size(); x++)
    {
      Gtk::MenuItem *_item = radio_item(_emulator_menu, em, items[x]);
      _item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::emulator_clicked), x));
    }
}

void MenuBar::documents(Conf& config)
{
  documents_menu = menu(_("_Documents"));
  _save_all = item(documents_menu, _("Save _All"));
  _close_all = item(documents_menu, _("Clos_e All"), GDK_w, Gdk::ModifierType(GDK_CONTROL_MASK|GDK_SHIFT_MASK));
  separator(documents_menu);

  opened_menu = menu(_("_Opened Documents"), documents_menu);
  closed_menu = menu(_("_Closed Documents"), documents_menu);
  documents_menu->show_all();
}

void MenuBar::help(Conf& config)
{
  help_menu = menu(_("_Help"));
  _about = item(help_menu, Gtk::Stock::ABOUT, GDK_t, Gdk::ModifierType(GDK_CONTROL_MASK));
}

void MenuBar::emulator_clicked(int x)
{
  emulator_activated.emit(x);
}

void MenuBar::recent_clicked(std::string file)
{
  recent_activated.emit(file);
}

void MenuBar::set_encoding(unsigned e)
{
  assert (e <= encoding_menu_items.size());

  _ignore_encoding_changed_signal_hack = true;

  Gtk::MenuItem *item = encoding_menu_items[e];
  dynamic_cast<Gtk::CheckMenuItem *>(item)->set_active(true);
  _ignore_encoding_changed_signal_hack = false;
}

void MenuBar::encoding_clicked(unsigned e)
{
  assert (e <= encoding_menu_items.size());
  if (_ignore_encoding_changed_signal_hack)
    return;

  Gtk::MenuItem *item = encoding_menu_items[e];
  if (dynamic_cast<Gtk::CheckMenuItem *>(item)->get_active())
    encoding_activated.emit(e);

#if 0
  e++;

  MenuList& items = encoding_menu->items();
  for (int x = 0; x < items.size(); x++)
    {
      MenuList& _items = items[x].get_submenu()->items();
      int t = e - _items.size();
      if (t > 0)
	e -= _items.size();
      else
	{
	  // TODO: BAD BAD BAD VERY BAD AND UGLY.
	  if (((Gtk::RadioMenuItem *)&_items[--e])->get_active())
	    {
	      encoding_activated.emit(e);
	      return;
	    }
	}
    }
#endif
}

void MenuBar::document_clicked(int x)
{
  if (_ignore_document_clicked_signal_hack)
    return;
  if (static_cast<Gtk::RadioMenuItem&>(opened_menu->items()[x]).get_active())
    document_activated.emit(x);
}

void MenuBar::document_add(std::string& label, bool ro, bool m)
{
  DocItem item(label, ro, m);
  _documents.push_back(item);

  documents_menu_clear();
  documents_menu_build();

  MenuList& ml = opened_menu->items();
  MenuList::iterator iter = ml.end();
  iter--;

  _ignore_document_clicked_signal_hack = true;
  Gtk::MenuItem& itm = *iter;
  static_cast<Gtk::RadioMenuItem&>(itm).set_active(true);
  _ignore_document_clicked_signal_hack = false;
}

void MenuBar::document_set_active(int x)
{
  _active = x;
  if (_active >= opened_menu->items().size())
    return;
  _ignore_document_clicked_signal_hack = true;
  Gtk::MenuItem& item = opened_menu->items()[x];
  static_cast<Gtk::RadioMenuItem&>(item).set_active(true);
  // For some strange reason, We lose the colors when we switch tabs.
  // I'm setting them here.
  document_set_modified(x, _documents[x].get_modified());
  document_set_readonly(x, _documents[x].get_readonly());
  _ignore_document_clicked_signal_hack = false;
}

void MenuBar::document_remove(int x)
{
  documents_menu_clear();

  std::vector<DocItem>::iterator iter = _documents.begin() + x;
  _documents.erase(iter);

  documents_menu_build();
}

void MenuBar::document_set_label(int x, std::string& str)
{
  _documents[x].set_label(str);
  dynamic_cast<Gtk::Label *>(opened_menu->items()[x].get_child())->set_text(str);
}

void MenuBar::document_set_modified(int x, bool m)
{
  assert (x < _documents.size());

  // A read only document can't be modified.
  if (_documents[x].get_readonly())
    return;

  _documents[x].set_modified(m);
  document_set_modified(opened_menu->items()[x], m);
}

void MenuBar::document_set_readonly(int x, bool ro)
{
  assert (x < _documents.size());

  // A modified document can't be read only.
  if (_documents[x].get_modified())
    return;

  _documents[x].set_readonly(ro);

  document_set_readonly(opened_menu->items()[x], ro);
}

void MenuBar::document_set_readonly(Gtk::MenuItem& item, bool ro)
{
  if (ro)
    katoob_set_color(_conf, dynamic_cast<Gtk::Label *>(item.get_child()), Utils::KATOOB_COLOR_READONLY);
  else
    document_set_normal(item);
}

void MenuBar::document_set_modified(Gtk::MenuItem& item, bool m)
{
  if (m)
    katoob_set_color(_conf, dynamic_cast<Gtk::Label *>(item.get_child()), Utils::KATOOB_COLOR_MODIFIED);
  else
    document_set_normal(item);
}

void MenuBar::document_set_normal(Gtk::MenuItem& item)
{
  katoob_set_color(_conf, dynamic_cast<Gtk::Label *>(item.get_child()), Utils::KATOOB_COLOR_NORMAL);
}

void MenuBar::documents_menu_clear()
{
  opened_menu->items().clear();
}

void MenuBar::documents_menu_build()
{
  Gtk::RadioButtonGroup documents_radio;

  for (unsigned i = 0; i < _documents.size(); i++)
    {
      Gtk::MenuItem *_item = radio_item(opened_menu, documents_radio, _documents[i].get_label());
      _item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::document_clicked), i));

      Gtk::MenuItem& itm = *_item;

      bool r = _documents[i].get_readonly();
      bool m = _documents[i].get_modified();
      if (r)
	document_set_readonly(itm, true);
      if (m)
	document_set_modified(itm, true);
      if ((!r) && (!m))
	document_set_normal(itm);
      _item->show();
    }
}

void MenuBar::signal_closed_document_erased_cb()
{
  // This is called when we remove an item from the beginning.
  _closed_documents.erase(_closed_documents.begin());
  closed_documents_rebuild();
}

void MenuBar::signal_closed_document_added(std::string title)
{
  // We will add an item to the end.
  ClosedDocItem doc;
  doc.title = title;
  doc.n = _closed_documents.size() +1;
  _closed_documents.push_back(doc);
  closed_documents_rebuild();
}

void MenuBar::closed_documents_rebuild()
{
  closed_menu->items().clear();

  Gtk::MenuItem *_item;
  for (unsigned i = 0; i < _closed_documents.size(); i++)
    {
      _item = item(closed_menu, _closed_documents[i].title);
      _item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this, &MenuBar::signal_closed_document_clicked), i));
    }
}

void MenuBar::signal_closed_document_clicked(int n)
{
  std::vector<ClosedDocItem>::iterator iter = _closed_documents.begin();
  iter += n;
  _closed_documents.erase(iter);
  closed_documents_rebuild();
  closed_document_activated.emit(n);
}

void MenuBar::import_clicked(Import imp)
{
  import_clicked_signal.emit(imp);
}

void MenuBar::export_clicked(Export exp)
{
  export_clicked_signal.emit(exp);
}

void MenuBar::reset_gui()
{
  dynamic_cast<Gtk::CheckMenuItem *>(_statusbar)->set_active(_conf.get("statusbar", true));
  dynamic_cast<Gtk::CheckMenuItem *>(_line_numbers)->set_active(_conf.get("linenumbers", true));
  dynamic_cast<Gtk::CheckMenuItem *>(_wrap_text)->set_active(_conf.get("textwrap", true));

  dynamic_cast<Gtk::CheckMenuItem *>(_toolbar)->set_active(_conf.get("toolbar", true));
  dynamic_cast<Gtk::CheckMenuItem *>(_extended_toolbar)->set_active(_conf.get("extended_toolbar", true));
#ifdef HAVE_SPELL
  dynamic_cast<Gtk::CheckMenuItem *>(_auto_spell)->set_active(_conf.get("spell_check", true));
#endif

  std::string toolbartype = _conf.get("toolbartype", "both");
  if (toolbartype == "text")
    dynamic_cast<Gtk::RadioMenuItem *>(_text)->set_active();
  else if (toolbartype == "icons")
    dynamic_cast<Gtk::RadioMenuItem *>(_icons)->set_active();
  else if (toolbartype == "both_horiz")
    dynamic_cast<Gtk::RadioMenuItem *>(_beside)->set_active();
  else
    dynamic_cast<Gtk::RadioMenuItem *>(_both)->set_active();

  if (_conf.get("recent", true))
    recent_menu_item->show();
  else
    recent_menu_item->hide();

  create_recent();

  documents_menu_clear();
  documents_menu_build();
  document_set_active(_active);
}

#ifdef ENABLE_HIGHLIGHT
std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > MenuBar::init_languages() {
  Glib::init();
  gtksourceview::init();
  static std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > _languages;
  Glib::RefPtr<gtksourceview::SourceLanguagesManager> manager = gtksourceview::SourceLanguagesManager::create();
  std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > languages = manager->get_available_language();
  std::map<std::string, std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > > cats;
  for (unsigned x = 0; x < languages.size(); x++) {
    cats[languages[x]->get_section()].push_back(languages[x]);
  }

  std::map<std::string, std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > >::iterator iter;

  for (iter = cats.begin(); iter != cats.end(); iter++) {
    std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> >& langs = iter->second;
    for (unsigned x = 0; x < langs.size(); x++) {
      _languages.push_back(langs[x]);
    }
  }
  return _languages;
}
#endif
