/*
 * menubar.hh
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __MENUBAR_HH__
#define __MENUBAR_HH__

#include <gtkmm/menubar.h>
#include "conf.hh"
#include "encodings.hh"
#include "import.hh"
#include "export.hh"
#ifdef ENABLE_HIGHLIGHT
#include <gtksourceviewmm/sourcelanguage.h>
#include <cassert>
#endif /* ENABLE_HIGHLIGHT */

class DocItem {
public:
  DocItem(std::string& str, bool ro, bool m) : _label(str), _ro(ro), _m(m) {}
  void set_readonly(bool ro) { _ro = ro; }
  bool get_readonly() { return _ro; }
  void set_modified(bool m) { _m = m; }
  bool get_modified() { return _m; }
  void set_label(std::string& str) { _label = str; }
  std::string get_label() { return _label; }
private:
  std::string _label;
  bool _ro, _m;
};

struct ClosedDocItem {
  int n;
  std::string title;
};

class MenuBar : public Gtk::MenuBar {
public:
  MenuBar(Conf&, Encodings&, std::vector<std::string>&);
  ~MenuBar();

  void create_recent();

  void set_encoding(unsigned);

  Gtk::MenuItem* create() { return _create; }
  Gtk::MenuItem* open() { return _open; }
  Gtk::MenuItem* open_location() { return _open_location; }
  Gtk::MenuItem* save() { return _save; }
  Gtk::MenuItem* save_as() { return _save_as; }
  Gtk::MenuItem* save_copy() { return _save_copy; }
  Gtk::MenuItem* revert() { return _revert; }
#ifdef ENABLE_PRINT
  Gtk::MenuItem* print() { return _print; }
#endif
  Gtk::MenuItem* close() { return _close; }
  Gtk::MenuItem* quit() { return _quit; }

  Gtk::MenuItem* undo() { return _undo; }
  Gtk::MenuItem* redo() { return _redo; }
  Gtk::MenuItem* cut() { return _cut; }
  Gtk::MenuItem* copy() { return _copy; }
  Gtk::MenuItem* paste() { return _paste; }
  Gtk::MenuItem* erase() { return _erase; }
  Gtk::MenuItem* select_all() { return _select_all; }
  Gtk::MenuItem* preferences() { return _preferences; }
  Gtk::MenuItem *insert_file() { return _insert_file; }

  Gtk::MenuItem* find() { return _find; }
  Gtk::MenuItem* find_next() { return _find_next; }
  Gtk::MenuItem* replace() { return _replace; }
  Gtk::MenuItem* goto_line() { return _goto_line; }

  Gtk::MenuItem* statusbar() { return _statusbar; }
  Gtk::MenuItem* wrap_text() { return _wrap_text; }
  Gtk::MenuItem* line_numbers() { return _line_numbers; }
  Gtk::MenuItem* toolbar() { return _toolbar; }
  Gtk::MenuItem* extended_toolbar() { return _extended_toolbar; }
  Gtk::MenuItem* text() { return _text; }
  Gtk::MenuItem* icons() { return _icons; }
  Gtk::MenuItem* both() { return _both; }
  Gtk::MenuItem* beside() { return _beside; }

  Gtk::MenuItem* execute() { return _execute; }

#ifdef HAVE_SPELL
  Gtk::MenuItem* spell() { return _spell; }
  Gtk::MenuItem* auto_spell() { return _auto_spell; }
#endif

  Gtk::MenuItem* save_all() { return _save_all; }
  Gtk::MenuItem* close_all() { return _close_all; }

  Gtk::MenuItem* about() { return _about; }

  Gtk::Menu *export_menu() { return _export_menu; }
  Gtk::Menu *encoding_menu() { return _encoding_menu; }
  Gtk::Menu *emulator_menu() { return _emulator_menu; }

  //  void document_set_modified(bool, int);
  //  void document_set_readonly(bool, int);
  void document_add(std::string&, bool, bool);
  void document_remove(int);
  void document_set_active(int);
  void document_set_modified(int, bool);
  void document_set_readonly(int, bool);
  void document_set_label(int, std::string&);

  sigc::signal<void, std::string&> recent_activated;
  sigc::signal<void, int> emulator_activated;
  sigc::signal<void, int> encoding_activated;
  sigc::signal<void, int> document_activated;

  sigc::signal<void, int> closed_document_activated;

  sigc::signal<void, Import> import_clicked_signal;
  sigc::signal<void, Export> export_clicked_signal;

  void signal_closed_document_erased_cb();
  void signal_closed_document_added(std::string);
  void signal_closed_document_clicked(int);

  void reset_gui();

#ifdef ENABLE_HIGHLIGHT
  static int get_language(Glib::RefPtr<gtksourceview::SourceLanguage>& language) {
    for (unsigned x = 0; x < MenuBar::languages.size(); x++) {
      if (languages[x]->get_name() == language->get_name())
	return ++x;
    }
    return 0;
  }
  static Glib::RefPtr<gtksourceview::SourceLanguage>&  get_language(int x) {
    assert(x != 0);
    return languages[x-1];
  }

  sigc::signal<void, int> highlighter_clicked;
  void highlighter_clicked_cb(unsigned x) {
    if (_ignore_highlighting_changed_signal_hack == true)
      return;
    if (dynamic_cast<Gtk::CheckMenuItem *>(highlighters[x])->get_active())
      highlighter_clicked.emit(x);
  }
  void set_highlight(int x) {
    _ignore_highlighting_changed_signal_hack = true;
    dynamic_cast<Gtk::CheckMenuItem *>(highlighters[x])->set_active(true);
    _ignore_highlighting_changed_signal_hack = false;
  }
#endif /* ENABLE_HIGHLIGHT */

private:
  void file(Conf&);
  void edit(Conf&);
  void search(Conf&);
  void view(Conf&, Encodings&);
  void tools(Conf&);
  void documents(Conf&);
  void help(Conf&);
  void recent(Conf&);
  void encodings(Conf&);

  void emulator(std::vector<std::string>&);

  void recent_clicked(std::string);
  void emulator_clicked(int);
  void encoding_clicked(unsigned);
  void document_clicked(int);

  void document_set_modified(Gtk::MenuItem&, bool);
  void document_set_readonly(Gtk::MenuItem&, bool);
  void document_set_normal(Gtk::MenuItem&);

  void documents_menu_clear();
  void documents_menu_build();

  void closed_documents_rebuild();

  void import_clicked(Import);
  void export_clicked(Export);

  Gtk::Menu *menu(char *, Gtk::Menu * = NULL);
  Gtk::MenuItem *item(Gtk::Menu *, const Gtk::StockID&, guint, Gdk::ModifierType);
  Gtk::MenuItem *item(Gtk::Menu *, const Gtk::StockID&);
  Gtk::MenuItem *item(Gtk::Menu *, const std::string&);
  Gtk::MenuItem *item(Gtk::Menu *, const std::string&, guint, Gdk::ModifierType);
  Gtk::MenuItem *check_item(Gtk::Menu *, const std::string&);
  Gtk::MenuItem *radio_item(Gtk::Menu *, Gtk::RadioButtonGroup&, const std::string&);

  void separator(Gtk::Menu *);

  Gtk::Menu *file_menu, *edit_menu, *search_menu,
    *view_menu, *tools_menu, *documents_menu, *help_menu, *opened_menu, *closed_menu;

  /* File */
  Gtk::Menu *recent_menu, *_import_menu, *_export_menu, *toolbars_menu, *_encoding_menu;
  Gtk::Menu *_emulator_menu;
  Gtk::MenuItem *recent_menu_item;

  Gtk::MenuItem *_create;
  Gtk::MenuItem *_open;
  Gtk::MenuItem *_open_location;
  Gtk::MenuItem *_save;
  Gtk::MenuItem *_save_as;
  Gtk::MenuItem *_save_copy;
  Gtk::MenuItem *_revert;
#ifdef ENABLE_PRINT
  Gtk::MenuItem *_print;
#endif
  Gtk::MenuItem *_close;
  Gtk::MenuItem *_quit;
  /* Edit */
  Gtk::MenuItem *_undo;
  Gtk::MenuItem *_redo;
  Gtk::MenuItem *_cut;
  Gtk::MenuItem *_copy;
  Gtk::MenuItem *_paste;
  Gtk::MenuItem *_erase;
  Gtk::MenuItem *_select_all;
  Gtk::MenuItem *_preferences;
  Gtk::MenuItem *_insert_file;
  /* Search */
  Gtk::MenuItem *_find;
  Gtk::MenuItem *_find_next;
  Gtk::MenuItem *_replace;
  Gtk::MenuItem *_goto_line;
  /* View */
  Gtk::MenuItem *_statusbar;
  Gtk::MenuItem *_wrap_text;
  Gtk::MenuItem *_line_numbers;
  Gtk::MenuItem *_toolbar;
  Gtk::MenuItem *_extended_toolbar;
  Gtk::MenuItem *_text;
  Gtk::MenuItem *_icons;
  Gtk::MenuItem *_both;
  Gtk::MenuItem *_beside;

  /* Tools */
  Gtk::MenuItem *_execute;
#ifdef HAVE_SPELL
  Gtk::MenuItem *_spell;
  Gtk::MenuItem *_auto_spell;
#endif
  /* Documents */
  Gtk::MenuItem *_save_all;
  Gtk::MenuItem *_close_all;
  /* Help */
  Gtk::MenuItem *_about;

  bool _ignore_encoding_changed_signal_hack;
  bool _ignore_document_clicked_signal_hack;
  bool _ignore_highlighting_changed_signal_hack;
  std::vector<Gtk::MenuItem *> encoding_menu_items;

  Conf& _conf;
  std::vector<DocItem> _documents;
  std::vector<ClosedDocItem> _closed_documents;
  int _active;
#ifdef ENABLE_HIGHLIGHT
  static std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > init_languages();
  std::vector<Gtk::MenuItem *> highlighters;
  static std::vector<Glib::RefPtr<gtksourceview::SourceLanguage> > languages;
#endif /* ENABLE_HIGHLIGHT */
};

#endif /* __MENUBAR_HH__ */
