/*
 * window.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iostream>
#include "aboutdialog.hh"
#include "window.hh"
#include "utils.hh"
#include "macros.h"
#include "dialogs.hh"
#include "preferencesdialog.hh"

Window::Window(Conf& conf, Encodings& encodings, std::vector<std::string>& files) :
  _emulator(-1),
  _conf(conf),
  _encodings(encodings),
  menubar(conf, encodings, list()),
  toolbar(conf),
  mdi(conf, encodings),
  box(false, 0)
{
  box.pack_start(menubar, Gtk::PACK_SHRINK, 0);
  box.pack_start(toolbar.main(), false, false, 0);
  box.pack_start(toolbar.extended(), Gtk::PACK_SHRINK, 0);
  box.pack_start(mdi, Gtk::PACK_EXPAND_WIDGET, 0);
  box.pack_start(statusbar, Gtk::PACK_SHRINK, 0);

  add(box);

  // Now let's connect our signals.
  /* Menubar */
  menubar.create()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::create_cb));
  menubar.open()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::open_cb));
  menubar.open_location()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::open_location_cb));
  menubar.save()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::save_cb));
  menubar.save_as()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::save_as_cb));
  menubar.save_copy()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::save_copy_cb));
  menubar.revert()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::revert_cb));
#ifdef ENABLE_PRINT
  menubar.print()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::print_cb));
#endif
  menubar.close()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::close_cb));
  menubar.quit()->signal_activate().connect(sigc::mem_fun(*this, &Window::quit_cb));

  menubar.undo()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::undo_cb));
  menubar.redo()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::redo_cb));
  menubar.cut()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::cut_cb));
  menubar.copy()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::copy_cb));
  menubar.paste()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::paste_cb));
  menubar.erase()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::erase_cb));
  menubar.select_all()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::select_all_cb));
  menubar.preferences()->signal_activate().connect(sigc::mem_fun(*this, &Window::preferences_cb));
  menubar.insert_file()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::insert_file_cb));

  menubar.find()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::find_cb));
  menubar.find_next()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::find_next_cb));
  menubar.replace()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::replace_cb));
  menubar.goto_line()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::goto_line_cb));

  wrap_text_conn = menubar.wrap_text()->signal_activate().connect(sigc::mem_fun(*this, &Window::wrap_text_cb));
  line_numbers_conn = menubar.line_numbers()->signal_activate().connect(sigc::mem_fun(*this, &Window::line_numbers_cb));
  menubar.statusbar()->signal_activate().connect(sigc::mem_fun(*this, &Window::statusbar_cb));
  menubar.toolbar()->signal_activate().connect(sigc::mem_fun(*this, &Window::toolbar_cb));
  menubar.extended_toolbar()->signal_activate().connect(sigc::mem_fun(*this, &Window::extended_toolbar_cb));
  menubar.text()->signal_activate().connect(sigc::mem_fun(*this, &Window::text_cb));
  menubar.icons()->signal_activate().connect(sigc::mem_fun(*this, &Window::icons_cb));
  menubar.both()->signal_activate().connect(sigc::mem_fun(*this, &Window::both_cb));
  menubar.beside()->signal_activate().connect(sigc::mem_fun(*this, &Window::beside_cb));

  menubar.execute()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::execute_cb));

#ifdef HAVE_SPELL
  menubar.spell()->signal_activate().connect(sigc::mem_fun(*this, &Window::spell_cb));
  mdi.signal_document_spell_enabled.connect(sigc::mem_fun(*this, &Window::on_document_spell_enabled_cb));
  auto_spell_conn = menubar.auto_spell()->signal_activate().connect(sigc::mem_fun(*this, &Window::auto_spell_cb));
#endif
  mdi.signal_document_wrap_text.connect(sigc::mem_fun(*this, &Window::signal_document_wrap_text_cb));
  mdi.signal_document_line_numbers.connect(sigc::mem_fun(*this, &Window::signal_document_line_numbers_cb));

  menubar.save_all()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::save_all_cb));
  menubar.close_all()->signal_activate().connect(sigc::mem_fun(mdi, &MDI::close_all_cb));

  menubar.about()->signal_activate().connect(sigc::mem_fun(*this, &Window::about_cb));

  /* Toolbar */
  toolbar.create().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::create_cb));
  toolbar.open().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::open_cb));
  toolbar.save().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::save_cb));
#ifdef ENABLE_PRINT
  toolbar.print().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::print_cb));
#endif
  toolbar.close().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::close_cb));
  toolbar.undo().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::undo_cb));
  toolbar.redo().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::redo_cb));
  toolbar.cut().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::cut_cb));
  toolbar.copy().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::copy_cb));
  toolbar.paste().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::paste_cb));
  toolbar.erase().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::erase_cb));

  menubar.recent_activated.connect(sigc::mem_fun(mdi, &MDI::recent_cb));
  mdi.recent_regenerate.connect(sigc::mem_fun(menubar, &MenuBar::create_recent));

  menubar.emulator_activated.connect(sigc::mem_fun(*this, &Window::on_emulator_activated_cb));
  menubar.encoding_activated.connect(sigc::mem_fun(*this, &Window::on_encoding_activated_cb));
  menubar.document_activated.connect(sigc::mem_fun(mdi, &MDI::activate));

#ifdef ENABLE_HIGHLIGHT
  menubar.highlighter_clicked.connect(sigc::mem_fun(mdi, &MDI::highlighter_clicked_cb));
  mdi.signal_document_highlight.connect(sigc::mem_fun(*this, &Window::signal_document_highlight_cb));
#endif
  // Document status signals.
  mdi.signal_document_encoding_changed.connect(sigc::mem_fun(*this, &Window::signal_document_encoding_changed_cb));
  mdi.signal_document_overwrite_toggled.connect(sigc::mem_fun(*this, &Window::signal_document_overwrite_toggled_cb));
  mdi.signal_document_cursor_moved.connect(sigc::mem_fun(*this, &Window::signal_document_cursor_moved_cb));
  mdi.signal_document_file_changed.connect(sigc::mem_fun(*this, &Window::signal_document_file_changed_cb));
  mdi.signal_document_readonly.connect(sigc::mem_fun(*this, &Window::signal_document_readonly_cb));
  mdi.signal_document_can_redo.connect(sigc::mem_fun(*this, &Window::signal_document_can_redo_cb));
  mdi.signal_document_can_undo.connect(sigc::mem_fun(*this, &Window::signal_document_can_undo_cb));
  mdi.signal_document_modified.connect(sigc::mem_fun(*this, &Window::signal_document_modified_cb));
  mdi.signal_document_title_changed.connect(sigc::mem_fun(*this, &Window::signal_document_title_changed_cb));
#ifdef HAVE_SPELL
  mdi.signal_document_dictionary_changed.connect(sigc::mem_fun(*this, &Window::signal_document_dictionary_changed_cb));
#endif

  mdi.signal_doc_activated.connect(sigc::mem_fun(*this, &Window::on_doc_activated));
  mdi.signal_reset_gui.connect(sigc::mem_fun(*this, &Window::on_reset_gui));
  mdi.signal_document_added.connect(sigc::mem_fun(*this, &Window::on_document_added_cb));
  mdi.signal_document_removed.connect(sigc::mem_fun(*this, &Window::on_document_removed_cb));

  // undo closed documents signals.
  mdi.signal_closed_document_erased.connect(sigc::mem_fun(menubar, &MenuBar::signal_closed_document_erased_cb));
  mdi.signal_closed_document_added.connect(sigc::mem_fun(menubar, &MenuBar::signal_closed_document_added));
  menubar.closed_document_activated.connect(sigc::mem_fun(mdi, &MDI::closed_document_activated_cb));

  // Import
  menubar.import_clicked_signal.connect(sigc::mem_fun(mdi, &MDI::import_cb));
  // Export
  menubar.export_clicked_signal.connect(sigc::mem_fun(mdi, &MDI::export_cb));

  // The rest of the extended toolbar signals.
#ifdef HAVE_SPELL
  toolbar.dictionary_changed.connect(sigc::mem_fun(*this, &Window::dictionary_cb));
  toolbar.spell().signal_clicked().connect(sigc::mem_fun(mdi, &MDI::spell));
#endif
  toolbar.search_activated.connect(sigc::mem_fun(*this, &Window::search_cb));
  toolbar.goto_activated.connect(sigc::mem_fun(mdi, &MDI::goto_line_cb2));

  toolbar.signal_extra_button_clicked.connect(sigc::mem_fun(mdi, &MDI::signal_extra_button_clicked_cb));

  reset_gui();

  // This is here after we connect the signals
  // So we can get the signals emitted when we create the first document.
  // Now let's create the children.
  if (files.size() == 0)
    mdi.document();
  else
    for (unsigned int x = 0; x < files.size(); x++)
      mdi.document(files[x]);
  files.clear();

  // It's possible that we might fail to detect the encoding of all the files
  // passed as arguments. In this case, we will create a blank document.
  // We can't the number of the documents because MDI has no size method.
  if (!mdi.active())
    mdi.document();

  move(conf.get("x", 50), conf.get("y", 50));
  resize (conf.get("w", 500), conf.get("h", 400));
  try {
    set_icon_from_file(Utils::get_data_path("katoob-small.png"));
  }
#ifndef _WIN32
  catch (Glib::Error& er)
    {
      std::cout << er.what() << std::endl;
    }
#else
 catch (...)
   {
     // NOTE: Why the hell can't I catch Glib::Error or Glib::FileError under win32 ?
#ifndef NDEBUG
     std::cout << "I can't set the main window icon to " << Utils::get_data_path("katoob-small.png") << std::endl;
#endif
   }
#endif

  //  set_title();
  signal_delete_event().connect(sigc::mem_fun(*this, &Window::on_delete_event));

  // We don't call show_all(); here as our statusbar needs to hide a few things!
  box.show();
  mdi.show_all();
  show();

  Document *doc = mdi.active();
  if (doc)
    doc->grab_focus();

  std::string err;
  if (!ok(err))
    {
      err += _(" The keyboard emulator will not work.");
      katoob_error(err);
    }

  std::string ver = conf.get_version();
  if ((ver.size() == 0) && conf.ok())
    {
      std::string message(_("A lot of the configuration options changed in this version.\nPlease adjust the configuration first."));
      katoob_info(message);
      preferences_cb();
    }

  // DnD
  std::list<Gtk::TargetEntry> targets;
  targets.push_back(Gtk::TargetEntry("text/uri-list") );
  drag_dest_set(targets);
  signal_drag_data_received().connect(sigc::mem_fun(*this, &Window::signal_drag_data_received_cb));
}

Window::~Window()
{

}

void Window::signal_drag_data_received_cb(const Glib::RefPtr<Gdk::DragContext>& context, int x,int y,const Gtk::SelectionData& selection, guint info, guint time)
{

  std::vector<std::string> files = selection.get_uris();
  for (unsigned x = 0; x < files.size(); x++)
    {
      std::string filename;
      try {
	filename = Glib::filename_from_uri(files[x]);
	mdi.document(filename);
      }
      catch(Glib::ConvertError& e)
	{
	  std::string err = e.what();
	  katoob_error(err);
	}
    }
}

void Window::set_title(char *str)
{
  std::string title;
  if (str)
    {
      title = str;
      title += " - ";
    }
  title += PACKAGE;
  title += " ";
  title += VERSION;
  Gtk::Window::set_title(title);
}

void Window::about_cb()
{
  AboutDialog dlg;
  dlg.run();
}

void Window::statusbar_cb()
{
  bool val = dynamic_cast<Gtk::CheckMenuItem *>(menubar.statusbar())->get_active();
  if (val)
    statusbar.show();
  else
    statusbar.hide();
  _conf.set("statusbar", val);
}

void Window::toolbar_cb()
{
  bool val = dynamic_cast<Gtk::CheckMenuItem *>(menubar.toolbar())->get_active();
  if (val)
    toolbar.main().show();
  else
    toolbar.main().hide();
  _conf.set("toolbar", val);
}

void Window::extended_toolbar_cb()
{
  bool val = dynamic_cast<Gtk::CheckMenuItem *>(menubar.extended_toolbar())->get_active();
  if(val)
    toolbar.extended().show();
  else
    toolbar.extended().hide();
  _conf.set("extended_toolbar", val);
}

void Window::preferences_cb()
{
  PreferencesDialog dialog(_conf, _encodings);
  dialog.signal_apply_clicked.connect(sigc::mem_fun(*this, &Window::preferences_dialog_apply_clicked_cb));
  if (dialog.run())
    reset_gui();
}

#ifdef HAVE_SPELL
void Window::auto_spell_cb()
{
  mdi.auto_spell(dynamic_cast<Gtk::CheckMenuItem *>(menubar.auto_spell())->get_active());
}

void Window::on_document_spell_enabled_cb(bool s)
{
  auto_spell_conn.block();
  dynamic_cast<Gtk::CheckMenuItem *>(menubar.auto_spell())->set_active(s);
  toolbar.dictionary().set_sensitive(s);
  auto_spell_conn.unblock();
}

void Window::spell_cb()
{
  mdi.spell();
}
#endif

bool Window::on_delete_event(GdkEventAny *event)
{
  if (mdi.close_all())
    {
      signal_quit.emit();
      return false;
    }
  else
    return true;
}

void Window::quit_cb()
{
  if (mdi.close_all())
    signal_quit.emit();
}

void Window::text_cb()
{
  if (dynamic_cast<Gtk::CheckMenuItem *>(menubar.text())->get_active())
    toolbar.text();
}

void Window::icons_cb()
{
  if (dynamic_cast<Gtk::CheckMenuItem *>(menubar.icons())->get_active())
    toolbar.icons();
}

void Window::both_cb()
{
  if (dynamic_cast<Gtk::CheckMenuItem *>(menubar.both())->get_active())
    toolbar.both();
}

void Window::beside_cb()
{
  if (dynamic_cast<Gtk::CheckMenuItem *>(menubar.beside())->get_active())
    toolbar.beside();
}

bool Window::on_key_press_event(GdkEventKey* event)
{
  // TODO: All our keyboard shortcuts are not working.

  Document *doc = mdi.active();
  if ((!doc) || (doc->readonly()) || (!doc->has_focus()))
    return Gtk::Window::on_key_press_event(event);

  if (_emulator < 0)
    {
      // NOTE: Handled in the insert event.
#if 0
      // Here we will handle the lam-alef inserts.
      // According to the xfree86 guys, It's not possible to map 1 key to 2 letter.
      // I'm not sure whether it's not possible or it's that they don't like it.
      // A workaround by Isam Bayazidi of Arabeyes was to map the key to the Unicode
      // isolated presentation form.
      // So what we save in the files is the presentation form not the lam+alef
      // characters.
      // There is no LAM-ALEF character in the unicode standard, As it's a
      // presentation form for the lam+alef characters togeather.
      // Though it works, But when converting the UTF-8 or unicode encoding to
      // cp1256 it breaks.
      // Here we'll try to catch the presentation form and insert lam followed by alef.
      // Look at: /usr/X11R6/lib/X11/xkb/symbols/ar
      // Or: /usr/share/X11/xkb/symbols/ara
      // Or: /etc/X11/xkb/symbols/ar
      // Finally: THIS SUCK A LOT.
      // Someone should contact the xorg guys and see.
      Glib::ustring res;
      if (Utils::is_lam_alef(event->keyval, res))
	{
	  doc->insert(res);
	  return true;
	}
      else
#endif
	return Gtk::Window::on_key_press_event(event);
    }

  if (event->length != 1)
    return Gtk::Window::on_key_press_event(event);

  gunichar x = gdk_keyval_to_unicode(event->keyval);
  if (!x)
    return Gtk::Window::on_key_press_event(event);

  Glib::ustring _key(1, x);

  std::string val, key(_key);

  // This cast is safe. We checked for < 0 above.
  if (get(static_cast<unsigned int>(_emulator), key, val))
    {
      doc->insert(val);
      return true;
    }
  else
    return Gtk::Window::on_key_press_event(event);
}

void Window::on_encoding_activated_cb(int e)
{
  int x;
  if(!mdi.set_encoding(e, x))
    menubar.set_encoding(x);
}

void Window::signal_document_readonly_cb(int x, bool r)
{
  menubar.save()->set_sensitive(!r);
  menubar.cut()->set_sensitive(!r);
  menubar.paste()->set_sensitive(!r);
  menubar.erase()->set_sensitive(!r);
  menubar.replace()->set_sensitive(!r);
  toolbar.save().set_sensitive(!r);
  toolbar.cut().set_sensitive(!r);
  toolbar.erase().set_sensitive(!r);
  toolbar.paste().set_sensitive(!r);

  menubar.document_set_readonly(x, r);
}

void Window::on_reset_gui(int x)
{
  bool enable = true;
  if (x == -1)
    enable = false;

  toolbar.save().set_sensitive(enable);
#ifdef ENABLE_PRINT
  toolbar.print().set_sensitive(enable);
#endif
  toolbar.close().set_sensitive(enable);
  toolbar.undo().set_sensitive(enable);
  toolbar.redo().set_sensitive(enable);
  toolbar.cut().set_sensitive(enable);
  toolbar.copy().set_sensitive(enable);
  toolbar.paste().set_sensitive(enable);
  toolbar.erase().set_sensitive(enable);
#ifdef HAVE_SPELL
  toolbar.dictionary().set_sensitive(enable);
#endif

  menubar.save()->set_sensitive(enable);
  menubar.save_as()->set_sensitive(enable);
  menubar.save_copy()->set_sensitive(enable);
  menubar.revert()->set_sensitive(enable);
  menubar.export_menu()->set_sensitive(enable);
#ifdef ENABLE_PRINT
  menubar.print()->set_sensitive(enable);
#endif
  menubar.close()->set_sensitive(enable);
  menubar.undo()->set_sensitive(enable);
  menubar.redo()->set_sensitive(enable);
  menubar.cut()->set_sensitive(enable);
  menubar.copy()->set_sensitive(enable);
  menubar.paste()->set_sensitive(enable);
  menubar.erase()->set_sensitive(enable);
  menubar.select_all()->set_sensitive(enable);
  menubar.find()->set_sensitive(enable);
  menubar.find_next()->set_sensitive(enable);
  menubar.replace()->set_sensitive(enable);
  menubar.goto_line()->set_sensitive(enable);
  menubar.line_numbers()->set_sensitive(enable);
  menubar.wrap_text()->set_sensitive(enable);
  menubar.encoding_menu()->set_sensitive(enable);
  menubar.execute()->set_sensitive(enable);
#ifdef HAVE_SPELL
  menubar.spell()->set_sensitive(enable);
  menubar.auto_spell()->set_sensitive(enable);
#endif
  menubar.emulator_menu()->set_sensitive(enable);
  menubar.save_all()->set_sensitive(enable);
  menubar.close_all()->set_sensitive(enable);

  statusbar.set_overwrite(false);
  set_title();

  // Extended toolbar.
#ifdef HAVE_SPELL
  toolbar.spell().set_sensitive(enable);
  toolbar.dictionary().set_sensitive(enable);
#endif
  toolbar.go_to().set_sensitive(enable);
#ifdef HAVE_SPELL
  toolbar.search().set_sensitive(enable);
#endif
  toolbar.extra_buttons().set_sensitive(enable);
}

void Window::signal_document_can_redo_cb(bool can)
{
  menubar.redo()->set_sensitive(can);
  toolbar.redo().set_sensitive(can);
}

void Window::signal_document_can_undo_cb(bool can)
{
  menubar.undo()->set_sensitive(can);
  toolbar.undo().set_sensitive(can);
}

void Window::signal_document_modified_cb(int x, bool m)
{
  statusbar.set_modified(m);
  menubar.document_set_modified(x, m);
}

void Window::signal_document_file_changed_cb(std::string f)
{
  // TODO: Do we need this ??
  //  set_title(const_cast<char *>(f.c_str()));
}

void Window::signal_document_encoding_changed_cb(int e)
{
  statusbar.set_encoding(_encodings.name(e));
  menubar.set_encoding(e);
}

#ifdef ENABLE_HIGHLIGHT
void Window::signal_document_highlight_cb(int x) {
  menubar.set_highlight(x);
}
#endif

void Window::signal_document_title_changed_cb(std::string t, int x)
{
  set_title(const_cast<char *>(t.c_str()));
  menubar.document_set_label(x, t);
}

void Window::on_document_added_cb(bool ro, bool m, std::string t)
{
  set_title(const_cast<char *>(t.c_str()));
  menubar.document_add(t, ro, m);
}

void Window::on_document_removed_cb(int x)
{
  menubar.document_remove(x);
}

void Window::on_doc_activated(int x)
{
  menubar.document_set_active(x);
}

void Window::signal_document_wrap_text_cb(bool w)
{
  wrap_text_conn.block();
  dynamic_cast<Gtk::CheckMenuItem *>(menubar.wrap_text())->set_active(w);
  wrap_text_conn.unblock();
}

void Window::signal_document_line_numbers_cb(bool ln)
{
  line_numbers_conn.block();
  dynamic_cast<Gtk::CheckMenuItem *>(menubar.line_numbers())->set_active(ln);
  line_numbers_conn.unblock();
}

void Window::search_cb(std::string s)
{
  mdi.find(s);
}

void Window::reset_gui()
{
  toolbar.reset_gui();
  menubar.reset_gui();
  statusbar.reset_gui();
  mdi.reset_gui();

  if (_conf.get("toolbar", true))
    toolbar.main().show();
  else
    toolbar.main().hide();

  if (_conf.get("extended_toolbar", true))
    toolbar.extended().show();
  else
    toolbar.extended().hide();

  if (_conf.get("statusbar", true))
    statusbar.show();
  else
    statusbar.hide();
  // NOTE: Should we enable this ??
  /*
    move(_conf.get("x", 50), _conf.get("y", 50));
    resize(_conf.get("w", 500), _conf.get("h", 400));
  */
}

#ifdef HAVE_SPELL
void Window::dictionary_cb(std::string d)
{
  std::string old;
  if (!mdi.set_dictionary(old, d))
    toolbar.set_dictionary(old);
}
#endif

#ifdef ENABLE_DBUS
void Window::open_files(std::vector<std::string>& f)
{
  for (unsigned int x = 0; x < f.size(); x++)
    mdi.document(f[x]);
  if (f.size() == 0)
    mdi.document();
  present();
}

#endif
