{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Language where

import List
import Debug.Trace

data Const = Num Int
	   | Ch Char
	   | Bo Bool
	   | Re Double
	   | Str String
	   | Exc String Int
	   | Empty
   deriving (Show, Eq)

data Name = UN String
	  | MN (String,Int)
	  | NS Name Name -- Name in a namespace
	  | None
   deriving Eq

type LongName = [Name]

instance Show Name where
    show (UN s) = "_D_" ++ quotename s
    show (NS s n) = "_ns_"++ show s ++ "_" ++ show n
    show (MN (s,i)) = "_my_" ++ quotename s ++ show i
    show None = "_DNAME__"

quotename [] = ""
quotename ('_':cs) = "_UN_"++quotename cs
quotename ('\'':cs) = "_PR_"++quotename cs
quotename ('?':cs) = "_QU_"++quotename cs
quotename ('':cs) = "_PO_"++quotename cs
quotename ('$':cs) = "_DO_"++quotename cs
quotename ('#':cs) = "_HA_"++quotename cs
quotename ('@':cs) = "_AT_"++quotename cs
quotename (c:cs) = c:(quotename cs)

showuser (UN s) = s
showuser (MN (s,i)) = "<"++show s ++ show i++">"
showuser (NS s n) = showuser s ++ "::" ++ showuser n
showuser None = error "Please don't do that"

data PrimType = Number
	      | Character
	      | Boolean
	      | RealNum
	      | StringType
	      | File
	      | Pointer
	      | Exception
	      | Void
   deriving Eq

instance Show PrimType where
  show Number = "Int"
  show Character = "Char"
  show Boolean = "Bool"
  show RealNum = "Float"
  show StringType = "String"
  show File = "File"
  show Pointer = "Ptr"
  show Exception = "Exception"
  show Void = "Void"

-- Return whether one type is "smaller" than another
-- X < Y if there is a (meaningful?) injection from X to Y.
-- There'll be a better way, if this table gets much bigger.
-- Should these be in a class PartialOrd?
tlt :: PrimType -> PrimType -> Bool
tlt Boolean Number = True
tlt Boolean StringType = True
tlt Character Number = True
tlt Character StringType = True
tlt Number RealNum = True
tlt Number StringType = True
tlt RealNum StringType = True
tlt _ _ = False

biggert :: Type -> Type -> Type
biggert (Prim x) (Prim y) | x `tlt` y = (Prim y)
biggert x y = x

data Type = Prim PrimType
	  | Fn [Maybe Raw] [Type] Type -- Defaults, arg types, return type
	  | Array Type
	  | User Name [Type] -- User type, parametrised over types
--	  | Syn Name -- Type synonym - rapidly expanded out by typechecker
	  | TyVar Name
	  | UnknownType -- We don't know what the type is yet.

instance Eq Type where
    (==) (Prim t1) (Prim t2) = t1==t2
    (==) (Fn _ ts t) (Fn _ ts2 t2) = ts==ts2 && t==t2
    (==) (Array t1) (Array t2) = t1==t2
    (==) (User n1 t1) (User n2 t2) = n1==n2 && t1==t2
--    (==) (Syn n1) (Syn n2) = n1==n2
    (==) (TyVar n1) (TyVar n2) = n1==n2
    (==) UnknownType UnknownType = True
    (==) _ _ = False

instance Show Type where
    show (Prim t) = show t
    show (Array t) = "[" ++ show t ++ "]"
    show (Fn ns ts t) = show t ++ "(" ++ showargs ts ++ ")"
	    where showargs [] = ""
		  showargs [x] = show x
		  showargs (x:xs) = show x ++ "," ++ showargs xs
    show (User n tys) = showuser n ++ showargs tys
	    where showargs [] = ""
		  showargs (x:xs) = "<" ++ sa' (x:xs) ++ ">"
		  sa' [] = ""
		  sa' [x] = show x
		  sa' (x:xs) = show x ++ "," ++ sa' xs
    show (TyVar (UN n)) = n
    show (TyVar (MN ("FV",n))) = "f" ++ show n
    show (TyVar (MN ("CLOS",n))) = "c" ++ show n
    show (TyVar (MN ("CLOSRET",n))) = "c" ++ show n
    show (TyVar (MN (_,n))) = "t" ++ show n
    show (TyVar n) = "<" ++ show n ++ ">"
--    show (Syn (UN n)) = n
    show (UnknownType) = "Unknown"

data InputType = Program
	       | Shebang
	       | Module
	       | Webapp
  deriving Show

-- Raw expressions
-- This really should be split up, at least so that things which syntactically
-- can't happen can't be represented. For example, RLambda can only occure
-- right at the top. But some other time.
data Raw = RVar String Int Name
	 | RQVar String Int Name -- Quoted (ie non evaluated) name
	 | RConst String Int Const
	 | RLambda String Int [Name] [(Name,Type)] Raw
	 | RClosure String Int [(Name,Type)] Raw
	 | RBind String Int Name Type Raw Raw
	 | RDeclare String Int Name Type Raw
	 | RReturn String Int Raw
	 | RVoidReturn String Int
	 | RAssign String Int RAssign Raw
	 | RAssignOp String Int Op RAssign Raw
	 | RSeq String Int Raw Raw
	 | RApply String Int Raw [Raw]
	 | RPartial String Int Raw [Raw]
	 | RForeign String Int Type Name [Raw]
	 | RWhile String Int Raw Raw
	 | RDoWhile String Int Raw Raw
	 | RFor String Int RAssign Raw Raw
	 | RTryCatch String Int Raw Raw Name Raw
	 | RThrow String Int Raw
	 | RExcept String Int Raw Raw
	 | RBreak String Int
	 | RPrint String Int Raw
	 | RInfix String Int Op Raw Raw 
	 | RUnary String Int UnOp Raw
	 | RCoerce String Int Type Raw
	 | RCase String Int Raw [RCaseAlt]
	 | RIf String Int Raw Raw Raw
	 | RIndex String Int Raw Raw
	 | RField String Int Raw Name
	 | RArrayInit String Int [Raw]
	 | RVMPtr String Int
	 | RNoop String Int
	 | RMetavar String Int Int
   deriving (Show, Eq)

data RCaseAlt = RAlt String Int Name [Name] Raw
   deriving (Show, Eq)

data RAssign = RAName String Int Name
	     | RAIndex String Int RAssign Raw
	     | RAField String Int RAssign Name
   deriving (Show, Eq)

-- Foreign declaration (doc string at the end)
data Foreign = ForeignDecl String Int [FOpt] Type 
                           [(Name,Type)] Name Name String

data Op = Plus | Minus | Times | Divide | Modulo | Power | Equal | NEqual |
	  OpLT | OpGT | OpLE | OpGE | OpAnd | OpOr | OpXOR | BAnd | BOr |
	  OpShLeft | OpShRight | OpAndBool | OpOrBool
   deriving Eq

instance Show Op where
    show Plus = "+"
    show Minus = "-"
    show Times = "*"
    show Divide = "/"
    show Modulo = "%"
    show Power = "**"
    show Equal = "=="
    show NEqual = "!="
    show OpLT = "<"
    show OpGT = ">"
    show OpLE = "<="
    show OpGE = ">="
    show OpAnd = "&"
    show OpOr = "|"
    show OpXOR = "^"
    show BAnd = "&&" -- FIXME: BAnd seems unnecessary
    show BOr = "||" -- FIXME: BOr seems unnecessary
    show OpShLeft = "<<"
    show OpShRight = ">>"
    show OpAndBool = "&&"
    show OpOrBool = "||"

boolops = [Equal, NEqual, OpLT, OpGT, OpLE, OpGE, BAnd, BOr]

data UnOp = Not | Neg
   deriving Eq

instance Show UnOp where
    show Not = "!"
    show Neg = "-"

{-
data RGlob = RGlob [RGDecl]
   deriving Show

data RGDecl = RGDecl Type Name
   deriving Show
-}

-- Type checked expressions
-- As for Raw expressions, this really ought to be split up.
data Expr n = Global n -- Global function definition
	    | Loc Int -- Local variable
	    | GVar Int -- Module global variable
 	    | GConst Const
	    | Lambda [(n,Type)] (Expr n)
	    | Closure [(n,Type)] Type (Expr n) -- remember return type
	    | Bind n Type (Expr n) (Expr n)
	    | Declare String Int n Type (Expr n)
	    | Return (Expr n)
	    | VoidReturn
	    | Assign (Assign n) (Expr n)
	    | AssignOp Op (Assign n) (Expr n)
	    | Seq (Expr n) (Expr n)
	    | Apply (Expr n) [Expr n]
	    | Partial (Expr n) [Expr n] Int
	    | Foreign Type n [(Expr n,Type)]
	    | While (Expr n) (Expr n)
	    | DoWhile (Expr n) (Expr n)
	      -- ints here are for counting through the loop
	    | For Int Int (Assign n) (Expr n) (Expr n) 
	    | TryCatch (Expr n) (Expr n) (Expr n) (Expr n)
	    | Throw (Expr n)
	    | Except (Expr n) (Expr n)
	    | Break String Int
	    | InferPrint (Expr n) Type String Int
	    | PrintNum (Expr n)
	    | PrintStr (Expr n)
	    | PrintExc (Expr n)
	    | Infix Op (Expr n) (Expr n)
	    | RealInfix Op (Expr n) (Expr n)
	    | InferInfix Op (Expr n) (Expr n) (Type,Type,Type) String Int
	    | CmpExcept Op (Expr n) (Expr n)
	    | CmpStr Op (Expr n) (Expr n)
	    | Append (Expr n) (Expr n)
	    | Unary UnOp (Expr n)
	    | RealUnary UnOp (Expr n)
	    | InferUnary UnOp (Expr n) (Type,Type) String Int
	    | Coerce Type Type (Expr n)
	    | InferCoerce Type Type (Expr n) String Int
	    | Case (Expr n) [CaseAlt n]
	    | If (Expr n) (Expr n) (Expr n)
	    | Index (Expr n) (Expr n)
	    | Field (Expr n) Name Int Int -- name, argument and tag
	    | ArrayInit [Expr n]
	    | VMPtr
	    | Error String
	    | Noop
	    | Metavar String Int Int -- Hole for optimisations
  deriving (Show, Eq)

data CaseAlt n = Alt Int Int [Expr n] (Expr n) -- tag, total
   deriving (Show, Eq)

instance Eq n => Ord (CaseAlt n) where
    compare (Alt x _ _ _) (Alt y _ _ _) = compare x y

data Assign n = AName Int -- Local name
	      | AGlob Int -- Module global name
	      | AIndex (Assign n) (Expr n)
	      | AField (Assign n) Name Int Int -- Name, argument and tag
  deriving (Show, Eq)

{-
data Glob n = Glob [GDecl n]
   deriving Show

data GDecl n = GDecl Type n
   deriving Show
-}

-- Function bindings
--type FBind = (Name, Type, Expr Name, [FOpt])

data FOpt = Public | Pure | NoArgs | Inline | Export | Generated
	  | DefaultDef
  deriving (Show, Eq)

data DOpt = DPublic | DAbstract | DExport
  deriving (Show, Eq)

-- *BOGGLE* Why do I have to do this?
-- Come to think of it, why am I even using a tuple?
instance (Show a, Show b, Show c, Show d, Show e, Show f) => Show (a,b,c,d,e,f)
    where show (a,b,c,d,e,f) = "(" ++ show a ++ "," ++
			       show b ++ "," ++
			       show c ++ "," ++
			       show d ++ "," ++
			       show e ++ "," ++
			       show f ++ ")"

data Decl = FunBind (String, Int, Name, Type, [FOpt], Binder (Expr Name)) String
	  | Glob (Name, Type, Int)
	  | Imported String
	  | Linker String
	  | TySyn (Name, [Name], Type)
	  | CInclude String
	  | DataDecl String Int [DOpt] Name [Type] [ConDecl] String
	  | AbstractDecl Name [Type]
  deriving Show

instance Eq Decl where
    (==) (FunBind (_,_,x1,t1,_,b1) _) (FunBind (_,_,x2,t2,_,b2) _) 
	= (x1==x2) && (t1==t2)
--    (==) (FunBind (_,_,x1,t1,_,b1)) (FunBind (_,_,x2,t2,_,b2)) 
--	= (x1==x2) && (t1==t2) && (b1==b2)
    (==) (Glob x) (Glob y) = x==y
    (==) (Imported x) (Imported y) = x==y
    (==) (Linker x) (Linker y) = x==y
    (==) (TySyn x) (TySyn y) = x==y
    (==) (CInclude x) (CInclude y) = x==y
    (==) (DataDecl _ _ _ n ty c _) (DataDecl _ _ _ n2 ty2 c2 _) 
	= (n==n2) && (ty==ty2) && (c==c2)
    (==) (AbstractDecl n ty) (AbstractDecl n2 ty2) = (n==n2) && (ty==ty2)
    (==) _ _ = False

-- Programs
type Program = [Decl]

getFunID :: Name -> Program -> Int
getFunID fn ds = gf' 0 ds
 where gf' _ [] = -1
       gf' i ((FunBind (_,_,n,_,_,_) _):xs) | fn == n = i
       gf' i (x:xs) = gf' (i+1) xs

data Binder a = Defined a  -- Defined function
	      | DataCon Int Int Bool -- Tagged constructor, with arity
		   -- Bool says whether or not to compile code for it
	      | Unbound -- Not bound/defined elsewhere
  deriving (Show, Eq)

data ParseResult = PR { inputtype :: InputType,
			parsemodulename :: Name,
			parsedefinitions :: [RawDecl] }
  deriving Show

addToPT :: Result ParseResult -> Result [RawDecl] -> Result ParseResult
addToPT (Success (PR i n rs)) (Success rs') = Success (PR i n (rs'++rs))
addToPT f _ = f

data RawDecl = FB (String,Int,Name,Type,[FOpt], Binder Raw) String
	     | GlobDecl String Int (Name, Type)
	     | CInc String
	     | Imp String
	     | SearchImport String -- First pass only, when chasing modules
	     | Link String
	     | TSyn (String,Int,Name,[Type],Type)
	     | DDecl String Int [DOpt] Name [Type] [ConDecl] String
	     | ADecl Name [Type]
	     | CDecl Name Type Int Int
  deriving Show

-- A constructor declaration is a name and its type --- data constructors
-- are a special kind of function.
-- The list of names is the list of field names.
data ConDecl = Con Name Type [Name] Bool -- Bool is whether to compile code
  deriving (Show, Eq)

getConName (Con n _ _ _) = n

-- Local contexts
type Context = [(Name,(Type,[FOpt]))]

-- Lookup in the context (ignores namespace if no ambiguity, returns
-- fully qualified name)
ctxtlookup :: Monad m => Name -> -- Current module
	                 Name -> Context -> m (Name, Type)
ctxtlookup mod n ctx = do (okn,okt,pub) <- cl' n ctx Nothing
			  if pub then return (okn,okt)
			    else fail $ "Can't use private " ++ showuser okn
 where cl' n [] (Just v) = return v
       cl' n [] Nothing = fail $ "Unknown name ctxt " ++ showuser n
       cl' n ((cn@(NS ns bn),(ty,opts)):xs) (Just (on,_,True)) 
	   | n == bn && on /= cn && (elem Public opts || ns==mod)
            = fail $ "Ambiguous name " ++ showuser n ++ " - " ++ showuser cn ++ " or " ++ showuser on ++ "?"
       cl' n ((cn@(NS ns bn),(ty,opts)):xs) Nothing -- Fully qualified n
	   | n == cn && (elem Public opts || ns==mod) 
	       = cl' n xs (Just (cn,ty,True))
	   | n == cn = cl' n xs (Just (cn,ty,False))
       -- Got a private name, override it with a public one
       cl' n ((cn@(NS ns bn),(ty,opts)):xs) (Just (_,_,False))
	   | n == bn && (elem Public opts || ns==mod) 
	       = cl' n xs (Just (cn,ty,True))
       -- Got no name yet, so take what we've got
       cl' n ((cn@(NS ns bn),(ty,opts)):xs) Nothing
	   | n == bn && (elem Public opts || ns==mod) 
	       = cl' n xs (Just (cn,ty,True))
	   | n == bn = cl' n xs (Just (cn,ty,False))
       cl' n (_:xs) j = cl' n xs j

ctxtdump :: Context -> String
ctxtdump [] = ""
ctxtdump ((n,(ty,_)):xs) = showuser n ++ " :: " ++ show ty ++ "\n" ++ ctxtdump xs

-- Global variables (Int is an id to refer to it by)
type GContext = [(Name,(Type,Int))]

-- Defined types
type Types = [(Name,TypeInfo)]
data TypeInfo = UserData [Type] -- User defined type
	      | Syn [Name] Type -- Type synonym
	      | Abstract -- Abstract type
	      | Private -- Private type
       deriving (Show, Eq)

-- Field names and which data type they belong to, 
-- mapping to the field type, which argument it projects to, and the tag.
type Fields = [((Name, Type), (Type, Int, Int))]

-- Constructor tags
type Tags = [(Name,(Int,Int))] -- map from constructor name to tag,number of
                               -- constructors in that type.

data Result r = Success r
              | Failure String String Int
    deriving (Show, Eq)

instance Monad Result where
    (Success r)   >>= k = k r
    (Failure err fn line) >>= k = Failure err fn line
    return              = Success
    fail s              = Failure s "(no file)" 0

------------ Gadgets -------------

convert :: Type -> Type -> Bool
convert = (==)

checkConv :: Monad m => Type -> Type -> String -> m ()
checkConv x y err = if convert x y 
		     then return ()
		     else fail err

getType :: Monad m => Name -> [(Name,b)] -> m b
getType n ctxt = case (lookup n ctxt) of
		    Nothing -> fail $ "Unknown name gettype " ++ show n
		    (Just t) -> return t

getVars :: Type -> [Name]
getVars = nub.gv
    where gv (Fn ns ts t) = concat (map gv (t:ts))
	  gv (Array t) = gv t
	  gv (User n ts) = concat (map gv ts)
	  gv (TyVar n) = [n]
	  gv _ = []

fudgevars :: Type -> Int -> (Type, Int)
fudgevars t next = let (vsmap,next') = newnames next (getUserVars t) in
		       (alpha vsmap t, next')
    where newnames n [] = ([],n)
	  newnames n (x:xs) = let (xsih,n') = newnames (n+1) xs in
				  ((x,MN ("TV",n)):xsih, n')
	  alpha vsmap (Prim x) = Prim x
	  alpha vsmap (Fn ns tys t) = Fn ns (map (alpha vsmap) tys)
				            (alpha vsmap t)
	  alpha vsmap (Array t) = Array (alpha vsmap t)
	  alpha vsmap (User n tys) = User n (map (alpha vsmap) tys)
--	  alpha vsmap (Syn n) = Syn n
	  alpha vsmap (TyVar x) = case lookup x vsmap of
				     Nothing -> TyVar x
				     (Just v) -> TyVar v
	  alpha vsmap UnknownType = UnknownType

getUserVars :: Type -> [Name]
getUserVars = nub.gv
    where gv (Fn ns ts t) = concat (map gv (t:ts))
	  gv (Array t) = gv t
	  gv (User n ts) = concat (map gv ts)
	  gv (TyVar (UN n)) = [UN n]
	  gv _ = []

lvaltoexp :: RAssign -> Raw
lvaltoexp (RAName f l n) = RVar f l n
lvaltoexp (RAIndex f l lv r) = RIndex f l (lvaltoexp lv) r
lvaltoexp (RAField f l lv r) = RField f l (lvaltoexp lv) r

showconst (Num x) = show x
showconst (Ch '\0') = "'\\0'"
showconst (Ch c) = show c
showconst (Bo True) = "true"
showconst (Bo False) = "false"
showconst (Str str) = show str
showconst (Exc str i) = "exception("++show str++","++show i++")"
showconst (Empty) = error "Can't show an empty constant"

-- Type normalisation; expand synonyms.

normalise :: Monad m => String -> Int -> Types -> Type -> m Type
normalise f l ti t = tn [] t
 where
   tn u (Fn ds ts t) 
       = do ts' <- mapM (tn u) ts
	    t' <- tn u t
	    return $ Fn ds ts' t'
   tn u (Array t) = do t' <- tn u t
		       return $ Array t'
   tn u t@(User n ts) = 
       case lookup n ti of
          Nothing -> fail $ f ++ ":" ++ show l ++ ":" ++
		       "Unknown type " ++ show t
	  (Just x) -> applyTI u n ts x
   tn u rest = return rest

   applyTI u n ts (UserData as)
	| length ts < length as 
	    = fail $ f ++ ":" ++ show l ++ ":" ++
	       "Type " ++ showuser n ++ " has too few parameters"
	| length ts > length as 
	    = fail $ f ++ ":" ++ show l ++ ":" ++
	       "Type " ++ showuser n ++ " has too many parameters"
	| otherwise = do ts' <- mapM (tn u) ts
			 return $ User n ts'
   -- Replace type with 't', replacing instances of as inside t with
   -- corresponding instances of ts.
   -- That probably makes no sense.
   applyTI u n ts (Syn as t)
	| length ts < length as
	    = fail $ f ++ ":" ++ show l ++ ":" ++
	       "Type synonym " ++ showuser n ++ " has too few parameters"
	| length ts > length as
	    = fail $ f ++ ":" ++ show l ++ ":" ++
	       "Type synonym " ++ showuser n ++ " has too many parameters"
	| otherwise = if elem n u 
		       then fail $ f ++ ":" ++ show l ++ ":" ++
			     "Cycle in type synonyms " ++ 
			     showsyns u
		       else do st <- substty (zip as ts) t
			       tn (n:u) st
     where showsyns [n] = showuser n
	   showsyns (n:ns) = showuser n ++ ", " ++ showsyns ns

   applyTI u n ts Private = fail $ f ++ ":" ++ show l ++ 
			     "Can't use private type " ++ showuser n

   substty tmap (TyVar n) = case lookup n tmap of
			      Nothing -> fail $ "Shouldn't happen" ++ show tmap
			      (Just t) -> return t
   substty tmap (Array t) = do t' <- substty tmap t
			       return $ Array t'
   substty tmap (Fn ds as r) = do as' <- mapM (substty tmap) as
				  r' <- substty tmap r
				  return $ Fn ds as' r'
   substty tmap (User n ts) = do ts' <- mapM (substty tmap) ts
				 return $ User n ts'
   substty _ rest = return rest

-- Fold constants in a raw term
-- TODO/FIXME: Check bounds?
cfold :: Raw -> Raw
cfold r@(RInfix f l op (RConst _ _ (Num x)) (RConst _ _ (Num y)))
    = case (foldint op x y) of
          Just c -> RConst f l c
	  Nothing -> r
cfold r@(RInfix f l op (RConst _ _ (Re x)) (RConst _ _ (Re y)))
    = case (foldreal op x y) of
          Just c -> RConst f l c
	  Nothing -> r
cfold r@(RUnary f l op (RConst _ _ (Num x)))
    = case (foldunint op x) of
          Just c -> RConst f l c
	  Nothing -> r
cfold r@(RUnary f l op (RConst _ _ (Re x)))
    = case (foldunreal op x) of
          Just c -> RConst f l c
	  Nothing -> r
cfold r = r

foldint :: Op -> Int -> Int -> Maybe Const
foldint Plus x y = Just $ Num (x+y)
foldint Minus x y = Just $ Num (x-y)
foldint Times x y = Just $ Num (x*y)
foldint Divide x y = Just $ Num (x `div` y)
foldint Modulo x y = Just $ Num (x `mod` y)
foldint Power x y = Just $ Num (floor ((fromIntegral x)**(fromIntegral y)))
foldint Equal x y = Just $ Bo (x==y)
foldint NEqual x y = Just $ Bo (x/=y)
foldint OpLT x y = Just $ Bo (x<y)
foldint OpGT x y = Just $ Bo (x>y)
foldint OpLE x y = Just $ Bo (x<=y)
foldint OpGE x y = Just $ Bo (x>=y)
foldint _ x y = Nothing 

foldreal :: Op -> Double -> Double -> Maybe Const
foldreal Plus x y = Just $ Re (x+y)
foldreal Minus x y = Just $ Re (x-y)
foldreal Times x y = Just $ Re (x*y)
foldreal Divide x y = Just $ Re (x/y)
foldreal Power x y = Just $ Re (x**y)
foldreal Equal x y = Just $ Bo (x==y)
foldreal NEqual x y = Just $ Bo (x/=y)
foldreal OpLT x y = Just $ Bo (x<y)
foldreal OpGT x y = Just $ Bo (x>y)
foldreal OpLE x y = Just $ Bo (x<=y)
foldreal OpGE x y = Just $ Bo (x>=y)
foldreal _ x y = Nothing 

foldunint :: UnOp -> Int -> Maybe Const
foldunint Neg x = Just $ Num (-x)
foldunint _ _ = Nothing

foldunreal :: UnOp -> Double -> Maybe Const
foldunreal Neg x = Just $ Re (-x)
foldunreal _ _ = Nothing

-- Apply a function (non-recursively) to every sub expression,
-- applying a different function to metavariables
-- (I don't know if this is actually that useful, but it is used by the 
-- optimiser...)
mapsubexpr f mf expr = app expr
  where app (Metavar fl l x) = mf fl l x
        app (Lambda args e) = Lambda args (f e)
	app (Closure args t e) = Closure args t (f e)
	app (Bind n ty e1 e2) = Bind n ty (f e1) (f e2)
	app (Declare fn l n t e) = Declare fn l n t (f e)
	app (Assign a e) = Assign (aapply a) (f e)
	app (AssignOp op a e) = AssignOp op (aapply a) (f e)
	app (Seq a b) = Seq (f a) (f b)
	app (Apply fn as) = Apply (f fn) (applys as)
	app (Partial fn as i) = Partial (f fn) (applys as) i
	app (Foreign ty n es) = Foreign ty n 
			        (zip (applys (map fst es)) (map snd es))
	app (While e b) = While (f e) (f b)
	app (DoWhile e b) = DoWhile (f e) (f b)
	app (For i j a e1 e2) = For i j (aapply a) (f e1) (f e2)
	app (TryCatch t e fl fin) = TryCatch (f t) (f e) (f fl) (f fin)
	app (Throw e) = Throw (f e)
	app (Except e i) = Except (f e) (f i)
	app (Infix op x y) = Infix op (f x) (f y)
	app (RealInfix op x y) = RealInfix op (f x) (f y)
	app (Append x y) = Append (f x) (f y)
	app (Unary op x) = Unary op (f x)
	app (RealUnary op x) = RealUnary op (f x)
	app (Coerce t1 t2 x) = Coerce t1 t2 (f x)
	app (Case e as) = Case (f e) (altapp as)
	app (If a t e) = If (f a) (f t) (f e)
	app (Index a b) = Index (f a) (f b)
	app (Field e n i j) = Field (f e) n i j
	app (ArrayInit as) = ArrayInit (applys as)
	app x = x

        aapply (AIndex a e) = AIndex (aapply a) (f e)
	aapply (AField a n i j) = AField (aapply a) n i j
	aapply x = x

        applys [] = []
	applys (x:xs) = (f x) : (applys xs)

        altapp [] = []
	altapp ((Alt i j es e):as) 
	    = (Alt i j (applys es) (f e)):(altapp as)
