/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "KayaAPI.h"
#include "Heap.h"
#include "Array.h"
#include "VM.h"
#include "VMState.h"

#include <stdarg.h>

KayaValue newKayaValue()
{
    return new Value(NULL,NULL);
}

KayaValue KayaInt(int x)
{
    KayaValue v = newKayaValue();
    KayaSetInt(v,x);
    return v;
}

KayaValue KayaChar(char c)
{
    KayaValue v = newKayaValue();
    KayaSetChar(v,c);
    return v;
}

KayaValue KayaFloat(float f)
{
    KayaValue v = newKayaValue();
    KayaSetFloat(v,f);
    return v;
}

KayaValue KayaString(char* str)
{
    KayaValue v = newKayaValue();
    KayaSetString(v,str);
    return v;
}

int KayaGetInt(KayaValue v)
{
    return v->getInt();
}

char KayaGetChar(KayaValue v)
{
    return (char)(v->getInt());
}

double KayaGetFloat(KayaValue v)
{
    return v->getReal();
}

char* KayaGetString(KayaValue v)
{
    return v->getString()->getVal();
}

void KayaSetInt(KayaValue v,int i)
{
    v->setInt(i);
}

void KayaSetChar(KayaValue v,char c)
{
    v->setInt((int)c);
}

void KayaSetFloat(KayaValue v,double f)
{
    v->setReal(f);
}

void KayaSetString(KayaValue v,char* s)
{
    v->setString(new String(s));
}


KayaArray newKayaArray(int size)
{
    return new Array(size);
}

KayaArray KayaGetArray(KayaValue v)
{
    return v->getArray();
}

void KayaSetArray(KayaValue v, KayaArray a)
{
    v->setArray(a);
}


void KayaArrayPush(KayaArray a,KayaValue v)
{
    a->push_back(v);
}

void KayaArraySet(KayaArray a, int index, KayaValue v)
{
    Value* vloc = a->lookup(index);
    vloc->setPtr(v);
}

KayaValue KayaArrayLookup(KayaArray a, int index)
{
    return a->lookup(index);
}

int KayaArraySize(KayaArray a)
{
    return a->size();
}

KayaValue KayaCall(KayaValue fn, int args, ...)
{
    VMState* vm = initstack();
//    Value* arg;
    Value** arglist = new Value*[args];
    va_list argp;
    va_start(argp,args);
    int i;
    // Gah! Need to push backwards!
    for(i=0;i<args;i++) {
	arglist[args-i-1] = va_arg(argp,Value*);
    }
    va_end(argp);
    for(i=0;i<args;i++) {
	PUSH(arglist[i]);
    }
    delete arglist;
    TRY(cbexcept);
    CALLFUN(fn);
    if (vm->emptyStack()) {
	return NULL;
    }
    else
	return vm->doPop();

    LABEL(cbexcept); // There was an exception in the callback. This is fatal.
    cerr << "Exception in callback: ";
    Exception* except = vm->doPop()->getExcept();
    except->show();
    exit(1);
}

KayaValue KayaUnion(int tag, int args)
{
    Union* u = new Union(NULL,tag,args,false);
    return new Value((void*)u,uniontable);
}

int KayaUnionGetTag(KayaValue v)
{
    Union* u = v->getUnion();
    return u->tag;
}

KayaValue KayaUnionGetArg(KayaValue v,int i)
{
    Union* u = v->getUnion();
    return u->args[i];
}

void KayaUnionSetArg(KayaValue v,int i,KayaValue val)
{
    Union* u = v->getUnion();
    u->args[i] = val;
}

void* KayaAlloc(int size)
{
    // Just use libgc...
    return GC_MALLOC_ATOMIC(size);
}

int orArray(KayaArray a, int(*func)(KayaValue))
{
    int val = 0;
    int size = KayaArraySize(a);
    for(int i=0;i<size;i++) {
	KayaValue v = KayaArrayLookup(a,i);
	val = val | func(v);
    }
    return val;
}
