/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "Array.h"
#include <gc/gc_allocator.h>

#include <iostream>
#include "Heap.h"
#include "ValueFuns.h"
#include "stdfuns.h"

#define ALLOC 1024

Array::Array(int size):m_size(0),m_first(NULL)
{
    if (size==0) { 
	size = ALLOC;
    }

    m_first = new chunk();
    m_first -> data = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*size);

    m_first -> size = size;
    m_first -> next = NULL;

}

Array::~Array() {
    chunk* c = m_first;
    
    // Delete all the chunks data blocks.
    while (c!=NULL) {
	GC_FREE(c->data);
	c = c->next;
    }
//    cout << "Array destructor called" << endl;
}

void Array::expectedSize(int size)
{
    if (size<=m_size) return; // Already big enough

    chunk* c = m_first;
    chunk* last = NULL;
    
    // Find the last chunk, then make a chunk big enough for <size>.

    while (c!=NULL) {
	last = c;
	c = c->next;
    }
    
    c = new chunk();
    last->next = c;
    int alloc = size;
//    cout << " New chunk, size " << alloc << " at " << c << endl;
    c -> data = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*alloc);
    c -> size = alloc;
    c -> next = NULL;
}

void Array::resize(int i)
{
//     int oldsize=m_size;
    if (i<m_size) {
	m_size=i;
    }
    else {
	// This is slow. Better to just set the size, and remember how
	// much is allocated. But this'll do for now.
	while(m_size<i) {
	    static Value* empty = new Value(NULL,KVT_NULL);
	    push_back(empty);
	}
    }
// Perhaps delete chunks we no longer need here?

}

Value* Array::shift()
{
    chunk* c = m_first;
    Value* shiftval = *(c->data);

    // Move all the values down one.
    while (c!=NULL) {
	memmove(c->data,(c->data)+1,(c->size-1)*sizeof(Value*));
	if (c->next!=NULL) {
	    c->data[(c->size)-1] = c->next->data[0];
	}
	c = c->next;
    }
    m_size--;
    return shiftval;
}

int Array::reserved_size() {
  chunk* c = m_first;
  int sz = 0;
  while (c!=NULL) {
      sz += c->size;
      c = c->next;
  }
  return sz;
}

void Array::push_back(Value *v)
{
    chunk* c = m_first;
    chunk* last = NULL;
    int i = m_size;
    while (c!=NULL) {
	if (i<c->size) {
	    break;
	} else {
	    i -= c->size;
	    last = c;
	    c = c->next;
	}
    }
    // If c is NULL make a new chunk.
    if (c==NULL) {
	c = new chunk();
	last->next = c;
	// Always double the size of the last chunk. Not sure how often
	// this is a good strategy...
	int alloc = last->size*2;
//	cout << " New chunk, size " << alloc << " at " << c << endl;
	c -> data = (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*alloc);
	c -> size = alloc;
//	cout << c->size << endl;
	c -> next = NULL;
	i=0;
    }
    c->data[i] = v;
    ++m_size;

}

/*
void append(Array* xs)
{
not implemented
}
*/

Value* Array::lookup(int i)
{
    chunk* c = m_first;
    register int x;
    while (c) {
	x = c->size;
	if (i<x) {
	    return *(c->data+i);
	} else {
	    i=i-x;
	    c = c->next;
	}
    }
    return NULL; // Out of bounds.
}

// TODO: These are too slow, since we can peek directly rather than lookup.
bool Array::eq(Array* x, map<Value*, Value*>& done)
{
    if (x->size()!=size()) return false;
    for(int i=0;i<m_size;++i) {
	Value* idx = lookup(i);
	Value* xidx = x->lookup(i);
	if (!funtable_eq_aux(idx,xidx,done)) return false;
    }
    return true;
}

int Array::cmp(Array* x)
{
    if (x->size()<size()) return -1;
    if (x->size()>size()) return 1;
    for(int i=0;i<m_size;++i) {
	Value* idx = lookup(i);
	Value* xidx = x->lookup(i);
	int cmp = funtable_compare(NULL,idx,xidx);
	if (cmp!=0) return cmp;
    }
    return 0;
}

Array* Array::copy(map<Value*,Value*>& done)
{
    Array* a = new Array(m_size);
    for(int i=0;i<m_size;++i) {
	Value* copied;
	Value* x = lookup(i);
	// if x is already copied, return the copied value
	if (done.find(x)!=done.end()) {
	    copied = done[x];
	}
	else {
	    copied = funtable_copy_aux(x, done);
	    done[x] = copied;
	}
	a->push_back(copied);
    }
    return a;
}

int Array::memusage()
{
    chunk* c = m_first;
    int size = sizeof(Array);
    int x;
    while(c!=NULL) {
	size+=sizeof(chunk);
	size+=c->size*sizeof(Value*);
	for(x=0;x<c->size;++x) {
	    if (c->data[x]!=NULL)
		size+=funtable_memusage(c->data[x]);
	}
	c = c->next;
    }
    return size;
}
