/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#ifndef WIN32
#include <sys/wait.h>
#include <sys/resource.h> 
#else
// CIM: for unlink and mkdir
#include <io.h> 
#include <windows.h>
#endif
// CIM: we now need this in mingw too
#include <sys/stat.h>
#include <iostream>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <wchar.h>
#include <time.h>
#include <sys/types.h>
#include <sys/time.h> 
#include <unistd.h>
#include <cstdlib>
#include <vector>
//#include <malloc.h>

// CIM: pthread.h is outside the thread definitions since we need it anyway
// for the gmtime_r functions.
#include <pthread.h>
#ifndef WIN32
#define GC_PTHREADS
#else
#define GC_WIN32_THREADS
#endif
#include <gc/gc.h>

#include "stdfuns.h"
#include "Heap.h"
#include "Closure.h"
#include "VM.h"
#include "VMState.h"
#include "KayaAPI.h"

// Read lines from files this many characters at a time
#define LINE_CHUNK 1024

KayaArray args;
ArgMap httpargs;

wchar_t* inttostr(int s, int base)
{
  // CIM: I'm sure we can make this smaller
    int sz = 255;
    wchar_t* buf=(wchar_t*)GC_MALLOC_ATOMIC(sz*sizeof(wchar_t));
    switch(base) {
    case 10:
	SWPRINTF(buf,sz,L"%d",s);
	break;
    case 16:
	SWPRINTF(buf,sz,L"%x",s);
	break;
    case 8:
	SWPRINTF(buf,sz,L"%o",s);
	break;
    default:
	SWPRINTF(buf,sz,L"%d",s); // Can't do weird bases yet.
	break;
    }
    return buf;
}

int strtoint(wchar_t* str, int base)
{
    return wcstol(str,NULL,base);
}

wchar_t* getsubstr(wchar_t* x,int i,int len)
{
    wchar_t* n = (wchar_t*)GC_MALLOC_ATOMIC((len+1)*sizeof(wchar_t));
    if (len > 0) {
        wcsncpy(n,x+i,len);
    }
    n[len] = '\0';
    return n;
}

void str_offset(Value* str, int inc)
{
    str->getString()->offset(inc);
}

void str_chop(Value* str, int inc)
{
    str->getString()->chop(inc);
}

wchar_t* getstrend(wchar_t* x,int i)
{
    int max = wcslen(x);
    wchar_t* n = (wchar_t*)GC_MALLOC_ATOMIC((max+1-i)*sizeof(wchar_t));
    if (max-i > 0) {
      wcsncpy(n,x+i,max-i);
    }
    n[max-i] = '\0';
    return n;
}
wchar_t getIndex(wchar_t* str,int i)
{
    return str[i];
}

void setIndex(wchar_t* str, wchar_t c,int i)
{
    str[i] = c;
}

wchar_t* getLine(FILE* f)
{
    unsigned size = LINE_CHUNK; // #define at top of file
    // buffer to read into
    char* buf=(char*)GC_MALLOC_ATOMIC(LINE_CHUNK*sizeof(char)+1); 
    // accumulator for result; but only allocate it when it turns out to be
    // needed (i.e., when one chunk isn't enough).
    char* res=NULL;
    char* r;
    do {
	r = fgets(buf,LINE_CHUNK,f);
	if (r == NULL) { // End of file, or read error. FIXME: Check which.
	    if (res==NULL)
	    {
		// Fail silently (should this throw an exception?)
		return emptyString(); 
	    } else {
		// File finished without a newline, return what we've got.
		return strtowc(res);
	    }
	}
	if (res==NULL) {
	    // FIXME: We're relying on \n or \r\n line endings. How much
	    // can we rely on this?
	    // C defines \n to be the native newline when reading
	    // or writing files in text mode so maybe we're okay...
	    if (strlen(buf)<(size-1) || buf[LINE_CHUNK-2]=='\n') {
		return strtowc(buf); // Success in one chunk
	    }
	    // Begin allocation for accumulator and carry on as normal.
	    size += LINE_CHUNK-1;
	    res = (char*)GC_MALLOC_ATOMIC(size*sizeof(char)+1);
	    strcpy(res,buf);
	} 
	else {
	    strcat(res,buf);
	    if (strlen(res)<(size-1) || buf[LINE_CHUNK-2]=='\n') { 
                // Finished successfully
		return strtowc(res); 
	    } else {
		size += LINE_CHUNK-1; // -1 to account for null termination of
		                      // string read into buf
		res = (char*)GC_REALLOC(res, size*sizeof(char)+1);
	    }
	}
    } while (r!=NULL);
    return emptyString(); // Failed, although we shouldn't get here
}

wchar_t* getString(FILE* f)
{
    unsigned size = LINE_CHUNK; // #define at top of file
    // buffer to read into
    char* buf=(char*)GC_MALLOC_ATOMIC(LINE_CHUNK*sizeof(char)+1); 
    // accumulator for result; but only allocate it when it turns out to be
    // needed (i.e., when one chunk isn't enough).
    int i=0;
    do {
      
      do {
	int c = fgetc(f);
	if (c == EOF) {
	  buf[i] = '\0';
	  return strtowc(buf);
	} 
	buf[i] = (unsigned char)c;
	if (buf[i] == '\0') {
	  return strtowc(buf);
	}
	i++;
      } while (i%LINE_CHUNK != 0);
      size += LINE_CHUNK;
      buf = (char*)GC_REALLOC(buf, size*sizeof(char)+1);

    } while (buf[i] != EOF);
    return emptyString(); // Failed, although we shouldn't get here
}

void putLine(FILE* f, wchar_t* c)
{
    fputs(wctostr(c),f);
}

void putString(FILE* f, wchar_t* c)
{
    fputs(wctostr(c),f);
    fputc('\0',f);
}

void do_fseek(FILE* f, kint p) {
    fseek(f,(long)p,SEEK_SET);
}

kint do_ftell(FILE* f) {
  return KINT(ftell(f));
}

void putStr(wchar_t* rawc)
{
    unsigned int len = 1+wcslen(rawc);
    int i = sizeof(char)*(len)*4;
    char c[i];
    mkUTF8(rawc,c,len);
    cout << c;
}

// Check if the top bit is set in a string (i.e., whether it is ASCII or
// includes UTF-8 characters)
int topSet(const char* str) {
    const char* sptr = str;
    for(;(*sptr)!=0;++sptr) {
	if (((*sptr)&128)!=0) {
	    return 1;
	}
    }
    return 0;
}

int checkUTF8(wchar_t* str)
{
    return(topSet(CSTRING(str)));
}

void printArray(KayaArray foo)
{
    for(int i=0;i<foo->size();++i) {
	cout << wctostr(foo->lookup(i)->getString()->getVal()) << endl;
    }
}

KayaArray reverseArray(KayaArray foo)
{
    KayaArray bar = new Array();
    for(unsigned i=foo->size();i>0;i--) {
	bar->push_back(foo->lookup(i-1));
    }
    return bar;
}

void shortenArray(KayaArray foo)
{
    if (foo->size()>0) {
	foo->resize(foo->size()-1);
    }
}

KayaValue shiftArray(KayaArray foo)
{
    return foo->shift();
}

KayaArray createArray(int size)
{
    return newKayaArray(size);
}

void resizeArray(KayaArray foo, int size)
{
    foo->resize(size);
}

int arraySize(KayaArray foo)
{
    return foo->size();
}


FILE* getstdin()
{
    return stdin;
}

FILE* getstdout()
{
    return stdout;
}

FILE* getstderr()
{
    return stderr;
}

void setStdErr(FILE* f)
{
#ifndef WIN32
//    stderr = f;
#endif
}

bool validFile(FILE* f)
{
    return f!=NULL;
}

void storeArgs(int argc,char* argv[])
{
    args = new Array(argc);
    for (int i=0;i<argc;++i) {
	KayaValue arg = new Value((void*)(new String(strtowc(argv[i]))),KVT_STRING);
	args->push_back(arg);
    }
}

KayaArray getArgs()
{
    return args;
}

int gettime()
{
    return (int)(time(NULL));
}

kint dogettimeofday() {
  timeval tv;
  gettimeofday(&tv,NULL);
  return tv.tv_usec;
}

#ifdef WIN32
// taken from libiberty (LGPL) in the GCC libraries
int gettimeofday (struct timeval *tp, void *tz)
{     
  /* Offset between 1/1/1601 and 1/1/1970 in 100 nanosec units */
  static const unsigned long long W32_FT_OFFSET = 116444736000000000ULL;
  SYSTEMTIME s_time;
  union {
    unsigned long long ns100;
    FILETIME f_time;
  } now;

  if (tz)
    abort ();
  /* GetSystemTimeAsFileTime, introduced with NT version 3.5 and
     also available on Win9x, would be more efficient.  We use
     this instead to provide support for NT versions back to 3.1.
     This will also work on WinCE.  We don't worry about NT 3.0
     and earlier since the CRT used by mingw (msvcrt.dll) is not
     compatible with NT 3.0 either.  */
  GetSystemTime (&s_time);
  if (!SystemTimeToFileTime (&s_time, &now.f_time))
    GetSystemTime (&s_time);
  if (!SystemTimeToFileTime (&s_time, &now.f_time))
    return -1; 
  tp->tv_usec=(long)((now.ns100 / 10ULL) % 1000000ULL );
  tp->tv_sec= (long)((now.ns100 - W32_FT_OFFSET) / 10000000ULL);
  return 0;
}
#endif

KayaValue dogmtime(int secs)
{
    Union* tu = new Union(NULL,0,9,false);
    struct tm ts;
    time_t t = (time_t)secs;
    gmtime_r(&t,&ts);
    Union* mon = new Union(NULL,ts.tm_mon,0,false);
    Union* wday = new Union(NULL,ts.tm_wday,0,false);

    tu->args[0]=mkint((void*)ts.tm_sec);
    tu->args[1]=mkint((void*)ts.tm_min);
    tu->args[2]=mkint((void*)ts.tm_hour);
    tu->args[3]=mkint((void*)ts.tm_mday);
    tu->args[4]=MKUNION(mon);
    tu->args[5]=mkint((void*)(ts.tm_year+1900));
    tu->args[6]=MKUNION(wday);
    tu->args[7]=mkint((void*)ts.tm_yday);
    tu->args[8]=mkint((void*)ts.tm_isdst);
    return new Value((void*)tu,KVT_UNION);
}

KayaValue dolocaltime(int secs)
{
    Union* tu = new Union(NULL,0,9,false);
    struct tm ts;
    time_t t = (time_t)secs;
    localtime_r(&t,&ts);
    Union* mon = new Union(NULL,ts.tm_mon,0,false);
    Union* wday = new Union(NULL,ts.tm_wday,0,false);

    tu->args[0]=mkint((void*)ts.tm_sec);
    tu->args[1]=mkint((void*)ts.tm_min);
    tu->args[2]=mkint((void*)ts.tm_hour);
    tu->args[3]=mkint((void*)ts.tm_mday);
    tu->args[4]=MKUNION(mon);
    tu->args[5]=mkint((void*)(ts.tm_year+1900));
    tu->args[6]=MKUNION(wday);
    tu->args[7]=mkint((void*)ts.tm_yday);
    tu->args[8]=mkint((void*)ts.tm_isdst);
    return new Value((void*)tu,KVT_UNION);
}

int domktime(KayaValue time)
{
    struct tm t;
    Union* tu = time->getUnion();
    t.tm_sec = tu->args[0]->getInt();
    t.tm_min = tu->args[1]->getInt();
    t.tm_hour = tu->args[2]->getInt();
    t.tm_mday = tu->args[3]->getInt();
    t.tm_year = (tu->args[5]->getInt())-1900;
    t.tm_yday = tu->args[7]->getInt();
    t.tm_isdst = tu->args[8]->getInt();
    t.tm_mon = U_TAG(tu->args[4]->getUnion());
    t.tm_wday = U_TAG(tu->args[6]->getUnion());

    int x = (int)mktime(&t);
    return x;
}

int getclock()
{
    return (int)(clock());
}
#ifndef WIN32
int runProgram(wchar_t* prog, KayaArray args)
{
    char** argv = new char*[args->size()+2];

    argv[0]=wctostr(prog);
    for(int i=0;i<args->size();++i) {
	argv[i+1]=wctostr(args->lookup(i)->getString()->getVal());
    }
    argv[args->size()+1]=NULL;
    pid_t pid = fork();
    int status;
    if (pid==0) {
	execvp(wctostr(prog),argv);
	exit(0);
    }
    else {
	wait(&status);
    }

    delete argv;
    return status;
}

int dofork()
{
    return (int)fork();
}

void reap()
{
    waitpid(-1,NULL,WNOHANG);
}

int dowait()
{
    int status;
    wait(&status);
    return status;
}

int dowaitpid(int pid)
{
    int status;
    waitpid(pid,&status,0);
    return status;
}
#endif


int dogetpid()
{
    pid_t t = getpid();
    return (int)t;
}

bool do_wcschr(wchar_t c, wchar_t* s) {
  return wcschr(s,c)!=NULL;
}

int do_wcscspn(wchar_t c, wchar_t* s) {
  wchar_t buf[2];
  buf[0] = c;
  buf[1] = '\0';
  return wcscspn(s,buf);
}

/** The following base64 functions are adapted from
 * http://base64.sourceforge.net/b64.c
 * By Bob Trower, Copyright (c) Trantor Standard Systems Inc., 2001
 * See COPYING.b64 for redistribution conditions
 */
// Modified by CIM to work with wchar_t input
/*
** Translation Table as described in RFC1113
*/
static const char cb64[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/*
** Translation Table to decode (created by author)
*/
static const char cd64[]="|$$$}rstuvwxyz{$$$$$$$>?@ABCDEFGHIJKLMNOPQRSTUVW$$$$$$XYZ[\\]^_`abcdefghijklmnopq";

/*
** encodeblock
**
** encode 3 8-bit binary bytes as 4 '6-bit' characters
*/
void encodeblock( unsigned char in[3], unsigned char out[4], int len )
{
    out[0] = cb64[ in[0] >> 2 ];
    out[1] = cb64[ ((in[0] & 0x03) << 4) | ((in[1] & 0xf0) >> 4) ];
    out[2] = (unsigned char) (len > 1 ? cb64[ ((in[1] & 0x0f) << 2) | ((in[2] & 0xc0) >> 6) ] : '=');
    out[3] = (unsigned char) (len > 2 ? cb64[ in[2] & 0x3f ] : '=');
}

/*
** decodeblock
**
** decode 4 '6-bit' characters into 3 8-bit binary bytes
*/
void decodeblock( unsigned char in[4], unsigned char out[3] )
{   
    out[ 0 ] = (unsigned char ) (in[0] << 2 | in[1] >> 4);
    out[ 1 ] = (unsigned char ) (in[1] << 4 | in[2] >> 2);
    out[ 2 ] = (unsigned char ) (((in[2] << 6) & 0xc0) | in[3]);
}

wchar_t* b64enc(wchar_t* wblock, int len)
{
  return strtowc(b64enc(wctostr(wblock),len));
}

// since may want to encode binary data
char* b64enc(char* block, int len) {
    char* outblock = (char*)GC_malloc(sizeof(char)*len*2);
    unsigned char in[3], out[4];
    int i,x,y;
    x=0; // Input pointer
    y=0; // Output pointer
    while(x<len) {
	for(i = 0; i<3;++i) {
	    if (x>=len) { 
		in[i]=0;
	    } else {
		unsigned char v = (unsigned char)block[x];
		in[i]=v;
	    }
	    ++x;
	}
	encodeblock(in,out,3);
	for(i=0;i<4;++i) {
	    outblock[y] = out[i];
	    ++y;
	}
    }
    outblock[y]='\0';
    return outblock;
}

wchar_t* b64dec(wchar_t* wblock, KayaValue outlen)
{
  return strtowc(b64dec(wctostr(wblock),outlen));
}
// again, may want to return binary data
char* b64dec(char* block, KayaValue outlen) {
    int len = strlen(block);
    char* outblock = (char*)GC_malloc(sizeof(char)*len);
    unsigned char in[4], out[3];
    int i,x,y;
    x=0; // Input pointer
    y=0; // Output pointer
    while(x<len) {
	for(i = 0; i<4;++i) {
	    if (x>=len) { 
		in[i]=0;
	    } else {
		unsigned char v = (unsigned char)block[x];
		v = (unsigned char) ((v < 43 || v > 122) ? 0 : cd64[ v - 43 ]);
                if( v ) {
                    v = (unsigned char) ((v == '$') ? 0 : v - 61);
                }
		in[i]=v-1;
	    }
	    ++x;
	}
	decodeblock(in,out);
	for(i=0;i<3;++i) {
	    outblock[y] = out[i];
	    ++y;
	}
    }
    outblock[y]='\0';
    KayaSetInt(outlen,y);
    return outblock;
}

/** End of Bob Trower's code */

typedef enum { GENERAL, INTQS, STRQS } QSver;

void qs(VMState* vm, KayaArray xs,int l, int r,Closure* f, QSver version);
int partition(VMState* vm, KayaArray xs, int l, int r, int pi, 
	      Closure* sortfn, QSver version);

void quicksort(void* vmptr, KayaArray xs, KayaValue sortfn)
{
    VMState* vm = (VMState*)vmptr;
    Closure* f = sortfn->getFunc();
    qs(vm,xs,0,xs->size()-1,f, GENERAL);
}

void qs(VMState* vm,KayaArray xs,int l, int r,Closure* f, QSver version)
{
    if (r>l) {
	int pi = partition(vm,xs,l,r,(l+r)/2,f, version);
	qs(vm,xs,l,pi-1,f, version);
	qs(vm,xs,pi+1,r,f, version);
    }
}

int partition(VMState* vm, KayaArray xs, int l, int r, int pi, 
	      Closure* sortfn, QSver version)
{
    KayaValue xpi = xs->lookup(pi);
    Value pivotval_stack(NULL,xpi->getFunTable()); KayaValue pivotval=&pivotval_stack;
    Value tmp_stack(NULL,KVT_INT); KayaValue tmp=&tmp_stack;
//    static KayaValue pivotval = new Value(NULL,inttable);
//    static KayaValue tmp = new Value(NULL,inttable);
    pivotval->setPtr(xpi);
//    swap(xpi,xs->lookup(r));
    KayaValue xsr = xs->lookup(r);
    KayaValue xssi, xsi;
    tmp->setPtr(xsr);
    xsr->setPtr(xpi);
    xpi->setPtr(tmp);

/*    cout << "Before Partition " << l << " - " << r << " : ";
    for(unsigned i=0;i<xs->size();i++) {
	cout << xs->lookup(i)->getInt() << ",";
    }
    cout << endl;*/

    int si = l;
    int res;

    for(int i=l; i<r; ++i) {
	xsi = xs->lookup(i);
	xssi = xs->lookup(si);

	switch(version) {
	case INTQS:
	    res = xsi->getInt()-pivotval->getInt();
	    break;
	case STRQS:
	    res = wcscmp(xsi->getString()->getVal(),
			 pivotval->getString()->getVal());
	    break;
	default:
	    vm->push(pivotval);
	    vm->push(xsi);
	    sortfn->run(vm);
	    res = vm->doPop()->getInt();
	    break;
	}
	if (res<0) {
	    tmp->setPtr(xssi);
	    xssi->setPtr(xsi);
	    xsi->setPtr(tmp);
//	    swap(xs->lookup(si),xs->lookup(i));
	    ++si;
	}
    }
//    swap(xs->lookup(r),xs->lookup(si));

    xssi = xs->lookup(si);

    tmp->setPtr(xsr);
    xsr->setPtr(xssi);
    xssi->setPtr(tmp);

/*    cout << "Partition: ";
    for(unsigned i=0;i<xs->size();i++) {
	cout << xs->lookup(i)->getInt() << ",";
    }
    cout << endl;*/
    return si;
}


int funtable_compare(void* vmptr,KayaValue x,KayaValue y) 
{
#ifndef NOCHECK
  // NULL pointer for vmptr means something else is confident x and y exist
    if (vmptr != NULL && (x == NULL || x->getType() == KVT_NULL || y == NULL || y->getType() == KVT_NULL)) {
	VMState* vm = (VMState*)vmptr;
	vm->kaya_rtsError(INVALID_VALUE);
    }
#endif
    int r = 0;
    FUNTABLE_OP(cmp,x,y)
	return r;
}

bool funtable_equal(void* vmptr,KayaValue x, KayaValue y)
{
#ifndef NOCHECK
    if (vmptr != NULL && (x == NULL || x->getType() == KVT_NULL || y == NULL || y->getType() == KVT_NULL)) {
	VMState* vm = (VMState*)vmptr;
	vm->kaya_rtsError(INVALID_VALUE);
    }
#endif
    valtype vt = x->getType();
    switch (vt) {
    case KVT_INT:
      return inttable_fasteq(x,y);
    case KVT_STRING:
      return stringtable_fasteq(x,y);
    case KVT_REAL:
      return realtable_fasteq(x,y);
    case KVT_EXCEPTION:
      return exceptiontable_fasteq(x,y);
    default:
      map<Value*, Value*> done;
      return funtable_equal_aux(x,y,done);
    }
}

bool funtable_equal_aux(KayaValue x, KayaValue y, map<Value*, Value*>& done)
{
    bool r = 0;
    if (x->getFunTable() != y->getFunTable()) {
      return 0;
    }
    switch (x->getFunTable()) { 
    case KVT_INT: 
	r = inttable_eq(x,y,done); 
	break; 
    case KVT_STRING: 
	r = stringtable_eq(x,y,done); 
	break; 
    case KVT_FUNC: 
	r = fntable_eq(x,y,done); 
	break; 
    case KVT_ARRAY: 
	r = arraytable_eq(x,y,done); 
	break; 
    case KVT_UNION: 
	r = uniontable_eq(x,y,done); 
	break; 
    case KVT_EXCEPTION: 
	r = exceptiontable_eq(x,y,done); 
	break; 
    case KVT_REAL: 
	r = realtable_eq(x,y,done); 
	break; 
    case KVT_NULL: 
	r = inttable_eq(x,y,done);
	break; // can't happen
    } 
    return r;
}


int funtable_hash(KayaValue x)
{
    Value* r = new Value(NULL,KVT_NULL);
    switch (x->getFunTable()) { 
    case KVT_INT: 
	r = inttable_hash(x); 
	break; 
    case KVT_STRING: 
	r = stringtable_hash(x); 
	break; 
    case KVT_FUNC: 
	r = fntable_hash(x); 
	break; 
    case KVT_ARRAY: 
	r = arraytable_hash(x); 
	break; 
    case KVT_UNION: 
	r = uniontable_hash(x); 
	break; 
    case KVT_EXCEPTION: 
	r = exceptiontable_hash(x); 
	break; 
    case KVT_REAL: 
	r = realtable_hash(x); 
	break; 
    case KVT_NULL: 
	r = inttable_hash(x);
	break; // can't happen
    } 
    return r->getInt();
}

int funtable_memusage(KayaValue x)
{
    int r = 0;
    switch (x->getFunTable()) { 
    case KVT_INT: 
	r = inttable_memusage(x); 
	break; 
    case KVT_STRING: 
	r = stringtable_memusage(x); 
	break; 
    case KVT_FUNC: 
	r = fntable_memusage(x); 
	break; 
    case KVT_ARRAY: 
	r = arraytable_memusage(x); 
	break; 
    case KVT_UNION: 
	r = uniontable_memusage(x); 
	break; 
    case KVT_EXCEPTION: 
	r = exceptiontable_memusage(x); 
	break; 
    case KVT_REAL: 
	r = realtable_memusage(x); 
	break; 
    case KVT_NULL: 
	r = inttable_memusage(x);
	break;
    } 
    return r;
}

wchar_t* funtable_marshal(void* vmptr,KayaValue x, int i)
{
#ifndef NOCHECK
    if (vmptr != NULL && (x == NULL || x->getType() == KVT_NULL)) {
	VMState* vm = (VMState*)vmptr;
	vm->kaya_rtsError(INVALID_VALUE);
    }
#endif
    vector<KayaValue> done;
    wchar_t* mdata = funtable_marshal_aux(vmptr,done,x,i);
    wchar_t mid[27]; // can be negative
    SWPRINTF(mid,27,L"[%d][%d]",i,getFnHash());
    String* idmdata = new String(mid);
    idmdata->append(mdata);
    return idmdata->getVal();
}

wchar_t* funtable_marshal_aux(void* vmptr,vector<KayaValue>& done,KayaValue x, int i)
{
    VMState* vm = (VMState*)vmptr;
    Value* r = new Value(NULL,KVT_NULL);
    switch (x->getFunTable()) { 
    case KVT_INT: 
      r = inttable_marshal(vm,done,x,i);
      break;
    case KVT_STRING: 
      r = stringtable_marshal(vm,done,x,i);
      break;
    case KVT_FUNC: 
      r = fntable_marshal(vm,done,x,i);
      break;
    case KVT_ARRAY: 
      r = arraytable_marshal(vm,done,x,i); 
      break;
    case KVT_UNION: 
      r = uniontable_marshal(vm,done,x,i);
      break;
    case KVT_EXCEPTION: 
      r = exceptiontable_marshal(vm,done,x,i);
      break;
    case KVT_REAL: 
      r = realtable_marshal(vm,done,x,i); 
      break;
    case KVT_NULL:
      r = inttable_marshal(vm,done,x,i);
      break; //can't happen
    } 
    return r->getString()->getVal();
}

Value* funtable_reflect(void* vmptr,KayaValue x)
{
#ifndef NOCHECK
    if (vmptr != NULL && (x == NULL || x->getType() == KVT_NULL)) {
	VMState* vm = (VMState*)vmptr;
	vm->kaya_rtsError(INVALID_VALUE);
    }
#endif
    map<KayaValue, KayaValue> done;
    return funtable_reflect_aux(vmptr,done,x);
}

Value* funtable_reflect_aux(void* vmptr,
			    map<KayaValue, KayaValue>& done,KayaValue x)
{
    VMState* vm = (VMState*)vmptr;
    Value* r = new Value(NULL,KVT_NULL);
    switch (x->getFunTable()) { 
    case KVT_INT: 
      r = inttable_reflect(vm,done,x);
      break;
    case KVT_STRING: 
      r = stringtable_reflect(vm,done,x);
      break;
    case KVT_FUNC: 
      r = fntable_reflect(vm,done,x);
      break;
    case KVT_ARRAY: 
      r = arraytable_reflect(vm,done,x); 
      break;
    case KVT_UNION: 
      r = uniontable_reflect(vm,done,x);
      break;
    case KVT_EXCEPTION: 
      r = exceptiontable_reflect(vm,done,x);
      break;
    case KVT_REAL: 
      r = realtable_reflect(vm,done,x); 
      break;
    case KVT_NULL:
      r = inttable_reflect(vm,done,x);
      break; //can't happen
    } 
    return r;
}

KayaValue reifyUnion(int tag, int arity)
{
    return KayaUnion(tag, arity);
}

void reifyUnionArgs(KayaValue v, KayaArray flds)
{
    for(int i=0;i<flds->size();++i) {
	KayaUnionSetArg(v,i,KayaArrayLookup(flds,i));
    }
}

KayaValue reifyClosure(int fnid, int arity)
{
    Closure* c = new Closure(NULL, getFn(fnid), arity, arity, false);
    return new Value((void*)c,KVT_FUNC);
}

void reifyClosureArgs(KayaValue v, KayaArray flds)
{
    Closure* c = v->getFunc();
    for(int i=0;i<flds->size();++i) {
	c->setArg(i,KayaArrayLookup(flds,i));
    }
}

KayaValue funtable_copy(void* vmptr,KayaValue x)
{
#ifndef NOCHECK
  if (x == NULL || x->getType() == KVT_NULL) {
    VMState* vm = (VMState*)vmptr;
    vm->kaya_rtsError(INVALID_VALUE);
  }
#endif
    valtype vt = x->getType();
    switch (vt) {
    case KVT_INT:
      return inttable_fastcopy(x);
    case KVT_STRING:
      return stringtable_fastcopy(x);
    case KVT_REAL:
      return realtable_fastcopy(x);
    case KVT_EXCEPTION:
      return exceptiontable_fastcopy(x);
    default:
      map<Value*, Value*> done;
      return funtable_copy_aux(x,done);
    }
}

KayaValue funtable_copy_aux(KayaValue x, map<Value*, Value*>& done)
{
    KayaValue r = new Value(NULL,KVT_NULL);
    FUNTABLE_OP(copy,x,done)
    return r;
}

bool is_initialised(KayaValue x) {
  return (x->getType() != KVT_NULL);
}

KayaValue unsafe_id(KayaValue x)
{
//    cout << "Unsafe id: " << x->getRaw() << endl;
    return x;
}

wchar_t* getAddr(KayaValue x)
{
    wchar_t* buf=(wchar_t*)GC_MALLOC_ATOMIC(65536*sizeof(wchar_t)); 
    SWPRINTF(buf,65536,L"Address of value at %p is %p",x,x->getRaw());
    return buf;
}

wchar_t* except_msg(KayaValue x)
{
    return x->getExcept()->err->getVal();
}

int except_code(KayaValue x)
{
    return x->getExcept()->code;
}

void except_dumpbt(KayaValue x)
{
    x->getExcept()->dumpBacktrace();
}

int doGetFnID(KayaValue fn)
{
//    if (fn->getType()!=KVT_FUNC) { return -1; }
//    cout << "Raw: " << fn->getRaw() << endl;
    return getFnID(fn->getFunc()->getfn());
}

void callFnID(void* vmptr, int id, KayaValue arg)
{
    VMState* vm = (VMState*)vmptr;
    func f = getFn(id);
    if (f!=NULL) {
	PUSH(arg);
	f(vm);
    } else {
	vm->kaya_rtsError(UNKNOWN_FUNCTION_ID);
    }
}
#ifndef WIN32
int memusage()
{
/*    struct rusage usage;
    int x = getrusage(RUSAGE_SELF,&usage);
    if (x!=0) return -1;
    int total = usage.ru_idrss+usage.ru_isrss+usage.ru_maxrss;
    return total; */
//    struct mallinfo info = mallinfo();

//    return info.arena;
    return 0;
}
#endif
int isNull(void* p)
{
    return p==NULL;
}

int isIdentical(KayaValue x, KayaValue y)
{
    return (x->getRaw()==y->getRaw());
}

/* Code to handle do_access() in IO.k 
// Added by Chris Morris, 18/6/05 */
int do_access(wchar_t* pathname, KayaValue mode) {
  int tag = KayaUnionGetTag(mode);
  int acmode = 0;
  switch(tag) {
  case 0:
    acmode = F_OK;
    break;
  case 1:
    acmode = X_OK;
    break;
  case 2:
    acmode = W_OK;
    break;
  case 3:
    acmode = R_OK;
    break;
  }
  return access(wctostr(pathname),acmode);

}
#ifndef WIN32
int getres(KayaValue lim, bool& ok)
{
    ok = true;
    switch(KayaUnionGetTag(lim)) {
    case 0:
	return RLIMIT_AS;
	break;
    case 1:
	return RLIMIT_CORE;
	break;
    case 2:
	return RLIMIT_CPU;
	break;
    case 3:
	return RLIMIT_DATA;
	break;
    case 4:
	return RLIMIT_FSIZE;
	break;
    case 5:
#ifdef RLIMIT_LOCKS
	return RLIMIT_LOCKS;
#else
	ok = false;
	return 0;
#endif
	break;
    case 6:
#ifdef RLIMIT_MEMLOCK
	return RLIMIT_MEMLOCK;
#else
	ok = false;
	return 0;
#endif
	break;
    case 7:
	return RLIMIT_NOFILE;
	break;
    case 8:
#ifdef RLIMIT_NPROC
	return RLIMIT_NPROC;
#else
	ok = false;
	return 0;
#endif
	break;
    case 9:
#ifdef RLIMIT_RSS
	return RLIMIT_RSS;
#else
	ok = false;
	return 0;
#endif
	break;
    case 10:
	return RLIMIT_STACK;
	break;
    default:
	return RLIMIT_AS; // Can't happen anyway
    }
}

int do_setrlimit(KayaValue lim, int soft, int hard)
{
    struct rlimit r;
    if (soft==-1) {
	r.rlim_cur = RLIM_INFINITY;
    } else {
	r.rlim_cur=soft;
    }
    if (soft==-1) {
	r.rlim_max=RLIM_INFINITY;
    } else {
	r.rlim_max=hard;
    }
    bool ok;
    int res = getres(lim,ok);
    if (ok) {
	setrlimit(res,&r);
	return 1;
    } else {
	return 0;
    }
}
#endif

int do_unlink(const wchar_t* fn) {
  return unlink(wctostr(fn));
}

wchar_t* dogetcwd() {
  unsigned int sz = 1024;
  char* buf;
  char* rv;
  do {
    buf = (char*)GC_MALLOC_ATOMIC(sz*sizeof(char));
    rv = getcwd(buf,sz);
    sz = sz*2;
  } while (rv == NULL);
  return KSTRING(buf);
}

int do_mkdir(const wchar_t* fn, int um) {
#ifdef WIN32
	return mkdir(wctostr(fn));
#else
	return mkdir(wctostr(fn),um);
#endif
}

void* do_opendir(wchar_t* name)
{
    DIR* d = opendir(wctostr(name));
    if (errno!=0) {
	//	cout << strerror(errno) << endl;
    }
    return (void*)d;
}

int do_closedir(void* dir)
{
    return closedir((DIR*)dir);
}

void* do_readdir(void* dir)
{
    struct dirent* ent = readdir((DIR*)dir);
    return (void*)ent;
}

wchar_t* dir_getname(void* ent)
{
    struct dirent* entd = (struct dirent*)ent;
    return strtowc(entd->d_name);
}

#define SSET(x,y) KayaUnionSetArg(val,x,KayaInt((int)(buf->y)))

#define MSET(x,y) if (x(mode)) { \
	KayaUnionSetArg(mval, y, KayaInt(1)); \
    } else { \
	KayaUnionSetArg(mval, y, KayaInt(0)); \
    }

#define MFSET(x,y) if (x & mode) { \
	KayaUnionSetArg(mval, y, KayaInt(1)); \
    } else { \
	KayaUnionSetArg(mval, y, KayaInt(0)); \
    }

KayaValue getKayaMode(mode_t mode) 
{
    KayaValue mval = KayaUnion(0,13);

    KayaArray uperm = newKayaArray(3);
    KayaArray gperm = newKayaArray(3);
    KayaArray operm = newKayaArray(3);

    static KayaValue read = KayaUnion(0,0);
    static KayaValue write = KayaUnion(1,0);
    static KayaValue exec = KayaUnion(2,0);

    // User permissions
    if (S_IRUSR & mode) {
	KayaArrayPush(uperm, read);
    }
    if (S_IWUSR & mode) {
	KayaArrayPush(uperm, write);
    }
    if (S_IXUSR & mode) {
	KayaArrayPush(uperm, exec);
    }
#ifndef WIN32
    // Group permissions
    if (S_IRGRP & mode) {
	KayaArrayPush(gperm, read);
    }
    if (S_IWGRP & mode) {
	KayaArrayPush(gperm, write);
    }
    if (S_IXGRP & mode) {
	KayaArrayPush(gperm, exec);
    }

    // Other permissions
    if (S_IROTH & mode) {
	KayaArrayPush(operm, read);
    }
    if (S_IWOTH & mode) {
	KayaArrayPush(operm, write);
    }
    if (S_IXOTH & mode) {
	KayaArrayPush(operm, exec);
    }
#endif
    KayaUnionSetArg(mval,0,KayaArrayVal(uperm));
    KayaUnionSetArg(mval,1,KayaArrayVal(gperm));
    KayaUnionSetArg(mval,2,KayaArrayVal(operm));
#ifndef WIN32
    MSET(S_ISSOCK,3);
#endif
#ifdef S_ISLNK
    MSET(S_ISLNK,4);
#endif
    MSET(S_ISREG,5);
    MSET(S_ISBLK,6);
    MSET(S_ISDIR,7);
    MSET(S_ISCHR,8);
    MSET(S_ISFIFO,9);
#ifndef WIN32
    MFSET(S_ISUID,10);
    MFSET(S_ISGID,11);
    MFSET(S_ISVTX,12);
#endif
    return mval;
}

KayaValue do_stat(void* vmptr, wchar_t* name)
{
    VMState* vm = (VMState*)vmptr;
    struct stat* buf = (struct stat*)KayaAlloc(sizeof(struct stat));
    int ret = stat(wctostr(name), buf);
    if (ret==-1) {
      	vm->kaya_internalError(errno);
    }

    KayaValue val = KayaUnion(0,5);
    SSET(1,st_nlink);
    SSET(2,st_atime);
    SSET(3,st_mtime);
    SSET(4,st_ctime);

    KayaUnionSetArg(val,0,getKayaMode(buf->st_mode));

    return val;
}

int maxMemUsage(void* vmptr)
{
    VMState* vm = (VMState*)vmptr;
    return vm->maxMemUsage();
}

wchar_t* do_urlencode(void* vm, wchar_t* strinwide)
{
  char* strin = CSTRING(strinwide);
    VMState* vmptr = (VMState*)vm;
    char* strout = (char*)KayaAlloc(strlen(strin)*3+1);
    if (strout==NULL) {
	// something broke, probably duff input.
	vmptr->kaya_internalError(1);
    }
    char* r = strout;
    while(*strin!='\0') {
	if (*strin==' ') {
	    *strout = '+';
	    ++strout;
	}
	else if (ispunct(*strin) || *strin<32)
	{
	  // 4 for length because we need to temporarily let snprintf
	  // put a \0 there
	    snprintf(strout,4,"%%%2x", (int)*strin);
	    strout+=3;
	}
	else {
	    *strout = *strin;
	    ++strout;
	}
	++strin;
    }
    *strout='\0';
    return KSTRING(r);
}

wchar_t* do_urldecode(void* vm, wchar_t* strinwide)
{
  char* strin = CSTRING(strinwide);
    VMState* vmptr = (VMState*)vm;
    char* strout = (char*)KayaAlloc(strlen(strin)+1);
    if (strout==NULL) {
	// something broke, probably duff input.
	vmptr->kaya_internalError(1);
    }
    char* r = strout;
    char buf[3];
    char* convcheck;

    buf[2]='\0';
    while(*strin!='\0') {
	if (*strin=='+') {
	    *strout = ' ';
	    ++strout;
	}
	else if (*strin == '%')
	{
	    if (strlen(strin)<3) {
		vmptr->kaya_internalError(1);
	    }
	    buf[0] = strin[1];
	    buf[1] = strin[2];
	    *strout = (char)strtol(buf,&convcheck,16);
	    if (*convcheck != '\0') {
		vmptr->kaya_internalError(1);
	    }
	    strin+=2;
	    ++strout;
	}
	else {
	    *strout = *strin;
	    ++strout;
	}
	++strin;
    }
    *strout='\0';
    return KSTRING(r);
}

FILE *do_fopen(const wchar_t* path, const wchar_t* mode) {
    return fopen(wctostr(path),wctostr(mode));
}

FILE *do_freopen(const wchar_t* path, const wchar_t* mode, FILE* stream) {
    return freopen(wctostr(path),wctostr(mode),stream);
}

void do_rename(const wchar_t* oldfile, const wchar_t* newfile) {
    rename(wctostr(oldfile),wctostr(newfile));
}

void do_chdir(const wchar_t* newdir) {
    chdir(wctostr(newdir));
}

wchar_t* do_getenv(const wchar_t* env) {
    return strtowc(getenv(wctostr(env)));
}

#ifndef WIN32
int do_system(const wchar_t* prog) {
    return system(wctostr(prog));
}

FILE* do_popen(const wchar_t* prog, int read) {
    if (read) {
	return popen(wctostr(prog),"r");
    } else {
	return popen(wctostr(prog),"w");
    }
}

int do_pclose(FILE* stream) {
    return pclose(stream);
}

#endif

kint binary_length(const wchar_t* str) {
    char* rawstr = CSTRING(str);
    return (kint)strlen(rawstr);
}

int nextPower(int x)
{
    int next = 1;
    while(x>next) {
	next<<=1;
    }
    return next;
}

int strAlloc(Value* str)
{
    return str->getString()->space()*sizeof(wchar_t);
}

Value* createString(int len)
{
    return new Value((void*)(new String(len)),KVT_STRING);
}

void do_GC_enable_incremental() {
#ifdef NOCHECK
  GC_enable_incremental();
#endif
}

void do_sleep(int len) {
#ifdef WIN32
	Sleep(len*1000);
#else
	sleep(len);
#endif
}

void do_usleep(int len) {
#ifdef WIN32
	Sleep(len/1000);
#else
	usleep(len);
#endif
}

int strHash(wchar_t* m)
{
    int hash = 0;
    int len = wcslen(m);
    for(int i=0;i<len;++i) {
	hash = 131*hash+m[i];
    }
    return hash;
}

int rtchecks()
{
    return RTCHECKS;
}
